/* 
 * tclGet.c --
 *
 *      This file contains procedures to convert strings into
 *      other forms, like integers or floating-point numbers or
 *      booleans, doing syntax checking along the way.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


extern "C" {
#include <string.h>
#include <malloc.h>
#include <ctype.h>

extern double strtod(const char *, char **);
}

#include "tclGet.h"
#include "../kernel/util.h"


/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetFontStyle --
 *
 *      Given a string, produce the corresponding FontStyle value.
 *
 * Results:
 *      The return value is normally TCL_OK;  in this case *intPtr
 *      will be set to the integer value equivalent to string.  If
 *      string is improperly formed then TCL_ERROR is returned and
 *      an error message will be left in interp->result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_GetFontStyle(Tcl_Interp *interp, char *string, FontStyle &style) {

  int result;

  result = String_to_FontStyle(string);

  if( result == -1 ) {
    Tcl_AppendResult(interp,
		     "Error: bad font style type \"", string,
		     "\": must normal, bold, italic or bold_italic",
		     (char *)NULL);
    return TCL_ERROR;
  }
  style = (FontStyle)result;
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetAlignment --
 *
 *      Given a string, produce the corresponding Alignment value.
 *
 * Results:
 *      The return value is normally TCL_OK;  in this case *intPtr
 *      will be set to the integer value equivalent to string.  If
 *      string is improperly formed then TCL_ERROR is returned and
 *      an error message will be left in interp->result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_GetAlignment(Tcl_Interp *interp, char *string, StyleAlignType &alignPtr) {

  int result;

  result = String_to_Alignment(string);
  if( result == -1 ) {
    Tcl_AppendResult(interp,
		     "Error: bad alignment type \"", string,
		     "\": must be left, right, center or full", (char *)NULL);
    return TCL_ERROR;
  }
  alignPtr = (StyleAlignType)result;
  return TCL_OK;    
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetVoffset --
 *
 *      Given a string, produce the corresponding Vertical Offet
 *      value.
 *
 * Results:
 *      The return value is normally TCL_OK;  in this case *intPtr
 *      will be set to the integer value equivalent to string.  If
 *      string is improperly formed then TCL_ERROR is returned and
 *      an error message will be left in interp->result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_GetVoffset(Tcl_Interp *interp, char *string, VOffsetType &voffsetPtr) {

  int c;
  unsigned char lenght;

  c = string[0];
  lenght = strlen(string);

  if( (c == 's') && (strncmp(string, "subscript", lenght) == 0 ) )
    voffsetPtr = SUBSCRIPT_VOFFSET;
  else if( (c == 's') && (strncmp(string, "superscript", lenght) == 0 ) )
    voffsetPtr = SUPERSCRIPT_VOFFSET;
  else {
    Tcl_AppendResult(interp,
		     "Error: bad voffset type \"", string,
		     "\": must be subscripts or superscript", (char *)NULL);
    return TCL_ERROR;
  }
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetFont --
 *
 *      Given a string, produce the corresponding Font value.
 *
 * Results:
 *      The return value is normally TCL_OK;  in this case *intPtr
 *      will be set to the integer value equivalent to string.  If
 *      string is improperly formed then TCL_ERROR is returned and
 *      an error message will be left in interp->result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_GetFont(Tcl_Interp *interp, char *string, FontItem *&fiPtr) {

  char **argsPtr;
  int numArgs, size;
  FontStyle style;


  if( Tcl_SplitList(interp, string, &numArgs, &argsPtr) != TCL_OK )
    return TCL_ERROR;    
  
  if( numArgs != 3 ) {
    Tcl_AppendResult(interp, "Error: bad font format \"", string,
		     "\": must be { family style size }",
		     (char *)NULL);
    return TCL_ERROR;
  }
  
  if( Tcl_GetFontStyle(interp, argsPtr[1], style) != TCL_OK )
    return TCL_ERROR;
  
  if( Tcl_GetInt(interp, argsPtr[2], &size) != TCL_OK )
    return TCL_ERROR;
  
  fiPtr = papyrus->get_font(argsPtr[0], style, size);
  
  if( fiPtr == NULL ) {
    char str[64];
    sprintf(str, "Error: font \"%s-%s-%d\" doesn't exit\n",
	    argsPtr[0], FontStyle_to_String(style), size);
    Tcl_AppendResult(interp, str, (char *)NULL);
    return TCL_ERROR;
  }
  
  free((char *)argsPtr);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetMargins --
 *
 *      Given a string, produce the corresponding Margins value.
 *
 * Results:
 *      The return value is normally TCL_OK;  in this case *intPtr
 *      will be set to the integer value equivalent to string.  If
 *      string is improperly formed then TCL_ERROR is returned and
 *      an error message will be left in interp->result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_GetMargins(Tcl_Interp *interp, char *string, int intPtr[]) {

  char **argsPtr;
  int numArgs;

  if( Tcl_SplitList(interp, string, &numArgs, &argsPtr) != TCL_OK )
    return TCL_ERROR;
  
  if( numArgs != 4 ) {
    Tcl_AppendResult(interp, "Error: bad margin format \"", string,
		     "\": must be { top bottom left right }",
		     (char *)NULL);
    return TCL_ERROR;
  }

  if( Tcl_GetPixels(interp, argsPtr[0], &intPtr[0]) != TCL_OK )
    return TCL_ERROR;

  if( Tcl_GetPixels(interp, argsPtr[1], &intPtr[1]) != TCL_OK )
    return TCL_ERROR;

  if( Tcl_GetPixels(interp, argsPtr[2], &intPtr[2]) != TCL_OK )
    return TCL_ERROR;

  if( Tcl_GetPixels(interp, argsPtr[3], &intPtr[3]) != TCL_OK )
    return TCL_ERROR;

  free((char *)argsPtr);

  return TCL_OK;
}



/*
 *--------------------------------------------------------------
 * This funciton was taken, and modified from the original TK4.1
 * source code.
 *
 * Copyright (c) 1991-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 *
 * Tk_GetPixels --
 *
 *      Given a string, returns the number of pixels corresponding
 *      to that string.
 *
 * Results:
 *      The return value is a standard Tcl return result.  If
 *      TCL_OK is returned, then everything went well and the
 *      rounded pixel distance is stored at *intPtr;  otherwise
 *      TCL_ERROR is returned and an error message is left in
 *      interp->result.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

int
Tcl_GetPixels(Tcl_Interp *interp, char *string, int *intPtr) {

  char *end;
  double d;
  float ppm;
  
  d = strtod(string, &end);
  if (end == string) {
  error:
    Tcl_AppendResult(interp, "bad screen distance \"", string,
		     "\"", (char *) NULL);
    return TCL_ERROR;
  }

  while( (*end != '\0') && isspace((unsigned char)*end) )
    end++;

  ppm = papyrus->get_xppm();

  switch (*end) {
  case 0:
    break;
  case 'c':
    d *= 10 * ppm;
    end++;
    break;
  case 'i':
    d *= 25.4 * ppm;
    end++;
    break;
  case 'm':
    d *= ppm;
    end++;
    break;
  case 'p':
    d *= (25.4/72.0) * ppm;
    end++;
    break;
  default:
    goto error;
  }

  while( (*end != '\0') && isspace((unsigned char)*end) )
    end++;
  if (*end != 0)
    goto error;
  if (d < 0)
    *intPtr = (int) (d - 0.5);
  else
    *intPtr = (int) (d + 0.5);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetChannelForWrite --
 *
 *      This function looks for a channel, which name is given, but
 *      it must be write enable.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_GetChannelForWrite(Tcl_Interp *interp, char *filename, Tcl_Channel &channel) {

  int mode;

  channel = Tcl_GetChannel(interp, filename, &mode);

  if( channel == (Tcl_Channel) NULL )
    return TCL_ERROR;

  if( (mode & TCL_WRITABLE) == 0 ) {
    Tcl_AppendResult(interp, "channel \"", filename,
		     "\" wasn't opened for writing", (char *) NULL);
    return TCL_ERROR;
  }
  return TCL_OK;
}



