/* 
 * commandsNZ.cc --
 *
 *      This file contains the top-level command routines for most of
 *      the PAPyRUS built-in commands whose names begin with the letters
 *      N to Z.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

extern "C" {
#include <stdio.h>

#ifdef HAVE_GUI
  #include <X11/Intrinsic.h>
#else
  #include <X11/Xlib.h>
#endif

#include <string.h>
extern char *strdup(const char *s);   // To prevent some compilation warnings

#include "../misc/shortcuts.h"
}


#include "../kernel/container.h"
#include "../kernel/wordSegment.h"
#include "../kernel/image.h"
#include "../kernel/word.h"
#include "../kernel/line.h"
#include "../kernel/page.h"
#include "../kernel/document.h"
#include "../kernel/papyrus.h"
#include "../kernel/isfuncs.h"
#include "../kernel/events.h"
#include "../kernel/util.h"
#include "tclGet.h"
#include "commands.h"


#ifdef HAVE_GUI
extern "C" {
extern Widget GUI_CreateNewDocument(void);
extern BOOL GUI_MakePageVisible(Display *, Window, int, int);
}
#endif

extern Tcl_Channel channel;



/*
 *----------------------------------------------------------------------
 *
 * NewDocument_Cmd --
 *
 *      This procedure is invoked to process the "NewDocument" or 
 *      "nd" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
NewDocument_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  if( argc != 1 ) {
    interp->result = "Usage: NewDocument";
    return TCL_ERROR;
  }

  Tcl_SetVar(interp, "alive", "1", TCL_GLOBAL_ONLY);

#ifdef HAVE_GUI
  Widget docWidget;

  docWidget = GUI_CreateNewDocument();
#endif
  current.doc = new Document(current);
  current.attr = new Attributes;

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * NewParagraph_Cmd --
 *
 *      This procedure is invoked to process the "NewParagraph" or 
 *      "np" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
NewParagraph_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int word_offset;
  Word *word, *new_word;
  Line *line, *line_parent, *tmp_line;
  Paragraph *new_para;
  Page *page;


  if( argc != 1 ) {
    interp->result = "Usage: NewParagraph";
    return TCL_ERROR;
  }

  if( is_first_of_para(current) == TRUE ) {
    CALL( Tcl_Eval(interp, "NewWord 1") );

    line = current.shape->get_line_parent();
    if( line->get_width() != 0 ) {
      page = (Page *)line->get_parent();
      new_para = new Paragraph( line->get_para() );
      
      line->split_container(1);
      line->set_to_redraw(REDRAW_PAGE);
      line->set_para( new_para );    
      page->can_fit(current);
      current.doc->draw_frame(0,0);
      
      Tcl_Eval(interp, "MoveCursorRight");
      return TCL_OK;
    }
  }


  if( is_first_of_line(current) == FALSE )
    CALL( Tcl_Eval(interp, "NewWord 1") );
  
  word = (Word *)current.shape->get_parent();
  line_parent = current.shape->get_line_parent();
  word_offset = word->get_offset();

  line = (Line *)line_parent->get_previous_same_container();
  if( line == NULL )
    line = line_parent;
  line->set_to_redraw(REDRAW_PAGE);

  line = (Line *)line_parent->split_container(word_offset);

  new_word = (Word *)line->get_child(0);

  /*
   * Now we have to test if there's any empty word, followed by
   * another word in the next line. If we find one, we remove it.
   */
  if( line->get_children_num() > 1 )
    if( new_word->get_children_num() == 1 )
      if( new_word->get_child(0)->get_children_num() == 0 ) {
	Tcl_Eval(interp, "MoveCursorRight");
	line->delete_children(1, 0);
      }

  /*
   * Now lets change the paragraph pointer of the lines which are
   * after the cursor. We only do it for the lines of the old 
   * paragraph.
   */

  new_para = new Paragraph( line_parent->get_para() );
  ((Line *)line)->set_to_redraw(REDRAW_PAGE);
  
  tmp_line = line;
  
  while( line_parent->is_in_same_para(line) == TRUE ) {
    line->set_para( new_para );
    line = (Line *)line->get_next_same_container();
  }
  
  line = (Line *)tmp_line->get_next_same_container();

  /*
   * So lets, format (down/up) the frames, and redraw them.
   */
  if( line != NULL )
    tmp_line->get_para()->recompute(current);

  /*
   * Now recompute the last line of the previous paragraph, because
   * there's perhaps a bottom margin (ie margin between the last line
   * of the paragraph and the first line of the next paragraph).
   */
  line_parent->recompute_size();

  current.shape->recompute_size();
  current.shape->get_line_parent()->format_frame();
  current.shape->get_line_parent()->can_fit(current, TRUE);

  current.doc->draw_frame(0,0);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * NewWord_Cmd --
 *
 *      This procedure is invoked to process the "NewWord" or
 *      "nw" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
NewWord_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Container *word_parent, *line_parent;
  Container *new_word;
  Shape *shape;
  int offset;
  int forced;


  if( argc == 1 ) {
    forced = 0;
  } else  {
    if( argc != 2 ) {
      interp->result = "Usage: NewWord [true|false]";
      return TCL_ERROR;
    }

    if( Tcl_GetBoolean(interp, argv[1], &forced) != TCL_OK )
      return TCL_ERROR;
  }

  if( forced == 0 ) {
    /*
     * It's not possible to add a new word beetween 2 other words (ie insert an space char
     * beetween 2 words).
     * So we can only add an space char if the cursor is at :
     *   - the middle of a word (neither the end and the beginning)
     *   - the end of a word, but also the end of the paragraph
     *
     * Test if we're not trying to add a new word beetween 2 words (ie insert an space char
     * beetween 2 words).
     * This rule has an exception : if the two words are in differents paragraphs, we can
     * add a word beetween them.
     */
    
    if( is_last_of_word(current) == TRUE && is_last_of_para(current) == FALSE ) {
      Tcl_Eval(interp, "MoveCursorRight");
      return TCL_OK;
    }
    
    if( is_first_of_word(current) == TRUE ) {
      Tcl_AppendResult(interp, "Can't add a new word", (char *)NULL);
      return TCL_OK;
    }
  }

  word_parent = current.shape->get_parent();
  line_parent = word_parent->get_parent();

  ((Line *)line_parent)->clear_frame();

  shape = (Shape *)current.shape->split_container(current.pos);
  offset = shape->get_offset();

  if( shape == current.shape && current.pos > 0 )
      offset++;

  shape->recompute_size();
  new_word = word_parent->split_container(offset);


  /*
   * If the cursor is at the beginning or at the end of a word,
   * we have to insert a new word in the line.
   */

  if( new_word == word_parent )
    Add_Empty_WordSegment();
  else
    if( current.shape->has_attributes() && current.attr->has_mark() == 0 )
      current.attr->copy_attributes(current.shape->get_attributes());

  /*
   * Test if the cursor is at the beginning of a line
   * If we have forced == 1, the cursor can't be beetween 2 words,
   * so if we find that it's at the end of a word, we deduce, that
   * it's at the beginning of a line
   */

  if( forced == 1 && current.pos == 0 ) {
    Container *previous;

    previous = current.shape->get_previous_same_container();
    
    /* Returns if :
     *    - we're at the beginning of the Text (previous==NULL)
     *    - we're at the beginning of a Word  (parent differs)
     */
    if( current.shape->has_same_parent(previous) == FALSE )
      return TCL_OK;
  }

 
  Line *line;

  current.shape->get_line_parent()->can_fit(current);
  
  line = current.shape->get_line_parent();

  if( line->get_previous_same_container() != NULL ) {
    line = (Line *)line->get_previous_same_container();
    line->set_to_redraw(REDRAW_ME);
  }

  line->format_frame();
  
  /*
   * Don't Move cursor if the cursor is at the beginning
   * of a wordSegment
   */
  if( current.pos > 0 )
    Tcl_Eval(interp, "MoveCursorRight");

  current.doc->draw_frame(0,0);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * PapyrusConfigure_Cmd --
 *
 *      This procedure is invoked to process the "PapyConfigure" or
 *      "ppc" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
PapyrusConfigure_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int i;
  int ma, hi;
  BOOL change;

  if( argc < 3 || (argc&1) == 0 ) {
    Tcl_AppendResult(interp,
		     "wrong # args: should be \"", argv[0], " options values\"",
		     (char *)NULL);
    return TCL_ERROR;
  }
  
  hi = ma = 0;
  change = FALSE;

  for(i=1; i<argc; i+=2) {
    if( argv[i][0] != '-' )
      goto error;
    if( strcmp(argv[i], "-mask") == 0 )       ma = i+1;
    else if( strcmp(argv[i], "-hideimages") == 0 ) hi = i+1;
    else {
      error:
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be mask or hideimages", (char *)NULL);
      return TCL_ERROR;
    }
  }
  if( ma != 0 ) {
    if( Tcl_GetBoolean(interp, argv[ma], &ma) != TCL_OK )
      return TCL_ERROR;
    
    if( ma != papyrus->mask_doc() ) {
      papyrus->mask_doc((BOOL)ma);
      change = TRUE;
    }
  }
  if( hi != 0 ) {
    if( Tcl_GetBoolean(interp, argv[hi], &hi) != TCL_OK )
      return TCL_ERROR;

    if( papyrus->hide_images() != hi ) {
      papyrus->hide_images((BOOL)hi);
      change = TRUE;
    }
  }
  if( change == TRUE )
    if( current.shape != NULL ) {
      Page *page;
      page = current.shape->get_page_parent();
      page->set_to_redraw(REDRAW_ME);
      ((Line *)page->get_child(0))->set_to_redraw(REDRAW_PAGE);
      page->draw_margins();
      current.doc->draw_frame(0,0);
    }

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * PapyrusQuery_Cmd --
 *
 *      This procedure is invoked to process the "PapyQuery" or
 *      "pq" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
PapyrusQuery_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int i;


  if( argc < 2 ) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], " options\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  for(i=1; i<argc; i++) {
    if( i > 1 )
      Tcl_AppendResult(interp, " ");

    if( argv[i][0] == 'm' && strcmp(argv[i], "mask") == 0 ) {
      Tcl_AppendResult(interp, (papyrus->mask_doc() == TRUE) ? "true " : "false",
		       (char *)NULL);

    } else if( argv[i][0] == 'i' && strcmp(argv[i], "hideimages") == 0 ) {
      Tcl_AppendResult(interp,
		       papyrus->hide_images() ? "true" : "false",
		       (char *)NULL);
    } else {
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be mask or hideimages", (char *)NULL);
      return TCL_ERROR;
    }
  }

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * ParagraphConfigure_Cmd --
 *
 *      This procedure is invoked to process the "ParaConfigure" or
 *      "pc" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
ParagraphConfigure_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int i;
  int al, fl, ma, ta, st, ns, fo;
  Paragraph *para;
  Page *page;
  Line *line;


  al = fl = ma = ta = st = ns = fo = 0;

  if( argc < 3 || (argc&1) == 0 ) {
    Tcl_AppendResult(interp,
		     "wrong # args:  should be \"", argv[0], " options values\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  for(i=1; i<argc; i += 2) {
    if( argv[i][0] != '-' )
      goto error;
    if( argv[i][1] == 'a' && strcmp(argv[i], "-align") == 0 )        al = i+1;
    else if( argv[i][1] == 'f' && strcmp(argv[i], "-fline") == 0 )   fl = i+1;
    else if( argv[i][1] == 'm' && strcmp(argv[i], "-margins") == 0 ) ma = i+1;
    else if( argv[i][1] == 't' && strcmp(argv[i], "-tag") == 0 )     ta = i+1;
    else if( argv[i][1] == 's' && strcmp(argv[i], "-style") == 0 )   st = i+1;
    else if( argv[i][1] == 'n' && strcmp(argv[i], "-nstyle") == 0 )  ns = i+1;
    else if( argv[i][1] == 'f' && strcmp(argv[i], "-font") == 0 )    fo = i+1;
    else {
    error:
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be -align, -fline, -margins, -tag, -style, -nstyle or -font",
		       (char *)NULL);
      return TCL_ERROR;
    }
  }

  para = current.shape->get_para();
  page = current.shape->get_page_parent();
  line = current.shape->get_line_parent();
  page->set_to_redraw(REDRAW_ME);

  if( st != 0 ) {
    StyleItem *si;
    
    si = papyrus->query_style(argv[st]);
      
      if( si == NULL ) {
	Tcl_AppendResult(interp, "unknown style \"",
			 argv[st], "\"", (char *)NULL);
	return TCL_ERROR;
      }
    para->configure(STYLE_POINTER, (void *)si);
    papyrus->set_current_font( (FontItem *)para->query(STYLE_FONT) );
    para->recompute(current);
  }

  if( al != 0 ) {
    StyleAlignType align;
    Line *first;

    if( Tcl_GetAlignment(interp, argv[al], align) != TCL_OK )
    return TCL_ERROR;

    para->configure(STYLE_ALIGNMENT, (void *)align);    
    first = First_of_Para_In_Page(para, page);
    first->set_to_redraw(REDRAW_PARA);
  }

  if( fl != 0 ) {
    Line *first;

    if( Tcl_GetPixels(interp, argv[fl], &fl) != TCL_OK )
      return TCL_ERROR;
    para->configure(STYLE_FLINE_MARGIN, (void *)fl);

    first = First_of_Para(line);
    first->can_fit(current);
    first->format_frame();
    first->set_to_redraw(REDRAW_ME);
  }
  
  if( ma != 0 ) {
    int m[4];

    if( Tcl_GetMargins(interp, argv[ma], m) != TCL_OK )
      return TCL_ERROR;
    
    para->configure(STYLE_TOP_MARGIN,    (void *)m[0]);
    para->configure(STYLE_BOTTOM_MARGIN, (void *)m[1]);
    para->configure(STYLE_LEFT_MARGIN,   (void *)m[2]);
    para->configure(STYLE_RIGHT_MARGIN,  (void *)m[3]);    

    para->recompute(current);
  }


  if( ta != 0 ) {
    Line *first;

    if( Tcl_GetInt(interp, argv[ta], &ta) != TCL_OK )
      return TCL_ERROR;
    para->configure(STYLE_TAG, (void *)ta);
    first = First_of_Para(line);
    first->set_to_redraw(REDRAW_ME);
  }

  if( ns != 0 )
    para->configure(STYLE_NEXT_STYLE, (void *)argv[ns]);
  
  if( fo != 0 ) {
    FontItem *fi;

    if( Tcl_GetFont(interp, argv[fo], fi) != TCL_OK )
      return TCL_ERROR;
    para->configure(STYLE_FONT, (void *)fi);
    papyrus->set_current_font( (FontItem *)para->query(STYLE_FONT) );
    para->recompute(current);
  }

  current.doc->draw_frame(0,0);
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * ParagraphQuery_Cmd --
 *
 *      This procedure is invoked to process the "ParaQuery" or
 *      "pq" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
ParagraphQuery_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Paragraph *para;
  char result[128];
  FontItem *fi;
  int i;


  if( argc < 2 ) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], " options\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  para = current.shape->get_para();

  for(i=1; i<argc; i++) {
    if( argv[i][0] == 'a' && strcmp(argv[i], "align") == 0 ) {
      strcpy(result, Alignment_to_String((StyleAlignType)
					 para->query(STYLE_ALIGNMENT)));
      
    } else if( argv[i][0] == 'f' && strcmp(argv[i], "fline") == 0 ) {
      sprintf(result, "%d", (int)para->query(STYLE_FLINE_MARGIN));

    } else if( argv[i][0] == 'm' && strcmp(argv[i], "margins") == 0 ) {
      sprintf(result, "%d %d %d %d",
	      (int)para->query(STYLE_TOP_MARGIN),
	      (int)para->query(STYLE_BOTTOM_MARGIN),
	      (int)para->query(STYLE_LEFT_MARGIN),
	      (int)para->query(STYLE_RIGHT_MARGIN));

    } else if( argv[i][0] == 't' && strcmp(argv[i], "tag") == 0 ) {
      sprintf(result, "%d", (int)para->query(STYLE_TAG));

    } else if( argv[i][0] == 's' && strcmp(argv[i], "style") == 0 ) {
      strcpy(result, para->get_bstyle()->get_attr(STYLE_NAME));

    } else if( argv[i][0] == 'n' && strcmp(argv[i], "nstyle") == 0 ) {
      strcpy(result, (char *)para->query(STYLE_NEXT_STYLE));

    } else if( argv[i][0] == 'f' && strcmp(argv[i], "font") == 0 ) {
      fi = (FontItem *)para->query(STYLE_FONT);
      sprintf(result, "%s %s %d",
	      fi->get_family(),
	      FontStyle_to_String(fi->get_style()),
	      fi->get_size());
    } else {
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be align, fline, margins, tag, style, nstyle or font",
		       (char *)NULL);
      return TCL_ERROR;
    }
    Tcl_AppendElement(interp, result);
  }
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * RefreshScreen_Cmd --
 *
 *      This procedure is invoked to process the "RefreshScreen" or
 *      "rs" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
RefreshScreen_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  Page *page;

  if( argc == 2 )
    papyrus->clear_area(0, 0, current.doc->get_width(), current.doc->get_height());
  else
    if( argc != 1 ) {
      interp->result = "Usage: RefreshScreen";
      return TCL_ERROR;
    }
  
  page = current.shape->get_page_parent();
  ((Frame *)page->get_child(0))->set_to_redraw(REDRAW_PAGE);
  
  page->set_to_redraw(REDRAW_ME);
  current.doc->draw_frame(0,0);
  
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * SelectRegion_Cmd --
 *
 *      This procedure is invoked to process the "SelectRegion" or
 *      "sr" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
SelectRegion_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  
  Frame *frame;
  int xsel, ysel, pos;
  int curX, curY;


  if( argc != 3 ) {
    interp->result = "Usage: RegionSelect";
    return TCL_ERROR;
  }

  if( Tcl_GetInt(interp, argv[1], &curX) != TCL_OK )
    return TCL_ERROR;

  if( Tcl_GetInt(interp, argv[2], &curY) != TCL_OK )
    return TCL_ERROR;


  frame = current.doc->xy_to_frame(curX, curY, pos);

  if( curY < 0 )
    return TCL_OK;

  frame->frame_to_xy(xsel, ysel, pos);

  fprintf(stderr, "%d %d\n", curY, ysel);

  if( xsel == current.doc->get_curXsel() && ysel == current.doc->get_curYsel() )
    return TCL_OK;


  if( ysel != current.doc->get_curYsel() ) {
    Line *line, *next_line;
    int x, y, xx, yy, pos, xxx, yyy;
    Shape *shape;
    

    shape = (Shape *)current.doc->xy_to_frame(current.doc->get_curXsel(),
					      current.doc->get_curYsel(), pos);
    line = shape->get_line_parent();
    line->frame_to_xy(x, y, 0);


    if( ysel > current.doc->get_curYsel() )
      next_line = (Line *)line->get_next_same_container();
    else {
      next_line = (Line *)line->get_previous_same_container();
    }
    if( next_line == NULL )
      return TCL_ERROR;
    

    next_line->frame_to_xy(xx, yy, 0);
        
    shape = (Shape *)current.doc->xy_to_frame(xsel, yy, pos);
    shape->frame_to_xy(xxx, yyy, pos);

    
    if( ysel > current.doc->get_curYsel() ) {
      papyrus->reverse_area(current.doc->get_curXsel(), current.doc->get_curYsel(),
			  x + line->get_real_width() - current.doc->get_curXsel(),
			  current.doc->get_curHeight());
      papyrus->reverse_area(xx, yy,
			  xxx-xx,
			  current.doc->get_curHeight());
    } else {
      papyrus->reverse_area(x, current.doc->get_curYsel(),
			  current.doc->get_curXsel() - x,
			  current.doc->get_curHeight());
      papyrus->reverse_area(xxx, yy,
			  next_line->get_real_width()+xx-xxx,
			  current.doc->get_curHeight());
    }
    current.doc->set_curYsel(yyy);
    current.doc->set_curXsel(xxx);
    return TCL_OK;
  }
  
  if( xsel < current.doc->get_curXsel() )
    papyrus->reverse_area(xsel, current.doc->get_curYsel(),
			current.doc->get_curXsel() - xsel,
			current.doc->get_curHeight());  
  else
    papyrus->reverse_area(current.doc->get_curXsel(), current.doc->get_curYsel(),
			xsel - current.doc->get_curXsel(),
			current.doc->get_curHeight());

  current.doc->set_curXsel(xsel);  

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * SetZoom_Cmd --
 *
 *      This procedure is invoked to process the "SetZoom" or
 *      "sz" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
SetZoom_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int nzoom;
  float new_zoom;
  Page *page;
  int i;

  if( argc != 2 ) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		     " zoom_percent\"", (char *)NULL);
    return TCL_ERROR;
  } 

  if( Tcl_GetInt(interp, argv[1], &nzoom) != TCL_OK )
    return TCL_ERROR;

  new_zoom = (float)((float)nzoom / 100);

  papyrus->resize_fonts((float)new_zoom/zoom);

  for(i=0; i<current.doc->get_children_num(); i++) {
    page = (Page *)current.doc->get_child(0);
    papyrus->resize_window(page->get_win(), (float)new_zoom/zoom, (float)new_zoom/zoom);
  }

  papyrus->set_current_font(NULL);   // To force 'set_current_font' to recompute
  zoom = new_zoom;

  return TRUE;
}



/*
 *----------------------------------------------------------------------
 *
 * SetKey_Cmd --
 *
 *      This procedure is invoked to process the "SetKey" or
 *      "sk" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
SetKey_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  ShortCut *node;

  if( argc != 4 ) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
                     " shortcut command comments\"", (char *)NULL);
    return TCL_ERROR;
  }

  switch( (int)is_global_key(argv[1]) ) {

  case PARTIAL:
    interp->result = "Error, this shortcup has same beginning than a stored shorcut";
    return TCL_ERROR;
    break;

  case NULL:
    node = new ShortCut;    
    node->len = strlen(argv[1]);
    node->keys = strdup(argv[1]);
    node->cmd = strdup(argv[2]);
    node->info = strdup(argv[3]);
    node->next = GKeys;
    GKeys = node;
    break;
    
  default:
    Tcl_AppendResult(interp, "Error, shortcut \"", argv[1],
		     "\": already exists", (char *)NULL);
    return TCL_ERROR;
  }
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * WordQuery_Cmd --
 *
 *      This procedure is invoked to process the "WordQuery" or
 *      "wq" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
WordQuery_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  char string[128];
  int i;
  void *result;

  if( argc < 2 ) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], " options\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  for(i=1; i<argc; i++) {
    if( argv[i][0] == 's' && strcmp(argv[i], "style") == 0 ){
      result = current.shape->get_absolute_attr(current.attr, FONT_STYLE_ATTR);
      strcpy(string, FontStyle_to_String((FontStyle)result));

    } else if( argv[i][0] == 's' && strcmp(argv[i], "size") == 0 ) {
      result = current.shape->get_absolute_attr(current.attr, FONT_SIZE_ATTR);
      sprintf(string, "%d", (int)result);

    } else if( argv[i][0] == 'u' && strcmp(argv[i], "underline") == 0 ) {
      result = current.shape->get_absolute_attr(current.attr, FONT_UNDERLINE_ATTR);
      sprintf(string, "%d", (int)result);

    } else if( argv[i][0] == 'f' && strcmp(argv[i], "family") == 0 ) {
      result = current.shape->get_absolute_attr(current.attr, FONT_FAMILY_ATTR);
      if( result != NULL )
	strcpy(string, (char *)result);

    } else if( argv[i][0] == 'v' && strcmp(argv[i], "voffset") == 0 ) {
      result = current.shape->get_absolute_attr(current.attr, VOFFSET_ATTR);
      sprintf(string, "%d", (int)result);

    } else if( argv[i][0] == 'b' && strcmp(argv[i], "background") == 0 ) {
      result = current.shape->get_absolute_attr(current.attr, BACKGROUND_ATTR);
      sprintf(string, "%d", (int)result);
      
    } else if( argv[i][0] == 'f' && strcmp(argv[i], "foreground") == 0 ) {
      result = current.shape->get_absolute_attr(current.attr, FOREGROUND_ATTR);
      sprintf(string, "%d", (int)result);
      
    } else {
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i],
		       "\": must be style, size, underline, family, ",
		       "voffset, background or foreground", (char *)NULL);
      return TCL_ERROR;
    }
    if( i > 1 )
      Tcl_AppendResult(interp, " ", (char *)NULL);
    Tcl_AppendResult(interp, string, (char *)NULL);
  }

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * SetCursor_Cmd --
 *
 *      This procedure is invoked to process the "SetCursor" or
 *      "sc" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
SetCursor_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int state;

  if( argc != 2 ) {
    Tcl_AppendResult(interp,
		     "wrong # args: should be \"", argv[0], " state\"", (char *)NULL);
    return TCL_ERROR;
  }

  if( Tcl_GetBoolean(interp, argv[1], &state) != TCL_OK )
    return TCL_ERROR;

  /*
   * If current document has just been killed
   */
  if( current.doc == NULL )
    return TCL_OK;

  if( state == 1 ) {
    current.doc->draw_cursor(current.shape, current.pos);
#ifdef HAVE_GUI
    GUI_MakePageVisible(papyrus->get_disp(),
			current.shape->get_page_parent()->get_win(),
			current.doc->get_curXpos(), current.doc->get_curYpos());
#endif
  } else
    current.doc->clear_cursor(current.shape, current.pos);
  
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * WordConfigure_Cmd --
 *
 *      This procedure is invoked to process the "WordConfigure" or
 *      "wc" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
WordConfigure_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  int i, sz, ud, fa, bg, fg, st, vo;
  FontStyle style;
  
  vo = st = sz = ud = fa = bg = fg = 0;

  if( argc == 1 ) {
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], " ?options?\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  for(i=2; i<argc; i += 2) {
    if( strcmp(argv[i-1], "-style") == 0 )           st = i;
    else if( strcmp(argv[i-1], "-size") == 0 )       sz = i;
    else if( strcmp(argv[i-1], "-underline") == 0 )  ud = i;
    else if( strcmp(argv[i-1], "-family") == 0 )     fa = i;
    else if( strcmp(argv[i-1], "-voffset") == 0 )    vo = i;
    else if( strcmp(argv[i-1], "-background") == 0 ) bg = i;
    else if( strcmp(argv[i-1], "-foreground") == 0 ) fg = i;
    else {
      Tcl_AppendResult(interp, "Error: unknown option \"", argv[i-1],
		       "\": must be -style, -size, -underline, -family",
		       "-voffset, -background or -foreground", (char *)NULL);
      return TCL_ERROR;
    }
  }

  if( i == argc ) {
    Tcl_AppendResult(interp, "Error: value for \"", argv[i-1], "\" missing", (char *)NULL);
    return TCL_ERROR;
  }

  if( st > 0 ) {
    if( Tcl_GetFontStyle(interp, argv[st], style) == TCL_OK
       && Check_Font_Attributes(FONT_STYLE_ATTR, (void *)style) == TRUE )
      current.attr->add_attr(FONT_STYLE_ATTR, (void *)style);
    else
      return TCL_ERROR;
  }
  
  if( sz > 0 ) {
    if( Tcl_GetInt(interp, argv[sz], &sz) != TCL_OK )
      return TCL_ERROR;
    current.attr->add_attr(FONT_SIZE_ATTR, (void *)sz);
  }
  
  if( ud > 0 ) {
    if( Tcl_GetBoolean(interp, argv[ud], &ud) != TCL_OK )
      return TCL_ERROR;      
    Tcl_AppendResult(interp, "WordConfigure: underline not yet implemented",
		     (char *)NULL);
  }

  if( fa > 0 )
    if( Check_Font_Attributes(FONT_FAMILY_ATTR, (void *)argv[fa]) == TRUE )
      current.attr->add_attr(FONT_FAMILY_ATTR, (void *)strdup(argv[fa]));
    else
      return TCL_ERROR;

  if( vo > 0 ) {
    if( Tcl_GetVoffset(interp, argv[vo], (VOffsetType&)vo) != TCL_OK )
      return TCL_ERROR;
    current.attr->add_attr(VOFFSET_ATTR, (void *)argv[vo]);
  }

  if( bg > 0 )
    current.attr->add_attr(BACKGROUND_ATTR, (void *)argv[bg]);

  if( fg > 0 )
    current.attr->add_attr(FOREGROUND_ATTR, (void *)argv[fg]);      

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * WordOutput_Cmd --
 *
 *      This procedure is invoked to process the "WordOutput" or
 *      "wo" PAPyRUS command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
WordOutput_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {

  switch( argc ) {

  case 1:
    Word *parent;

    parent = (Word *)current.shape->get_parent();
    for(int i=0; i<parent->get_children_num(); i++)
      Tcl_AppendResult(interp, ((Shape *)parent->get_child(i))->get_string(),
		       (char *)NULL);
    break;

  case 2:
    if( Tcl_GetChannelForWrite(interp, argv[1], channel) != TCL_OK )
      return TCL_ERROR;
    ((Frame *)current.shape->get_parent())->output_ascii();
    break;

  default:
    Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0], " ?channelId?\"",
		     (char *)NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}
