/* 
 * shortcuts.c --
 *
 *      This file contains the fonctions which manage the key
 *      bindings.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/keysym.h>

#include "shortcuts.h"
#include "../kernel/globals.h"
#include "../tcl/mytcl.h"

ShortCut *GKeys = NULL;



/*
 *----------------------------------------------------------------------
 *
 * is_global_key --
 *
 *      This method is invoked to know if a string is a registered
 *      shortcut.
 *
 * Results:
 *      Returns the shortcut, PARTIAL if the shortcut is not
 *      complete and NULL else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

ShortCut *
is_global_key(char *s) {

  int l;
  ShortCut *node, *begin;

  l = strlen(s);
  node = GKeys;
  begin = NULL;
  
  while( node != NULL ) {
    if( (node->keys[0] == s[0]) && (strncmp(node->keys, s, l) == 0) )
      if( node->len == l )
	return node;
      else
	begin = node;
    node = node->next;
  }

  if( begin != NULL && begin->keys[l] == ' ' )
    return (ShortCut *)PARTIAL;
  return NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * keysym_to_string --
 *
 *      Given a keysym, this method translates it into a string.
 *
 * Results:
 *      Returns the lenght of the string generated.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
keysym_to_string(KeySym keysym, unsigned int state, char *string) {

  char *tmp, *keystr;

  tmp = string;
  
  if( state & ControlMask) {
    *string++ = 'C';
    *string++ = '-';
  } else
    if( state & Mod1Mask) {
      *string++ = 'M';
      *string++ = '-';
    }

  if( (int)keysym <= 32 || (int)keysym > 255  ) {
    keystr = XKeysymToString(keysym);
    strcpy(string, keystr);
  } else {
    *string++ = keysym;
    *string = '\0';
  }

  return strlen(tmp);
}



/*
 *----------------------------------------------------------------------
 *
 * key_compute --
 *
 *      Given a keysym, this method executes the registered command
 *      or waits for more inputs (keysyms).
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
key_compute(KeySym keysym, unsigned int state) {

  static char string[32];
  static int offset = 0;
  char cmd[128];
  ShortCut *gkey;
  int i;


  if( offset > 0 )
    string[offset++] = ' ';

  offset += keysym_to_string(keysym, state, string+offset);
  gkey = is_global_key(string);
  if( gkey == PARTIAL )
    return;

  Tcl_MyEval("SetCursor off");

  if( gkey == NULL ) {
    if( offset == 1 || string[1] == ' ' ) { 
      i = 1;
      string[1] = '\0';
      do {
	if( string[0] == ';' )
	  sprintf(cmd, "InsertString \";\"");
	else
	  sprintf(cmd, "InsertCharacter %c", string[0]);
	Tcl_MyEval(cmd);
	i += 2;
	string[0] = string[i-1];
      } while( (string[i] == ' ' || string[i] == '\0') && i <= offset );
    }
  } else
    Tcl_MyEval(gkey->cmd);
  
  Tcl_MyEval("SetCursor on");

  offset = 0;
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * key_compute --
 *
 *      Given a keysym, this method records keysyms since the "C-g"
 *      is not entered.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
key_record(KeySym keysym, unsigned int state) {

  static int offset = 0;
  char string[8];
  static char sc[32];
  char cmd[32], info[32], command[128];

  offset += keysym_to_string(keysym, state, string);

  if( strcmp(string, "C-g") == 0 ) {
    offset -= 3;
    if( offset == 0 ) {
      fprintf(stderr, "Record global key aborded\n");
    } else {
      sc[ offset-1 ] = '\0';
      fprintf(stdout, "Enter infos to describe the shortcup '%s' :", sc); gets(info);
      fprintf(stdout, "Enter command to shortcup '%s' :", sc); gets(cmd);

      sprintf(command, "%s %s %s", sc, cmd, info);
      Tcl_MyEval(command);
    }
    offset =0;
    sc[0] = '\0';
    return;
  }
  strcat(sc, string);
  strcat(sc, " ");
  offset++;
}



/*
 *----------------------------------------------------------------------
 *
 * keyPress_handler --
 *
 *      This method is used to process key pressed press events.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
keyPress_handler(
#ifdef HAVE_GUI
Widget widget, XtPointer closure, XEvent *event, Boolean *b) {
#else
XEvent *event) {
#endif

  char string[2];
  KeySym keysym;
  
  XLookupString((XKeyPressedEvent*)event, string, 1, &keysym, NULL);

  if( keysym != XK_Control_L && keysym != XK_Control_R
     && keysym != XK_Alt_L && keysym != XK_Alt_R
     && keysym != XK_Shift_L && keysym != XK_Shift_R 
     && keysym != XK_Caps_Lock )
    key_compute(keysym, event->xkey.state);
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * display_all_shortcuts --
 *
 *      This method is used to display all the registered binding
 *      keys.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
display_all_shortcuts(void) {

  ShortCut *node;
  int i;
  
  i = 0;

  for(node=GKeys; node!=NULL; node=node->next) {
    i++;
    fprintf(stderr, "%s\t->\t%s : %s\n", node->keys, node->info, node->cmd);
  }
  return i;
}
