/* 
 * wordSegment.cc --
 *
 *      This file contains the definitions of the 'WordSegment' class
 *      methods.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


extern "C" {
#include <stdio.h>
#include <string.h>
#include <malloc.h>
}


#include "wordSegment.h"
#include "util.h"
#include "papyrus.h"



/*
 *----------------------------------------------------------------------
 *
 * WordSegment --
 *
 *      This procedure is invoked every time a WordSegment is created
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

WordSegment::WordSegment() {

}



/*
 *----------------------------------------------------------------------
 *
 * ~WordSegment --
 *
 *      This procedure is invoked every time a WordSegment is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Privates datas are deleted.
 *
 *----------------------------------------------------------------------
 */


WordSegment::~WordSegment() {

  delete(char *)_children;
  _children_num = 0;
}



/*
 *----------------------------------------------------------------------
 *
 * new_of_the_same_type --
 *
 *      This procedure is invoked by functions who want to instanciate a
 *      WordSegment without known anything about it.
 *
 * Results:
 *      A new WordSegment is returned.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
WordSegment::new_of_same_type(void) {

  return new WordSegment;
}



/*
 *----------------------------------------------------------------------
 *
 * alloc_offset --
 *
 *      This procedure is invoked everytime a child is inserted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Memory is reallocated only if there's not enough space to
 *      insert 'num' children.
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::alloc_offset(int num) {
  
  if( _children_max == 0 ) {
    _children_max = num+1;
    _children = (Container **)new char[_children_max];
  } else
    if( _children_max <= num ) {
      _children_max = num+1;
      _children = (Container **)realloc(_children, sizeof(char)*_children_max);
    }
  if( _children == NULL ) {
    fprintf(stderr, "Container::alloc_offset: Can't alloc children array\n");
    exit(1);
  }
}



/*
 *----------------------------------------------------------------------
 *
 * delete_children --
 *
 *      This procedure deletes children from position 'where' to
 *      'where'+'howmany'.
 *      position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childs are moved if "where" is not equal to "_children_num".
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::delete_children(int howmany, int where) {

  char *tab = (char *)_children;

  if( _children_num > 0 ) {
    strncpy(tab+where, tab+where+howmany, _children_num-howmany-where);
    _children_num -= howmany;
  }

  if( _children_num == 0 ) {          // Tells its father to delete it
    _parent->delete_children(1, get_offset());
    delete this;
    return;
  }

  recompute_size();
  tab[_children_num] = '\0';
}



/*
 *----------------------------------------------------------------------
 *
 * insert_children --
 *
 *      This procedure inserts 'howmany' children ('what') at 'where'
 *      position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childs are moved if 'where' is not equal to '_children_num'.
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::insert_children(Container **what, int howmany, int where) {

  int i;
  char *tab;

  alloc_offset( _children_num + howmany );
  tab = (char *)_children;

  for(i=_children_num-1; i>=where; i--)
    tab[i+howmany] = tab[i];

  strncpy(tab+where, (char *)what, howmany);

  _children_num += howmany;

  recompute_size();
  tab[_children_num] = '\0';
}



/*
 *----------------------------------------------------------------------
 *
 * move_children --
 *
 *      This function moves 'homany' children from position 'isrc'
 *      to container 'dst' at position 'idst'.
 *      The function returns if 'delete_children' was called (TRUE/FALSE),
 *      so we can call 'recompute_size' for example.
 *
 *      WordSegment daesn't use the container method 'move_children'
 *      because of the computation of '_children[isrc]' :
 *        - Container:    _children[isrc] = _children + iscr * sizeof(Container *)
 *        - WordSegment:  _children[isrc] = _children + iscr * sizeof(char)
 *
 *      and sizeof(char) != sizeof(Container *)
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Childrens are moved.
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::move_children(int howmany, int isrc, Container *dst, int idst) {
  
  char *tab = (char *)_children;

  dst->insert_children((Container **)&tab[isrc], howmany, idst);
  delete_children(howmany, isrc);
}



/*
 *----------------------------------------------------------------------
 *
 * debug --
 *
 *      This function gives some informations about the WordSegment.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::debug(int depth) {
  
  int i;

  for(i=0; i<depth; i++)
    fprintf(stderr, "    ");

  fprintf(stderr, 
	  "%s (%d) (%d %d) %s : [%s]\n", type(), _children_num,
	  UNZOOM(get_width()), UNZOOM(get_height()), 
	  (_attr == NULL) ? "" : "X", (char *)_children);
}



/*
 *----------------------------------------------------------------------
 *
 * can_fit --
 *
 *      This function looks if the WordSegment is correctly composed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The WordSegment may be splited.
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::can_fit(ThePosition &cur) {

  if( cur.attr->has_mark() ) {
    WordSegment *child, *ws;
    
    child = (WordSegment *)split_container(cur.pos);
    ws = (WordSegment *)child->split_container(1);
    
    if( child->_attr == NULL )
      child->_attr = new Attributes;
    
    /*
     * Adds before the 'cur.shape' attributes
     */
    cur.shape = child;
    cur.pos = 0;

    child->copy_attributes( cur.attr );
    cur.attr->delete_attributes();

    if( interactive_flag == TRUE && child->get_children_num() > 1 )
      child = (WordSegment *)child->split_container(1);
    child->recompute_size();
  }

  if( cur.pos == 0 && cur.shape->has_attributes() )
    Sort_Attributes(get_para(), cur.shape->get_attributes());

  if( interactive_flag == TRUE )
    get_line_parent()->can_fit(cur);
}



/*
 *----------------------------------------------------------------------
 *
 * recompute_size --
 *
 *      This function recomputes the WordSegment dimensions (height & width).
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The dimensions are updated.
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::recompute_size(BOOL propagate) {
  
  FontItem *fi;
  int offset;

  fi = get_ws_font();

  /*
   * Compute a font for an exponent
   */
  if( fi == NULL ) {
    fi = (FontItem *)get_para()->query(STYLE_FONT);
    fprintf(stderr, "Erreur !!!");
    exit(1);
  }

  set_width( fi->get_text_width((char *)_children, _children_num) );
  set_ascent( fi->get_ascent() );
  set_descent( fi->get_descent() );    


  if( has_attributes(VOFFSET_ATTR) ) {
    offset = ((FontItem *)get_para()->query(STYLE_FONT))->get_ascent() / 3;

    if( (VOffsetType)_attr->get_attr(VOFFSET_ATTR) == SUPERSCRIPT_VOFFSET )
      set_ascent(get_ascent() + offset);
    else                                    // SUBSCRIPT_VOFFSET
      set_descent(get_descent() + offset);
  }

  if( (propagate == TRUE) && (_parent != NULL) )
    ((Frame *)_parent)->recompute_size();
}



/*
 *----------------------------------------------------------------------
 *
 * draw_frame --
 *
 *      This function draws the WordSegment.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::draw_frame(int x, int y) {

  FontItem *fi;
  int offset;

  fi = get_ws_font();

  if( has_attributes(VOFFSET_ATTR) ) {
    offset = (VOffsetType)_attr->get_attr(VOFFSET_ATTR);
    y += (offset * ((FontItem *)get_para()->query(STYLE_FONT))->get_ascent() / 3);
  }

  papyrus->set_current_font(fi);
  papyrus->draw_string(x, y+get_line_parent()->get_ascent(), (char *)_children, _children_num);
}



/*
 *----------------------------------------------------------------------
 *
 * frame_to_xy --
 *
 *      This function returns the coordinates of the 'child' character
 *      of the WordSegment.
 *
 * Results:
 *      Returns the coordinates of the 'child' character.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
WordSegment::frame_to_xy(int &x, int &y, int child) {
  
  Container *parent;
  FontItem *fi;

  x = y = 0;

  parent = get_parent();
  if( parent == NULL )
    return;

  ((Frame *)parent)->frame_to_xy(x,y, get_offset());
  
  fi = get_ws_font();
  
  x += fi->get_text_width((char *)_children, child);
}



/*
 *----------------------------------------------------------------------
 *
 * get_absolute_attr --
 *
 *      This function returns the value of the AttrType 't' of the
 *      current WordSegment which has the 'attr' attributes.
 *
 *      The functions looks :
 *          - in 'attr' values
 *          - if none is found, then it looks in shape attributes
 *          - so, in last of case, the function returns Paragraph
 *            (where shape is) attributes. 
 *
 * Results:
 *      Returns the value of the attribute.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void *
WordSegment::get_absolute_attr(Attributes *attr, AttrType t) const {

  void *r;
  Paragraph *para;

  para = get_para();

  if( attr != NULL && attr->has_mark(t) )
    r = attr->get_attr(t);
  else if( has_attributes(t) )
    r = get_attributes()->get_attr(t);
  else
    switch( t ) {
    case FONT_STYLE_ATTR:
      r = (void *)((FontItem *)para->query(STYLE_FONT))->get_style();
      break;

    case FONT_SIZE_ATTR:
      r = (void *)((FontItem *)para->query(STYLE_FONT))->get_size();
      break;

    case FONT_FAMILY_ATTR:
      r = (void *)((FontItem *)para->query(STYLE_FONT))->get_family();
      break;

//    case FONT_UNDERLINE_ATTR:
//      r = (void *)((FontItem *)para->query(STYLE_FONT))->get_uline();
//      break;

    default:
      fprintf(stderr, "WordSegment::get_abolute_attr: attribute %d not yet available", t);
      r = NULL;
    }
  return r;
}



/*
 *----------------------------------------------------------------------
 *
 * get_ws_font --
 *
 *      This function returns the FontItem of the WordSegment.
 *      The functions looks :
 *          - in 'attr' values
 *          - if none is found, then it looks in Paragraph attributes.
 *
 * Results:
 *      Returns the FontItem of the WordSegment.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontItem *
WordSegment::get_ws_font(void) {

  FontItem *fi;
  char *fa;
  FontStyle st;
  int sz;

  fi = (FontItem *)get_para()->query(STYLE_FONT);
  fa = fi->get_family();
  st = fi->get_style();
  sz = fi->get_size();

  if( has_attributes(FONT_ATTR) ) {
    if( has_attributes(FONT_STYLE_ATTR) )
      st = (FontStyle)_attr->get_attr(FONT_STYLE_ATTR);
    
    if( has_attributes(FONT_FAMILY_ATTR) )
      fa = (char *)_attr->get_attr(FONT_FAMILY_ATTR);
    
    if( has_attributes(FONT_SIZE_ATTR) )
      sz = (int)_attr->get_attr(FONT_SIZE_ATTR);
    
    fi = papyrus->get_font(fa, st, sz);
  }
  return fi;
}



/*
 *----------------------------------------------------------------------
 *
 * xy_to_frame --
 *
 *      This function returns the 'child' which has the correct
 *      coordinates.
 *
 * Results:
 *      Returns the 'child' specified.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Frame *
WordSegment::xy_to_frame(int x, int y, int &offset) {

  int xx;
  FontItem *fi;
  
  fi = get_ws_font();

  for(offset=1; offset<=_children_num; offset++) {
    xx = fi->get_text_width((char *)_children, offset);
    if( xx > x )
      break;
  }
  offset--;
  return this;
}



/*
 *----------------------------------------------------------------------
 *
 * merge_container --
 *
 *      This is a convenience function to concatenate two WordSegment.
 *      In fact, this function puts all the 'con' children at the end
 *      of the container 'this'.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The result is : this + con -> this
 *
 *----------------------------------------------------------------------
 */


void
WordSegment::merge_container(Container *con) {

  con->move_children(con->get_children_num(), 0, this, get_children_num());
}



/*
 *----------------------------------------------------------------------
 *
 * split_container --
 *
 *      This function splits the WordSegment, at 'offset' position.
 *
 * Results:
 *      The new WordSegment created (second part of the current).
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
WordSegment::split_container(int offset) {

  WordSegment *result;

  result = (WordSegment *)Frame::split_container(offset);
  
  if( result != this && has_attributes() ) {
    Attributes *a;
    
    a = new Attributes;
    result->set_attributes(a);

    a->copy_attributes(get_attributes());
  }
  return result;
}
