/* 
 * word.cc --
 *
 *      This file contains the definitions of the 'Word' class
 *      methods.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include "wordSegment.h"
#include "word.h"
#include "line.h"
#include "../tcl/commands.h"



/*
 *----------------------------------------------------------------------
 *
 * Word --
 *
 *      This procedure is invoked every time a Word is created.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Word::Word() {
}



/*
 *----------------------------------------------------------------------
 *
 * ~Word --
 *
 *      This procedure is invoked every time a Word is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Word::~Word() {

}



/*
 *----------------------------------------------------------------------
 *
 * new_of_the_same_type --
 *
 *      This procedure is invoked by functions who want to instanciate a
 *      Word without known anything about it.
 *
 * Results:
 *      A new Word is returned.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
Word::new_of_same_type(void) {

  return new Word;
}



/*
 *----------------------------------------------------------------------
 *
 * format_frame --
 *
 *      This function looks if the word correctly composed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The current Word is splited if there's some child with
 *      differents attributes.
 *
 *----------------------------------------------------------------------
 */

void
Word::format_frame(ThePosition &cur) {

  Shape *previous, *shape;

  shape = previous = NULL;

  if( cur.pos == 0 ) {
    shape = cur.shape;
    previous = (Shape *)shape->get_previous_same_container();
  } else {
    if( cur.pos == cur.shape->get_children_num() ) {
      shape = (Shape *)cur.shape->get_next_same_container();
      previous = cur.shape;
    }
  }

  /*
   * Be carreful to change this, because either 'shape' and 'previous'
   * can be NULL
   */
  if( shape != NULL && shape->has_same_parent(previous) == TRUE )
    if( ((WordSegment *)previous)->get_ws_font() == 
        ((WordSegment *)shape)->get_ws_font() ) {
      cur.pos = previous->get_children_num();
      cur.shape = previous;
      previous->merge_container(shape);
    }
  
  cur.shape->get_line_parent()->format_frame();
}



/*
 *----------------------------------------------------------------------
 *
 * recompute_size --
 *
 *      This function recomputes the Word dimensions (ascent & descent).
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The dimensions are updated.
 *
 *----------------------------------------------------------------------
 */

void
Word::recompute_size(BOOL propagate) {

  unsigned int w, a, d, max_ascent, max_descent;
  int i;

  w = 0;
  max_ascent = max_descent = 0;

  for(i=0; i<_children_num; i++) {

    w += ((WordSegment *)_children[i])->get_width();
    a = ((WordSegment *)_children[i])->get_ascent();
    d = ((WordSegment *)_children[i])->get_descent();
    if( a > max_ascent )
      max_ascent = a;
    if( d > max_descent )
      max_descent = d;
  }
  set_width( w );
  set_ascent( max_ascent );
  set_descent( max_descent );

  if( (propagate == TRUE) && (_parent != NULL) )
    ((Line *)_parent)->recompute_size();
}



/*
 *----------------------------------------------------------------------
 *
 * debug --
 *
 *      This function gives some informations about the Word.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Word::debug(int depth) {

  int i;

  for(i=0; i<depth; i++)
    fprintf(stderr, "    ");

  fprintf(stderr,
	  "%s (%d) (%d %d)\n", type(), _children_num,
	  UNZOOM(get_width()), UNZOOM(get_height()));

  for(i=0; i<_children_num; i++)
    _children[ i ]->debug( depth+1 );
}



/*
 *----------------------------------------------------------------------
 *
 * draw_frame --
 *
 *      This function draw the Word.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Word::draw_frame(int x, int y) {

  int i;

  for(i=0; i<_children_num; i++) {
    ((Frame *)_children[i])->draw_frame(x, y);
    x += ((Frame *)_children[i])->get_width();
  }
}



/*
 *----------------------------------------------------------------------
 *
 * frame_to_xy --
 *
 *      This function returns the coordinates of the 'child' child
 *      of the Word.
 *
 * Results:
 *      Returns the coordinates of the 'child'.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Word::frame_to_xy(int &x, int &y, int child) {
  
  int i;
  Container *parent;

  x = y = 0;

  parent = get_parent();

  if( parent == NULL )
    return;

  ((Frame *)parent)->frame_to_xy(x, y, get_offset());

  for(i=0; i<child; i++)
    x += ((Frame *)get_child(i))->get_width();
}
