/* 
 * paragraph.cc --
 *
 *      This file contains the definitions of the 'Paragraph' class
 *      methods.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

extern "C" {
#include <string.h>  // for strdup
}

#include "page.h"
#include "line.h"
#include "shape.h"
#include "paragraph.h"
#include "../tcl/commands.h"
#include "util.h"
#include "papyrus.h"




/*
 *----------------------------------------------------------------------
 *
 * Paragraph --
 *
 *      This procedure is invoked every time a Paragraph is created.
 *      This function has no argument, so it is invoked for the
 *      creation of the first Paragraph of the documents.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized.
 *
 *----------------------------------------------------------------------
 */

Paragraph::Paragraph() {

  _bstyle = papyrus->query_style("Normal");

  if( _bstyle == NULL ) {
    fprintf(stderr, "Error: \"Normal\" style not found, ");
    fprintf(stderr, "check if the init files aren't corrupted.\n");
    exit(1);
  }
  _dstyle = NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * Paragraph --
 *
 *      Given a Paragraph, this function creates th 'Next' paragraph.
 *      This function is invoked for all the Paragraph creation
 *      except the first one.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized.
 *
 *----------------------------------------------------------------------
 */

Paragraph::Paragraph(Paragraph *p) {

  char *nstyle;

  nstyle = (char *)p->query(STYLE_NEXT_STYLE);

  if( nstyle == NULL )
    _bstyle = papyrus->query_style("Normal");
  else {
    _bstyle = papyrus->query_style(nstyle);
    if( _bstyle == NULL )
      _bstyle = papyrus->query_style("Normal");
  }
  _dstyle = NULL;

  if( _bstyle == p->get_bstyle() )
    p->copy_dstyle( this );
}



/*
 *----------------------------------------------------------------------
 *
 * ~Paragraph --
 *
 *      This procedure is invoked every time a Paragraph is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Paragraph::~Paragraph() {
}



/*
 *----------------------------------------------------------------------
 *
 * recompute --
 *
 *      This function updates the contents of a Paragraph.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas are updated.
 *
 *----------------------------------------------------------------------
 */

void
Paragraph::recompute(ThePosition &cur) {

  Line *line, *first, *last;
  Page *page;
  int i, j, pos;
  Container *child, *parent;


  line = cur.shape->get_line_parent();
  page = cur.shape->get_page_parent();

  line = First_of_Para_In_Page(this, page);
  page->set_to_redraw(REDRAW_ME);
  line->set_to_redraw(REDRAW_PAGE);
  
  first = First_of_Para(line);
  
  /*
   * Recompute all Frames of the paragraph
   */
  line = first;
  pos = 0;
  parent = cur.shape->get_parent();

  for(i=0; i<cur.shape->get_offset(); i++)
    pos += parent->get_child(i)->get_children_num();
  pos += cur.pos;


  while( first->is_in_same_para(line) == TRUE ) {
    
    line->set_x_offset( ((FontItem *)line->get_para()->query(STYLE_FONT))
		       ->get_spacing() );
    
    for(i=0; i<line->get_children_num(); i++) {
      child = line->get_child(i);
      
      for(j=0; j<child->get_children_num(); j++) {
	if( ((Shape *)child->get_child(j))->has_attributes() )
	  ((Shape *)child->get_child(j))->set_attributes(NULL);
	
	if( j > 0 ) {
	  child->get_child(j-1)->merge_container(child->get_child(j));
	  j--;
	}
      }

      ((Frame *)child->get_child(0))->recompute_size(FALSE);
      ((Frame *)child)->recompute_size(FALSE);
    }
    line->recompute_size();
    line = (Line *)line->get_next_same_container();
  }

  /*
   * When the style of a paragraph is changed, the shape attributes
   * are cleared, so we have to update its position.
   */
  cur.shape = (Shape *)cur.shape->get_parent()->get_child(0);
  cur.pos = pos;

  first->can_fit(cur, TRUE);
  first->format_frame();

  /*
   * Recompute the last line of paragraph, because of the computation
   * last line may not be the same as before. So if the last line has
   * a bottom margin, the size dasn't include it.
   */

  if( (int)query(STYLE_BOTTOM_MARGIN) != 0 ) {
    last = Last_of_Para(current.shape->get_line_parent());
    last->recompute_size();
  }
}



/*
 *----------------------------------------------------------------------
 *
 * configure --
 *
 *      Given an StyleAttrType and a value, this function adds or
 *      or removes (if the object has already it) an attribute of the
 *      Paragraph.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The class attributes are updated.
 *
 *----------------------------------------------------------------------
 */

void
Paragraph::configure(StyleAttrType t, void *value) {

  if( t == STYLE_POINTER ) {
    _bstyle = (StyleItem *)value;
    set_marker(0);
  } else {
    if( _dstyle == NULL )
      _dstyle = new StyleItem;
    
    if( _bstyle->get_attr(t) != value ) {
      add_mark(t);
      _dstyle->set_attr(t, value);
    } else
      unset_mark(t);
  }
  if( has_mark() == 0 ) {
    delete _dstyle;
    _dstyle = NULL;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * copy_dstyle --
 *
 *      This function copies the contents of the filed '_dstyle'
 *      into Paragraph 'p' given in argument.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The 'p' attributes may be updated.
 *
 *----------------------------------------------------------------------
 */

void
Paragraph::copy_dstyle(Paragraph *p) {

  int i;
  StyleAttrType at;
  
  if( _dstyle == NULL )
    return;

  for(i=1; i<=STYLE_FONT; i<<=1) {
    at = (StyleAttrType)i;
    if( has_mark(at) )
      p->configure(at, (void *)query(at));
  }
}
