/* 
 * papyrus_output.cc --
 *
 *      This file contains all the methods used to translate the
 *      document into PAPyRYS specific format.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <tcl.h>

#include "frame.h"
#include "document.h"
#include "page.h"
#include "line.h"
#include "wordSegment.h"
#include "image.h"
#include "util.h"
#include "output.h"



static Paragraph *cur_para;
static Attributes *cur_attr;
static Tcl_DString outputString, wsString, paraString;



/*
 *----------------------------------------------------------------------
 *
 * WordSegment::output_papyrus --
 *
 *      This function generates the contents of the 'WordSegment'
 *      object into PAPyRUS specific format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
WordSegment::output_papyrus(void) {

  char *string;
  int i;

  if( get_children_num() == 0 )
    return TCL_OK;

  if( has_attributes() || (cur_attr != NULL && cur_attr->has_mark()) ) {
    int i;
    void *value;    
    
    for(i=0; i<8; i++)
      if( has_attributes(1<<i) || cur_attr != NULL && cur_attr->has_mark(1<<i) ) {
	value = get_absolute_attr(get_attributes(), (AttrType)(1<<i));
	
	if( cur_attr != NULL && cur_attr->get_attributes() != NULL &&
	   cur_attr->get_attr((AttrType)(1<<i)) == value )
	  continue;

	switch( (AttrType) 1<<i ) {
	  
	case FONT_SIZE_ATTR:
	  OUTS2(wsString, "-size ");
	  OUTI2(wsString, value);
	  break;
	  
	case FONT_STYLE_ATTR:
	  OUTS2(wsString, "-style");
	  OUTE2(wsString, FontStyle_to_String((FontStyle)value));	  
	  break;
	  
	case FONT_FAMILY_ATTR:
	  OUTS2(wsString, "-family");
	  OUTE2(wsString, value);
	  break;  
	  
	case BACKGROUND_ATTR:
	case FOREGROUND_ATTR:
	case VOFFSET_ATTR:
	case FONT_UNDERLINE_ATTR:
	  break;
	}
      }
    if( Tcl_DStringLength(&wsString) > 0 ) {
      FLUSH(outputString, "InsertString \"", "\"\n");
      OUTS("WordConfigure ");
      OUTS(OUTS2(wsString, "\n"));
      Tcl_DStringSetLength(&wsString, 0);
    }
  }

  string = (char *)_children;
  for(i=0; i<_children_num; i++) {
    if( string[i] == '{' || string[i] == '}' ||
       string[i] == ']' || string[i] == '[' ||
       string[i] == '"' )
      OUTS2(outputString, "\\");
    OUTC2(outputString, string[i]);
  }
  cur_attr = get_attributes();
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Image::output_papyrus --
 *
 *      This function generates the contents of the 'Image'
 *      object into PAPyRUS specific format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Image::output_papyrus(void) {
 
  FLUSH(outputString, "InsertString \"", "\"\n");
  OUTS("InsertImage ");
  OUTS(_filename);
  OUTS("\n");
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Line::output_papyrus --
 *
 *      This function generates the contents of the 'Line'
 *      object into PAPyRUS specific format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Line::output_papyrus(void) {

  int i;

  for(i=0; i<_children_num-1; i++) {
    ((Frame *)_children[i])->output_papyrus();
    OUTS2(outputString, " ");
  }
  if( i < _children_num )
    ((Frame *)_children[i])->output_papyrus();
  return TCL_OK;
}




/*
 *----------------------------------------------------------------------
 *
 * Document::output_papyrus --
 *
 *      This function generates the contents of the 'Document'
 *      object into PAPyRUS specific format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Document::output_papyrus(void) {

  int i, j;
  Page *page;
  Line *line;
  Paragraph *para;


  cur_attr = NULL;
  cur_para = NULL;

  Tcl_DStringInit(&outputString);
  Tcl_DStringInit(&wsString);
  Tcl_DStringInit(&paraString);

  
  if( _title != NULL ) {
    OUTS2(outputString, " -title");
    OUTE2(outputString, _title);
  }
  if( _owner != NULL ) {
    OUTS2(outputString, " -owner");
    OUTE2(outputString, _owner);    
  }
  FLUSH(outputString, "DocConfigure", "\n");

  line = (Line *)get_child(0)->get_child(0);
  para = line->get_para();

  /*
   * Force to output the attributes of the first paragraph
   * for two reasons:
   *   - to prevent wrong configuration of the first paragraph,
   *     if the user inserts the document in another one.
   *   - to speed up the code generation, else we have to insert
   *     in the loop a test to prevent from NULL comparison.
   */
  para->output_papyrus();

  for(i=0; i<_children_num; i++) {
    page = (Page *)get_child(i);

    for(j=0; j<page->get_children_num(); j++) {
      line = (Line *)page->get_child(j);
      
      if( line->get_para() != para ) {
	FLUSH(outputString, "InsertString \"", "\"\n");
	OUTS("NewParagraph \n");
	line->get_para()->output_papyrus();
	para = line->get_para();
      } else
	if( i != 0 || j != 0 )
	  OUTS2(outputString, " ");
      line->output_papyrus();
    }
  }
  FLUSH(outputString, "InsertString \"", "\"\n");

  Tcl_DStringFree(&outputString);

  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Frame::output_papyrus --
 *
 *      This function generates the contents of the 'Frame'
 *      object into PAPyRUS specific format. This function
 *      is used by all the object which have no output_papyrus
 *      method.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Frame::output_papyrus(void) {
  
  int i;

  for(i=0; i<_children_num; i++)
    ((Frame *)_children[i])->output_papyrus();
  return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Paragraph::output_papyrus --
 *
 *      This function generates the contents of the 'Paragraph'
 *      object into PAPyRUS specific format.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Paragraph::output_papyrus(void) {

  static Paragraph *para = new Paragraph;


  if( strcmp((char *)query(STYLE_NAME), (char *)para->query(STYLE_NAME)) ) {
    OUTS2(paraString, " -style");
    OUTE2(paraString, query(STYLE_NAME));
  }

  if( has_mark(STYLE_ALIGNMENT) || para->has_mark(STYLE_ALIGNMENT) ) {
    StyleAlignType align;
    
    align = (StyleAlignType)query(STYLE_ALIGNMENT);
    if( align != (StyleAlignType)para->query(STYLE_ALIGNMENT) ) {
      OUTS2(paraString, " -align");
      OUTE2(paraString, Alignment_to_String(align));
    }
  }
  if( has_mark(STYLE_FLINE_MARGIN) || para->has_mark(STYLE_FLINE_MARGIN) ) {
    int fline;
    
    fline = (int)query(STYLE_FLINE_MARGIN);
    if( fline != (int)para->query(STYLE_FLINE_MARGIN) ) {
      OUTS2(paraString, " -fline ");
      OUTI2(paraString, fline);
    }
  }
  if( has_mark(STYLE_MARGINS) ) {
    OUTS2(paraString, " -margins ");
    if( has_mark(STYLE_TOP_MARGIN) )     OUTI2(paraString, query(STYLE_TOP_MARGIN));
    if( has_mark(STYLE_BOTTOM_MARGIN) )  OUTI2(paraString, query(STYLE_BOTTOM_MARGIN));
    if( has_mark(STYLE_LEFT_MARGIN) )    OUTI2(paraString, query(STYLE_LEFT_MARGIN));
    if( has_mark(STYLE_RIGHT_MARGIN) )   OUTI2(paraString, query(STYLE_RIGHT_MARGIN));
  }
  if( has_mark(STYLE_TAG) ) {
    OUTS2(paraString, " -tag ");
    OUTI2(paraString, query(STYLE_TAG));
  }
  FLUSH(paraString, "ParaConfigure", "\n");
  para = this;

  return TCL_OK;
}
