/* 
 * line.cc --
 *
 *      This file contains the definitions of the 'Line' class
 *      methods.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include "line.h"
#include "word.h"
#include "page.h"

#include "isfuncs.h"
#include "util.h"
#include "papyrus.h"



/*
 *----------------------------------------------------------------------
 *
 * Line --
 *
 *      This procedure is invoked every time a Line is created.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized.
 *
 *----------------------------------------------------------------------
 */

Line::Line(Paragraph *p) {

  FontItem *fi;

  fi = (FontItem *)p->query(STYLE_FONT);
  _x_offset  = fi->get_spacing();

  set_ascent( fi->get_ascent() );
  set_descent( fi->get_descent() );

  _para = p;
}



/*
 *----------------------------------------------------------------------
 *
 * ~Line --
 *
 *      This procedure is invoked every time a Line is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Line::~Line() {
}



/*
 *----------------------------------------------------------------------
 *
 * new_of_the_same_type --
 *
 *      This procedure is invoked by functions who want to instanciate a
 *      Line without known anything about it.
 *
 * Results:
 *      A new Line is returned.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Container *
Line::new_of_same_type(void) {

  return new Line(get_para());
}



/*
 *----------------------------------------------------------------------
 *
 * recompute_size --
 *
 *      This function recomputes the Line dimensions (height & width).
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The dimensions are updated.
 *
 *----------------------------------------------------------------------
 */

void
Line::recompute_size(BOOL propagate) {

  unsigned int w, a, d, max_ascent, max_descent;
  int i;
  FontItem *fi;

  fi = (FontItem *)get_para()->query(STYLE_FONT);

  w = 0;
  max_ascent = fi->get_ascent();
  max_descent = fi->get_descent();
  
  for(i=0; i<_children_num; i++) {

    w += ((Word *)_children[i])->get_width();
    a = ((Word *)_children[i])->get_ascent();
    d = ((Word *)_children[i])->get_descent();

    if( a > max_ascent )
      max_ascent = a;
    if( d > max_descent )
      max_descent = d;
  }

  if( is_first_of_para() == TRUE )
    max_ascent  += (unsigned int)get_para()->query(STYLE_TOP_MARGIN);

  if( is_last_of_para() == TRUE )
    max_descent += (unsigned int)get_para()->query(STYLE_BOTTOM_MARGIN);

  if( get_height() != (max_ascent + max_descent) )
    set_to_redraw(REDRAW_PAGE);

  set_width( w );
  set_ascent( max_ascent );
  set_descent( max_descent );

  if( (propagate == TRUE) && (_parent != NULL) )
    ((Page *)_parent)->recompute_size();
}



/*
 *----------------------------------------------------------------------
 *
 * debug --
 *
 *      This function gives some informations about the Line.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Line::debug(int depth) {

  int i;

  for(i=0; i<depth; i++)
    fprintf(stderr, "    ");

  fprintf(stderr, "%s (%d) (%d %d : %d)  %c...%p \t %d\n", type(), _children_num,
	  UNZOOM(get_width()), UNZOOM(get_height()), UNZOOM(get_max_width()),
	  (has_to_redraw() == REDRAW_ME) ? 'M' : 
	  (has_to_redraw() == REDRAW_PAGE) ? 'P' :  ' ', get_para(),
	  get_ascent());

  for(i=0; i<_children_num; i++)
    _children[ i ]->debug( depth+1 );
}



/*
 *----------------------------------------------------------------------
 *
 * can_fit --
 *
 *      This function looks if the Line is not too big.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      If the Line is too big then the last Words are passed to
 *      the next Line.
 *
 *----------------------------------------------------------------------
 */

void
Line::can_fit(ThePosition &cur, BOOL forced) {

  float offset, width;
  Line *next, *new_line;
  int i;
  BOOL samePara;
  static BOOL first = TRUE;


  offset = _x_offset;
  _x_offset = ((FontItem *)get_para()->query(STYLE_FONT))->get_spacing();

  next = new_line = NULL;
  i = get_children_num();
  width = get_real_width();

  while( width > get_max_width() && i > 1 ) 
    if( ((Frame *)_children[--i])->get_width() > 0 )
      width -= UNZOOM(((Frame *)_children[i])->get_width() + _x_offset);

  if( i == get_children_num() )
    if( forced == FALSE)
      return;
    else
      new_line = (Line *)get_next_same_container();
  else {
    
    next = (Line *)get_next_same_container();
    samePara = is_in_same_para(next);
    
    if( samePara == FALSE &&  next != NULL )
      next->set_to_redraw(REDRAW_PAGE);

    if( samePara == FALSE ) {
      new_line = (Line *)split_container(i);
      new_line->set_to_redraw(REDRAW_ME);
    } else {
      move_children(get_children_num()-i, i, next, 0);
      new_line = next;
      recompute_size();
      next->set_to_redraw(REDRAW_ME);
    }
  }

  /*
   * Test if the next line (if exists) is in an another page.
   * If true, the current page has to call its 'can_fit' method
   */
  if( has_same_parent(next) == FALSE ) 
    ((Page *)get_parent())->can_fit(cur);
  else {
    if( first == TRUE ) {
      first = FALSE;
      ((Page *)get_parent())->can_fit(cur);      
      first = TRUE;
    }
  }
  
  if( is_in_same_para(new_line) == TRUE )
    new_line->can_fit(cur, forced);
  
  _x_offset = offset;
  
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * format_frame --
 *
 *      This function looks if the Line is big enough.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      If the current Line can accept more Words from the next
 *      Line, they are moved.
 *
 *----------------------------------------------------------------------
 */

void
Line::format_frame() {

  Line *next;
  Word *child;
  int width;
  int i = 0, x, y;
  float spacing;
  static BOOL first = TRUE;
  Container *previous;


  next = (Line *)get_next_same_container();

  if( is_in_same_para(next) == FALSE ) {
    ((Page *)get_parent())->format_frame();
    return;
  }

  spacing = _x_offset;
  _x_offset = ((FontItem *)get_para()->query(STYLE_FONT))->get_spacing();
  width = get_real_width();

  while( width <= get_max_width() ) {
    if( i == next->get_children_num() ) {
      next->frame_to_xy(x, y, 0);
      set_to_redraw(REDRAW_PAGE);
      merge_container(next);

      next = (Line *)get_next_same_container();

      if( is_in_same_para(next) == FALSE ) {
	if( next != NULL )
	  next->set_to_redraw(REDRAW_PAGE);
	break;
      }
      i = 0;
    }
    
    child = (Word *)next->get_child(i++);
    if( child->get_width() > 0 )
      width += UNZOOM(child->get_width() + _x_offset);
  }

  if( i > 1 && is_in_same_para(next) == TRUE ) {
    next->set_to_redraw(REDRAW_ME);
    next->move_children(i-1, 0, this, get_children_num());
  }
  
  next = (Line *)get_next_same_container();
  
  if( i > 0 && is_in_same_para(next) == TRUE ) {
      next->set_to_redraw(REDRAW_ME);
      
      if( first == TRUE ) {
	first = FALSE;
	next->format_frame();
	first = TRUE;
	previous = get_parent()->get_previous_same_container();
	if( previous != NULL )
       	  ((Page *)previous)->format_frame();
	else
	  ((Page *)get_parent())->format_frame();
      } else
	next->format_frame();
    }
  _x_offset = spacing;
}



/*
 *----------------------------------------------------------------------
 *
 * draw_frame --
 *
 *      This function draws the Line.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Line::draw_frame(int x, int y, BOOL end_of_para) {
  
  int i, offset, inter_offset;
  FontItem *fi;
     
  fi = (FontItem *)get_para()->query(STYLE_FONT);
  _x_offset = fi->get_spacing();
  inter_offset = 0;

  /*
   * Redrawing is done in a temporary pixmap that is allocated
   * here and freed at the end of the procedure.  All drawing is
   * done to the pixmap, and the pixmap is copied to the screen
   * at the end of the procedure.  This provides the smoothest
   * possible visual effects (no flashing on the screen).
   */

  papyrus->alloc_pixmap(x, y, (int)(((Frame *)_parent)->get_width() +
				  1+_x_offset + 
				  fi->get_text_width("", 1)),
		      get_height()+INTERLIGNE);
  

  x += get_lmargin();

  switch( (StyleAlignType)_para->query(STYLE_ALIGNMENT) ) {
    
  case STYLE_ALIGN_FULL:
    if( is_last_of_para() == FALSE )
      if( _children_num > 1 )
	_x_offset = ((float)(get_max_width() - get_width())) / ((float)_children_num - 1.00);
    inter_offset = (int)(_x_offset / 2.00);
    break;


  case STYLE_ALIGN_LEFT:
    break;


  case STYLE_ALIGN_RIGHT:
    x += (get_max_width() - ZOOM(get_real_width()));
    break;


  case STYLE_ALIGN_CENTER:
    x += (get_max_width() - ZOOM(get_real_width())) / 2;
    break;
  }

  /*
   * If the current paragraph is a list, and if the current
   * line is the first one, we have to draw a tag.
   */

  if( (int)get_para()->query(STYLE_TAG) != 0 && is_first_of_para() == TRUE )
    draw_line_tag(x, y); // +get_ascent());


  for(i=0; i<_children_num; i++) {
    offset = (int)(i*_x_offset);
    ((Frame *)_children[i])->draw_frame(x + offset, y);
    x += ((Frame *)_children[i])->get_width();

    if( papyrus->mask_doc() == TRUE )
      if( i < _children_num-1 )
	papyrus->draw_string(x + offset + inter_offset, y + get_ascent(), "", 1);
  }
  if( papyrus->mask_doc() == TRUE )
    if( end_of_para == TRUE )
      papyrus->draw_string(x + (int)((i-1)*_x_offset), y + get_ascent(), "", 1);
    
  set_to_redraw(REDRAW_NONE);
  papyrus->flush_pixmap();
}



/*
 *----------------------------------------------------------------------
 *
 * frame_to_xy --
 *
 *      This function returns the coordinates of the 'child' child
 *      of the Line.
 *
 * Results:
 *      Returns the coordinates of the 'child'.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Line::frame_to_xy(int &x, int &y, int child) {
  
  int i;
  Container *parent;

  x = y = 0;
  parent = get_parent();

  if( parent == NULL )
    return;
  
  ((Frame *)parent)->frame_to_xy(x, y, get_offset());
  x += get_lmargin();

  for(i=0; i<child; i++)
    x += ((Line *)get_child(i))->get_width();
  
  x += (int)(get_x_offset() * (float)child);

  switch( (StyleAlignType)_para->query(STYLE_ALIGNMENT) ) {

  case STYLE_ALIGN_RIGHT:
    if( ZOOM(get_real_width()) < get_max_width() )
      x += (get_max_width() - ZOOM(get_real_width()));
    break;
    
  case STYLE_ALIGN_CENTER:
    x += (get_max_width() - ZOOM(get_real_width())) / 2;
    break;
    
  default:
    break;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * xy_to_frame --
 *
 *      This function returns the 'child' which has the correct
 *      coordinates.
 *
 * Results:
 *      Returns the 'child' specified.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Frame *
Line::xy_to_frame(int x, int y, int &offset) {

  int i, width, inter;


  x -= get_lmargin();
  
  switch( (StyleAlignType)_para->query(STYLE_ALIGNMENT) ) {
    
  case STYLE_ALIGN_RIGHT:
    x -= (get_max_width() - ZOOM(get_real_width()));
    if( x < 0 ) x = 0;
    break;
    
  case STYLE_ALIGN_CENTER:
    x -= ((get_max_width() - ZOOM(get_real_width())) / 2);
    if( x < 0 ) x = 0;
    break;
    
  default:
    break;
  }

  width = inter = 0;
  
  for(i=0; i<_children_num; i++) {
    
    inter = (int)((float)(i+1)*_x_offset);
    
    if( (width + inter + ((Frame *)_children[i])->get_width()) <= x ) {
      width += ((Frame *)_children[i])->get_width();
    } else {
      inter = (int)((float)(i)*_x_offset);
      x -= (width + inter);
      return ((Frame *)_children[i])->xy_to_frame(x, y, offset);
    }
    
  }
  return ((Frame *)_children[_children_num-1])->xy_to_frame(x, y, offset);
}



/*
 *----------------------------------------------------------------------
 *
 * clear_frame --
 *
 *      This function clears the Line in the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Line::clear_frame(void) {

  ((Frame *)_parent)->set_to_redraw(REDRAW_ME);
  set_to_redraw(REDRAW_ME);
}



/*
 *----------------------------------------------------------------------
 *
 * is_last_of_para --
 *
 *      This function tells if a Line is the last of a Paragraph.
 *
 * Results:
 *      A boolean.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
Line::is_last_of_para(void) {
  
  Line *next_line;

  next_line = (Line *)get_next_same_container();

  return (is_in_same_para( next_line ) == TRUE) ? FALSE : TRUE;
}



/*
 *----------------------------------------------------------------------
 *
 * is_first_of_para --
 *
 *      This function tells if a Line is the first of a Paragraph.
 *
 * Results:
 *      A boolean.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
Line::is_first_of_para(void) {
  
  Line *prev_line;

  prev_line = (Line *)get_previous_same_container();

  return (is_in_same_para( prev_line ) == TRUE) ? FALSE : TRUE;
}



/*
 *----------------------------------------------------------------------
 *
 * is_in_same_para --
 *
 *      This function tells if two Lines are in the same Paragraph.
 *
 * Results:
 *      A boolean.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
Line::is_in_same_para(Container *con) {
  
  Line *line = (Line *)con;

  if( line != NULL )
    if( line->get_para() == get_para() )
      return TRUE;
  
  return FALSE;
}



/*
 *----------------------------------------------------------------------
 *
 * get_lmargin --
 *
 *      This function returns the left margin of the Line.
 *
 * Results:
 *      An integer.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Line::get_lmargin(void) {

  int marg;

  marg = (int)get_para()->query(STYLE_LEFT_MARGIN);
  
  if( is_first_of_para() == TRUE ) {
    marg += (int)get_para()->query(STYLE_FLINE_MARGIN);

    if( (int)get_para()->query(STYLE_TAG) != 0 )
      marg += 30;
  }
  return marg;
}



/*
 *----------------------------------------------------------------------
 *
 * draw_line_tag --
 *
 *      This function draws the tag of the beginning of the Line.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Line::draw_line_tag(int x, int y) {

  FontItem *fi_save, *fi;
  char label[2];
  int tag, size;
  
  tag = (int)get_para()->query(STYLE_TAG);
  fi_save = papyrus->get_current_font();

  size = ((FontItem *)get_para()->query(STYLE_FONT))->get_size();

  fi = papyrus->get_font("Symbol", NORMAL, size);
  if( fi == NULL ) {
    fprintf(stderr, "Line::draw_line_tag: font \"%s-%s-%d\" doesn't exist\n",
	    "Symbol", "normal",
	    size);
    exit(1);
  }
  papyrus->set_current_font(fi);
  
  label[0] = (char)tag;
  label[1] = '\0';
  papyrus->draw_string(x-15, y + get_ascent(), label, 1);  // - 0.5 cm
  papyrus->set_current_font(fi_save);
}
