/* 
 * isfuncs.cc --
 *
 *      This file contains some convenience functions which
 *      facilitate the kernel code.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */

#include "isfuncs.h"
#include "word.h"
#include "wordSegment.h"



/*
 *----------------------------------------------------------------------
 *
 * is_first_of_word --
 *
 *      This method tells if the cursor (given by 'cur') is at the
 *      beginning of a Word object.
 *
 * Results:
 *      Return TRUE if the cursor is at the beginning of a Word object
 *      and FALSE else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
is_first_of_word(ThePosition &cur) {

  if( cur.pos == 0 && cur.shape->get_offset() == 0 )
    return TRUE;
  return FALSE;
}



/*
 *----------------------------------------------------------------------
 *
 * is_last_of_word --
 *
 *      This method tells if the cursor (given by 'cur') is at the
 *      end of a Word object.
 *
 * Results:
 *      Return TRUE if the cursor is at the end of a Word object
 *      and FALSE else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
is_last_of_word(ThePosition &cur) {
  Word *word_parent;
  
  if( cur.pos == cur.shape->get_children_num() ) {
    word_parent = (Word *)cur.shape->get_parent();

    if( word_parent->get_child(LAST_POSITION) == cur.shape )
      return TRUE;
  }
  return FALSE;
}



/*
 *----------------------------------------------------------------------
 *
 * is_first_of_line --
 *
 *      This method tells if the cursor (given by 'cur') is at the
 *      beginning of a Line object.
 *
 * Results:
 *      Return TRUE if the cursor is at the beginning of a Line object
 *      and FALSE else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
is_first_of_line(ThePosition &cur) {

  if( is_first_of_word(cur) == TRUE )
    if( cur.shape->get_parent()->get_offset() == 0 )
      return TRUE;
  return FALSE;
}



/*
 *----------------------------------------------------------------------
 *
 * is_last_of_line --
 *
 *      This method tells if the cursor (given by 'cur') is at the
 *      end of a Line object.
 *
 * Results:
 *      Return TRUE if the cursor is at the end of a Line object
 *      and FALSE else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
is_last_of_line(ThePosition &cur) {
  Line *line_parent;

  if( is_last_of_word(cur) == TRUE ) {
    line_parent = cur.shape->get_line_parent();
    if( line_parent->get_child(LAST_POSITION) == cur.shape->get_parent() )
      return TRUE;
  }
  return FALSE;
}



/*
 *----------------------------------------------------------------------
 *
 * is_first_of_para -- (with a ThePosition as argument)
 *
 *      This method tells if the cursor (given by 'cur') is at the
 *      beginning of a Paragraph object.
 *
 * Results:
 *      Return TRUE if the cursor is at the beginning of a Paragraph
 *      object and FALSE else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
is_first_of_para(ThePosition &cur) {
  Line *line_parent, *prev_line;

  if( is_first_of_line(cur) == TRUE ) {
    line_parent = cur.shape->get_line_parent();
    prev_line = (Line *)line_parent->get_previous_same_container();

    if( line_parent->is_in_same_para(prev_line) == FALSE )
      return TRUE;
  }
  return FALSE;
}



/*
 *----------------------------------------------------------------------
 *
 * is_first_of_line -- (with a Line as argument)
 *
 *      This method tells if the Line 'line' is the first of the
 *      Paragraph object.
 *
 * Results:
 *      Return TRUE if 'line' is the first Line of the Paragraph
 *      object and FALSE else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
is_first_of_para(Line *line) {
  Line *prev_line;

  prev_line = (Line *)line->get_previous_same_container();
  
  return (BOOL)!line->is_in_same_para(prev_line);
}



/*
 *----------------------------------------------------------------------
 *
 * is_last_of_para --
 *
 *      This method tells if the cursor (given by 'cur') is at the
 *      end of a Paragraph object.
 *
 * Results:
 *      Return TRUE if the cursor is at the end of a Paragraph
 *      object and FALSE else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

BOOL
is_last_of_para(ThePosition &cur) {
  Line *line_parent, *next_line;

  if( is_last_of_line(cur) == TRUE ) {
    line_parent = cur.shape->get_line_parent();
    next_line = (Line *)line_parent->get_next_same_container();

    if( line_parent->is_in_same_para(next_line) == FALSE )
      return TRUE;
  }
  return FALSE;
}
