/* 
 * document.cc --
 *
 *      This file contains the definitions of the 'Document' class
 *      methods.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include "papyrus.h"
#include "document.h"
#include "page.h"
#include "line.h"
#include "word.h"
#include "wordSegment.h"
#include "paragraph.h"
#include "util.h"




/*
 *----------------------------------------------------------------------
 *
 * Document --
 *
 *      This procedure is invoked every time a Document is created,
 *      given the current position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized and the object tree that
 *      represents the document datas is created.
 *
 *----------------------------------------------------------------------
 */

Document::Document(ThePosition &cur) {

  Page *page;
  Line *line;
  Word *word;
  WordSegment *ws;


  _title = NULL;
  _owner = NULL;

  set_width( papyrus->xmm_to_pixels(210) );
  set_ascent( papyrus->ymm_to_pixels(297) );

  set_tmargin( papyrus->ymm_to_pixels(25) );
  set_bmargin( papyrus->ymm_to_pixels(25) );
  set_lmargin( papyrus->xmm_to_pixels(25) );
  set_rmargin( papyrus->xmm_to_pixels(25) );

  page = new Page;
  insert_children((Container **)&page,  1, 0);
  line = new Line( new Paragraph );
  page->insert_children((Container **)&line, 1, 0);
  word = new Word;
  line->insert_children((Container **)&word, 1, 0);
  ws = new WordSegment();
  word->insert_children((Container **)&ws, 1, 0);
  
  cur.shape = (Shape *)ws;
  cur.pos = 0;
}



/*
 *----------------------------------------------------------------------
 *
 * Document --
 *
 *      This procedure is invoked every time a Document is created, but
 *      without arguments. This function never have to be call.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Document::Document() {
}



/*
 *----------------------------------------------------------------------
 *
 * ~Document --
 *
 *      This procedure is invoked every time a Document is destroyed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas are deleted.
 *
 *----------------------------------------------------------------------
 */

Document::~Document() {

  if( _title != NULL )
    delete _title;

  if( _owner != NULL )
    delete _owner;
}



/*
 *----------------------------------------------------------------------
 *
 * new_of_the_same_type --
 *
 *      This procedure is invoked by functions who want to instanciate a
 *      Document without known anything about it.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      A new Document is returned.
 *
 *----------------------------------------------------------------------
 */

Container *
Document::new_of_same_type(void) {

  return new Document;
}



/*
 *----------------------------------------------------------------------
 *
 * draw_frame --
 *
 *      This function draws the document.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Document::draw_frame(int, int) {

  int i;

  if( interactive_flag == FALSE )
    return;

  for(i=0; i<_children_num; i++)
    if( ((Frame *)_children[i])->has_to_redraw() == REDRAW_ME)
      ((Frame *)_children[i])->draw_frame(0, 0);
  
  papyrus->set_win( current.shape->get_page_parent()->get_win() );
}



/*
 *----------------------------------------------------------------------
 *
 * recompute --
 *
 *      This function recomputes the document dimensions.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The dimensions are updated.
 *
 *----------------------------------------------------------------------
 */

void
Document::recompute(ThePosition &cur) {
  
  Line *line;
  Paragraph *para;


  /*
   * Save the current position
   */
  PositionSave();

  /*
   * Takes the first line of the document
   */
  line = (Line *)get_child(0)->get_child(0);
  cur.pos = 0;

  while( line != NULL ) {
    cur.shape = (Shape *)line->get_child(0)->get_child(0);
    para = line->get_para();
    para->recompute(cur);
    line = Last_of_Para(line);
    line = (Line *)line->get_next_same_container();
  }
}



/*
 *----------------------------------------------------------------------
 *
 * set_width --
 *
 *      This function sets the 'width' (width of all the pages) of
 *      the document.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The dimensions are updated.
 *
 *----------------------------------------------------------------------
 */

void
Document::set_width(unsigned int w) {

  float size;
  int i;
  Page *page;

  size = (float)get_width();
  Frame::set_width(w);

  /*
   * When a document is created, 'size' equal to zero,
   * so w/size will cause an Arithmetic exception.
   */
  if( size != 0 ) {
    for(i=0; i<_children_num; i++) {
      page = (Page *)_children[i];
      papyrus->resize_window( page->get_win(), ((float)w)/size, 1 );
    }
  }
}
