/* 
 * display.h --
 *
 *      This file contains the declaration of the 'XDisplay' class.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifndef DISPLAY_H
#define DISPLAY_H

extern "C" {
#include <X11/Xlib.h>
}

#include "paragraph.h"

#define POINT_TO_MM(pt)  (((float)(pt)) / 72.00 * 2.54)
#define MM_TO_POINT(mm)  (((float)(mm)) / 2.54 * 72.00)

/*
 * Be carrefull ROUND doesn't work correctly with negative fingers
 */
#define ROUND(x) ((int)((x) + 0.5))



class XDisplay {

private:

 public:
  Display     *_disp;    /* X11 Display                       */
  int        _screen;    /* X11 Screen                        */
  Window        _win;    /* Current X11 Window                */
  Window       _root;    /* X11 root Window                   */
  GC             _gc;    /* Normal X11 Graphic Context        */
  GC         _xor_gc;    /* XORed X11 Graphic Context         */
  GC      _dashed_gc;    /* DASHed X11 Graphic Context        */
  GC       _clear_gc;    /* X11 Graphic Context for clearing  */
  float        _xppm;    /* X number of pixels per minimeter  */
  float        _yppm;    /* Y number of pixels per minimeter  */
  FontItem      *_fi;    /* Current FontItem                  */
  Pixmap       _pix;     /* Pixmap used for drawing           */
  int          _destX;   /* Coordinates of the end os the selection */
  int          _destY;
  unsigned int _pixWidth, _pixHeight;        /* Dimensions of the pixmap         */
  unsigned int _pixMaxWidth, _pixMaxHeight;  /* Maximum dimensions of the pixmap */ 
  BOOL         _hide_images;       /* boolean that tells if images are displayed */


public:

  XDisplay(Display *); //, Window);
  ~XDisplay();

  inline Display *get_disp(void)     const   { return _disp;   }
  inline Window get_win(void)        const   { return _win;    }
  inline GC get_xor_gc(void)         const   { return _xor_gc; }
  inline GC get_clear_gc(void)       const   { return _clear_gc;  }
  inline GC get_dashed_gc(void)      const   { return _dashed_gc; }
  inline float get_xppm(void)        const   { return _xppm; }
  inline float get_yppm(void)        const   { return _yppm; }

  inline void set_win(Window w)              { _win = w;    }

  inline int xmm_to_pixels(int mm)  const
    { return ROUND( ((float)mm) * _xppm * WIDTH_SCALE ); }
  inline int ymm_to_pixels(int mm)  const
    { return ROUND( ((float)mm) * _yppm * HEIGHT_SCALE ); }
  
  inline int xpixels_to_mm(int pix)  const
    { return (int)(((float)pix) / _xppm * WIDTH_SCALE); }
  inline int ypixels_to_mm(int pix)  const
    { return (int)(((float)pix) / _yppm * HEIGHT_SCALE); }
  
  inline float xpixels_to_points(float pix)  const
    { return pix / _xppm / WIDTH_SCALE / 25.4 * 72.0; }
  inline int xpixels_to_points(int pix)  const
    { return (int)((float)pix / _xppm / WIDTH_SCALE / 25.4 * 72.0); }

  inline float ypixels_to_points(float pix)  const
    { return pix / _yppm / HEIGHT_SCALE / 25.4 * 72.0; }
  inline int ypixels_to_points(int pix)  const
    { return (int)((float)pix / _yppm / HEIGHT_SCALE / 25.4 * 72.0); }

  void draw_string(int, int, char *, int);
  void draw_box(int, int, int, int);
  void draw_pixmap(Pixmap *, int, int, unsigned short, unsigned short);

  void clear_area(int, int, unsigned int, unsigned int);
  void reverse_area(int, int, unsigned int, unsigned int);

  void resize_window(Window, float, float);

  inline FontItem *get_current_font(void)  const  { return _fi; }
  void set_current_font(FontItem *);

  inline void flush_disp(void)       const   { XFlush(_disp); }

  void alloc_pixmap(int, int, unsigned int, unsigned int);
  void flush_pixmap(void);

  Pixmap *get_xpm_file(char *, unsigned short &, unsigned short &);

  inline BOOL hide_images(void)  const  { return _hide_images; }
  inline void hide_images(BOOL b)       { _hide_images = b; }
};

#endif
