/* 
 * attributes.cc --
 *
 *      This file contains the definitions of the 'Attributes'
 *      class methods.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */

extern "C" {
#include <stdio.h>      // for NULL
}

#include "attributes.h"



/*
 *----------------------------------------------------------------------
 *
 * Attributes --
 *
 *      This method is invoked every time an Attributes object is
 *      created
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The field _attr is initialized.
 *
 *----------------------------------------------------------------------
 */

Attributes::Attributes() {
  _attr = NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * ~Attributes --
 *
 *      This method is invoked every time an Attributes object is
 *      deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Attributes::~Attributes() {
  _attr = NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * add_attr --
 *
 *      This method hads an attribute which type is 't' and value is
 *      'v' to the Attributes object.
 *      deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Attributes::add_attr(AttrType t, void *v) {

  Attribute *at;

  at = query_attr(t);
  if( at != NULL )
    del_attr(t);

  at = new Attribute;
  at->next = _attr;
  _attr = at;

  at->type  = t;
  at->value = v;
  
  add_mark(t);
}



/*
 *----------------------------------------------------------------------
 *
 * add_attr --
 *
 *      This method deletes the attribute which type is 't' from the
 *      Attributes object.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Attributes::del_attr(AttrType t) {

  Attribute *prev, *at;

  prev = at = _attr;
  while( at->type != t ) {
    prev = at;
    at = at->next;
  }
  if( prev != at )
    prev->next = at->next;
  else
    _attr = at->next;

  delete at;
  unset_mark(t);

  if( has_mark() == 0 ) {
    delete _attr;
    _attr = NULL;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * query_attr --
 *
 *      This method searches for an attribute which type is 't'.
 *
 * Results:
 *      Returns the attribute if found and NULL else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Attribute *
Attributes::query_attr(AttrType type) {

  Attribute *at;
  
  if( has_mark(type) ) {
    at = _attr;
    while( at != NULL )
      if( at->type == type )
	return at;
      else
	at = at->next;
  }
  return NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * operator= --
 *
 *      This method duplicates the Attributes object into another
 *      object.
 *
 * Results:
 *      Returns the created Attributes object.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Attributes *
Attributes::operator=(const Attributes &a) {

  Attributes *newA;
  Attribute *at;

  newA = new Attributes;

  newA->_attr = new Attribute;
  at = a._attr;
  
  while( at != NULL ) {
    newA->add_attr(at->type, at->value);
    at = at->next;
  }
  return newA;
}



/*
 *----------------------------------------------------------------------
 *
 * copy_attributes --
 *
 *      This method copies the contents of the Attributes object into
 *      another Attributes object.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Attributes::copy_attributes(Attributes *a) {
  
  Attribute *at;

  if( a == NULL )
    return;
  
  at = a->_attr;

  while( at != NULL ) {
    add_attr(at->type, at->value);
    at = at->next;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * delete_attributes --
 *
 *      This method deletes all the attributes of the object.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Attributes::delete_attributes(void) {

  Attribute *att;
  Attribute *at;

  at = _attr;
  while( at != NULL ) {
    att = at->next;
    delete at;
    at = att;
  }
  set_marker(0);
  _attr = NULL;
}


/*
 *----------------------------------------------------------------------
 *
 * operator== --
 *
 *      This method compares two Attributes objects.
 *
 * Results:
 *      Returns 0 if they are the same, and 1 else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Attributes::operator==(Attributes &a) {

  return !(_attr!=a._attr);
}


/*
 *----------------------------------------------------------------------
 *
 * operator!= --
 *
 *      This method compares two Attributes objects.
 *
 * Results:
 *      Returns 1 if they are the same, and 0 else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
Attributes::operator!=(Attributes &a) {

  Attribute *at;

  if( this == NULL )
    return 1;

  if( has_mark() != a.has_mark() )
    return 1;

  at = a._attr;

  while( at != NULL ) {
    if( get_attr(at->type) !=  a.get_attr(at->type) )
      return 1;
    at = at->next;
  }

  return 0;
}
