/* 
 * fontFamily.cc --
 *
 *      This file contains the definitions of the 'FontFamily' class
 *      methods.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


extern "C" {
#include <stdio.h>
#include <string.h>
}

#include "fontFamily.h"
#include "../globals.h"      // for strdup declaration



/*
 *----------------------------------------------------------------------
 *
 * FontFamily --
 *
 *      This method is invoked every time a FontFamily is created.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The datas class are initialized.
 *
 *----------------------------------------------------------------------
 */

FontFamily::FontFamily(char *family, FontStyle style, char *x11, char *ps) {

  _family = strdup(family);
  _style = style;
  _x11_name = strdup(x11);
  _ps_name = strdup(ps);
  _fontsPtr = NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * ~FontFamily --
 *
 *      This method is invoked every time a FontFamily is deleted.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontFamily::~FontFamily(void) {

  delete _x11_name;
  delete _ps_name;
}



/*
 *----------------------------------------------------------------------
 *
 * add_item --
 *
 *      This method is invoked to add a new fontItem in the FontFamily.
 *
 * Results:
 *      A created FontItem is returned.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontItem *
FontFamily::add_item(Display *display, int size) {

  FontItem *fiPtr;
  FontItem *tracePtr, *searchPtr;
  XFontStruct *fs;  
  char fontName[128], *ptr, prefix[64];

  fiPtr = new FontItem;
  fiPtr->size = size;
  fiPtr->refCount = 1;
  fiPtr->xfs = NULL;
  fiPtr->parent = this;

  ptr = strstr(_x11_name, "--");
  strncpy(prefix, _x11_name, ptr-_x11_name);
  prefix[ptr-_x11_name] = '\0';
  sprintf(fontName, "%s--%d-%s", prefix, size, ptr+4);

  fs = XLoadQueryFont(display, fontName);
  fiPtr->xfs = fs;

  fiPtr->spacing = fs->per_char[(int)'w' - fs->min_char_or_byte2].width;

  searchPtr = tracePtr = _fontsPtr;
  
  while( searchPtr != NULL ) {
    if( searchPtr->size > size )
      break;
    tracePtr = searchPtr;
    searchPtr = searchPtr->next;
  }
  if( tracePtr == searchPtr ) {
    fiPtr->next = _fontsPtr;
    _fontsPtr = fiPtr;
  } else {
    fiPtr->next = tracePtr->next;
    tracePtr->next = fiPtr;
  }
  return fiPtr;
}



/*
 *----------------------------------------------------------------------
 *
 * del_item --
 *
 *      This method is invoked to delete a FontItem from the FontManager.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
FontFamily::del_item(int size) {

  FontItem *tracePtr, *searchPtr;

  searchPtr = tracePtr = _fontsPtr;
  
  if( searchPtr == NULL )
    goto error;

  while( searchPtr->size < size ) {
    tracePtr = searchPtr;
    searchPtr = searchPtr->next;
    if( searchPtr == NULL )
      goto error;
  }
  /*
   * If the FontItem we have to delete is the first one,
   * the '_fontPtr' has to change.
   */
  if( tracePtr == searchPtr )
    _fontsPtr = searchPtr->next;
  else
    tracePtr->next = searchPtr->next;

  delete searchPtr;

  return;

 error:
  fprintf(stderr, "FontFamily::del_item: can't delete fontItem '%d'\n", size);
  exit(1);
}



/*
 *----------------------------------------------------------------------
 *
 * query_item --
 *
 *      This method is invoked to look for a FontItem.
 *
 * Results:
 *      Returns a FontItem if found and NULL else.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

FontItem *
FontFamily::query_item(int size) {

  FontItem *searchPtr;

  for(searchPtr=_fontsPtr; searchPtr != NULL; searchPtr=searchPtr->next)
    if( searchPtr->size == size )
      return searchPtr;
    else
      if( searchPtr->size > size )
	break;
  
  return NULL;
}
