/* 
 * widgets.cc --
 *
 *      This file contains several functions that creates
 *      or manipulates widgets.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <X11/cursorfont.h>

#include <Xm/Notebook.h>
#include <Xm/PushBG.h>
#include <Xm/LabelG.h>
#include <Xm/Label.h>
#include <Xm/ComboBox.h>
#include <Xm/RowColumn.h>
#include <Xm/MenuShell.h>
#include <Xm/PushB.h>
#include <Xm/List.h>
#include <Xm/Form.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/Frame.h>
#include <Xm/FileSB.h>
#include <Xm/ToggleBG.h>

#include <Xm/ScrolledW.h>
#include <Xm/ScrollBar.h>

#include <Xm/DrawingA.h>
#include <X11/Xmu/Drawing.h>

#include <X11/xpm.h>


#include "globals.h"
#include "widgets.h"
#include "callbacks.h"
#include "dialogs.h"
#include "modify_text.h"


#include "../pixmaps/new.xpm"
#include "../pixmaps/open.xpm"
#include "../pixmaps/save.xpm"
#include "../pixmaps/print.xpm"
#include "../pixmaps/spell.xpm"
#include "../pixmaps/cut.xpm"
#include "../pixmaps/copy.xpm"
#include "../pixmaps/paste.xpm"
#include "../pixmaps/clear.xpm"
#include "../pixmaps/macro.xpm"
#include "../pixmaps/list.xpm"
#include "../pixmaps/margin_sub.xpm"
#include "../pixmaps/margin_add.xpm"
#include "../pixmaps/special.xpm"
#include "../pixmaps/bold.xpm"
#include "../pixmaps/oblique.xpm"
#include "../pixmaps/underline.xpm"
#include "../pixmaps/left_align.xpm"
#include "../pixmaps/right_align.xpm"
#include "../pixmaps/center_align.xpm"
#include "../pixmaps/justify.xpm"

#include "../pixmaps/inew.xpm"
#include "../pixmaps/iopen.xpm"
#include "../pixmaps/isave.xpm"
#include "../pixmaps/iprint.xpm"
#include "../pixmaps/ispell.xpm"
#include "../pixmaps/icut.xpm"
#include "../pixmaps/icopy.xpm"
#include "../pixmaps/ipaste.xpm"
#include "../pixmaps/iclear.xpm"
#include "../pixmaps/imacro.xpm"
#include "../pixmaps/ilist.xpm"
#include "../pixmaps/imargin_sub.xpm"
#include "../pixmaps/imargin_add.xpm"
#include "../pixmaps/ispecial.xpm"
#include "../pixmaps/ibold.xpm"
#include "../pixmaps/ioblique.xpm"
#include "../pixmaps/iunderline.xpm"
#include "../pixmaps/ileft_align.xpm"
#include "../pixmaps/iright_align.xpm"
#include "../pixmaps/icenter_align.xpm"
#include "../pixmaps/ijustify.xpm"


BOOL Allow_Balloons;
Widget balloon;
GC drawA_gc;
Widget mainDrawing_hsb, mainDrawing_vsb;



char *fontFamily[] = { 
  "courier",
  "helvetica",
  "new century schoolbook",
  "times"
    
};

char *fontStyle[] = { 
  "normal",
  "italique",
  "gras",
  "gras italique"
};

char *fontSize[] = { 
   "8",  "9", "10", "11", "12", "14", "16", "18",
  "20", "22", "24", "26", "28", "36", "48", "72"
};


char **icon_pixmaps[] = { new_xpm, open_xpm, save_xpm, print_xpm, spell_xpm,
			  cut_xpm, copy_xpm, paste_xpm, clear_xpm, macro_xpm,
			  list_xpm, margin_sub_xpm, margin_add_xpm, special_xpm,
			  bold_xpm, oblique_xpm, underline_xpm, left_align_xpm,
			  right_align_xpm, center_align_xpm, justify_xpm
};


char **iicon_pixmaps[] = { inew_xpm, iopen_xpm, isave_xpm, iprint_xpm, ispell_xpm,
			   icut_xpm, icopy_xpm, ipaste_xpm, iclear_xpm, imacro_xpm,
			   ilist_xpm, imargin_sub_xpm, imargin_add_xpm, ispecial_xpm,
			   ibold_xpm, ioblique_xpm, iunderline_xpm, ileft_align_xpm,
			   iright_align_xpm, icenter_align_xpm, ijustify_xpm
};




/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateNoteBook --
 *
 *      This function creates a notebook widget with three pages.
 *
 * Results:
 *      Returns the notebook created.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateNoteBook(Widget parent) { 
  Widget notebook, tmp;
  
  notebook = XtVaCreateManagedWidget("notebook",
				     xmNotebookWidgetClass,
				     parent,
				     XmNbindingType, XmNONE, 
				     XmNnotebookChildType, XmMAJOR_TAB,
				     XmNorientation, XmVERTICAL,
				     XmNbackPagePlacement, XmTOP_RIGHT,
				     XmNpageChangedCallback, XmCR_NONE,
				     NULL);
  XtVaCreateManagedWidget("page1",
			  xmPushButtonGadgetClass,
			  notebook,
			  XmNnotebookChildType, XmMAJOR_TAB,
			  XmNpageNumber, 1,
			  XmNhighlightThickness, 0,
			  NULL);

  XtVaCreateManagedWidget("page2",
			  xmPushButtonGadgetClass,
			  notebook,
			  XmNnotebookChildType, XmMAJOR_TAB,
			  XmNpageNumber, 2,
			  XmNhighlightThickness, 0,
			  NULL);
  
  XtVaCreateManagedWidget("page3",
			  xmPushButtonGadgetClass,
			  notebook,
			  XmNnotebookChildType, XmMAJOR_TAB,
			  XmNpageNumber, 3,
			  XmNhighlightThickness, 0,
			  NULL);
  
  tmp = XtVaCreateWidget("label3",
			 xmPushButtonGadgetClass,
			 notebook,
			 XmNnotebookChildType, XmPAGE_SCROLLER,
			 NULL);
  XtDestroyWidget(tmp);

  return notebook;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateComboBox --
 *
 *      This function creates a combobox widget with three pages.
 *
 * Results:
 *      Returns the combobox created.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateComboBox(Widget parent) {
  Widget combo, combo_list;
  XmString xmstr;

  combo = XtVaCreateManagedWidget("ComboBox",
				  xmComboBoxWidgetClass,
				  parent,
				  XmNcomboBoxType, XmDROP_DOWN_COMBO_BOX,
				  NULL);

  combo_list = XtNameToWidget(combo, "*List");
		       
  XtVaSetValues(combo_list,
		XmNvisibleItemCount, 3,
		NULL);

  xmstr = XmStringCreateSimple("Courier");
  XmListAddItem(combo_list, xmstr, 0);
  xmstr = XmStringCreateSimple("Times");
  XmListAddItem(combo_list, xmstr, 0);
  xmstr = XmStringCreateSimple("Helvetica");
  XmListAddItem(combo_list, xmstr, 0);
  xmstr = XmStringCreateSimple("New Century Schoolbook");
  XmListAddItem(combo_list, xmstr, 0);

  XmComboBoxUpdate(combo);

  XmStringFree( xmstr );

  return combo;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateOptionMenu --
 *
 *      This function creates an option menu width 'number' entries,
 *      whose Intrinsic name is "*postfix...".
 *
 * Results:
 *      Returns the Option Menu created..
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateOptionMenu(Widget parent, unsigned int number, char *postfix,
		     XtCallbackProc cb) {
  
  Widget option_menu;
  int i;
  Widget pulldown_shell, pulldown;
  char name[32], label[32];

  pulldown_shell = XtVaCreatePopupShell("pulldown_shell",
					xmMenuShellWidgetClass, parent,
					XmNwidth, 1,
					XmNheight, 1,
					NULL );

  sprintf(name, "%sOM", postfix);
  pulldown = XtVaCreateWidget(name,
			      xmRowColumnWidgetClass,
			      pulldown_shell,
			      XmNrowColumnType, XmMENU_PULLDOWN,
			      NULL );
  
  for(i=0; i<number; i++) {
    Widget widget;
    
    sprintf(label, "%s_b%d", name, i);
    widget = XtVaCreateManagedWidget(label,
				     xmPushButtonGadgetClass,
				     pulldown,
				     NULL );
    XtAddCallback(widget, XmNactivateCallback, cb, (XtPointer)i);
  }
  sprintf(label, "%s_rc", name);
  option_menu = XtVaCreateWidget(label,
				 xmRowColumnWidgetClass,
				 parent,
				 XmNrowColumnType, XmMENU_OPTION,
				 XmNsubMenuId, pulldown,
				 XmNmarginWidth, 0,
				 XmNspacing, 0,
				 NULL);
  if( i > 0 )
    XtManageChild(option_menu);
  
  return option_menu;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateBalloon --
 *
 *      This function is invoked to create the help balloon on the
 *      push buttons.
 *
 * Results:
 *      Returns the Balloon created.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateBalloon(Widget parent) {

  Widget widget;

  widget = XtVaCreatePopupShell("balloon",
				overrideShellWidgetClass, parent,
				XmNallowShellResize, True,
				NULL);

  XtVaCreateManagedWidget("balloonLabel",
			  xmLabelWidgetClass,
			  widget,
			  XmNmarginWidth, 1,
			  XmNmarginHeight, 1,
			  NULL);
  Allow_Balloons = TRUE;
  return widget;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateIconBar --
 *
 *      This function is invoked to create 'num' push buttons (icons)
 *      in the tool bar.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_CreateIconBar(Widget parent, unsigned int num) {

  int i;
  static int offset=0;
  char name[32];
  Widget box, widget;
  Pixmap pixmap, ipixmap;

  box = XtVaCreateManagedWidget("iconBar_box",
				xmRowColumnWidgetClass,
				parent,
				XmNorientation, XmHORIZONTAL,
				XmNspacing, 0,
				NULL);

  for(i=0; i<num; i++) {
    sprintf(name, "button_b%d", offset);

    XpmCreatePixmapFromData(XtDisplay(toplevel),
			    RootWindowOfScreen(XtScreen(toplevel)),
			    icon_pixmaps[offset],
			    &pixmap,
			    NULL, NULL);

    XpmCreatePixmapFromData(XtDisplay(toplevel),
			    RootWindowOfScreen(XtScreen(toplevel)),
			    iicon_pixmaps[offset],
			    &ipixmap,
			    NULL, NULL);

    widget = XtVaCreateManagedWidget(name,
				     xmPushButtonWidgetClass,
				     box,
				     XmNlabelType, XmPIXMAP,
				     XmNlabelPixmap, pixmap,
				     XmNlabelInsensitivePixmap, ipixmap,
				     XmNmarginWidth, 0,
				     XmNmarginHeight, 0,
				     XmNhighlightThickness, 0,
				     NULL);
    XtAddCallback(widget, XmNactivateCallback, iconBar_CB, (XtPointer)offset++ );
  }
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateFieldItem --
 *
 *      This function is invoked to create a Field Item.
 *
 * Results:
 *      Returns the Field Item created.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateFieldItem(Widget parent, FIELD_TYPE type, unsigned int numColumns,
		    const char *label, int cd) {
  char name[32];
  Widget form, text_widget;
  void (*callback)();

  sprintf(name, "%s_form", label);
  form = XtVaCreateManagedWidget(name, xmFormWidgetClass, parent,
				 XmNfractionBase, 10,
				 NULL);

  sprintf(name, "%sFI", label);
  XtVaCreateManagedWidget(name,
			  xmLabelGadgetClass, form,
			  XmNtopAttachment,    XmATTACH_FORM,
			  XmNbottomAttachment, XmATTACH_FORM,
			  XmNleftAttachment,   XmATTACH_FORM,
			  XmNrightAttachment,   XmATTACH_POSITION,
			  XmNrightPosition,    3,
			  XmNalignment,        XmALIGNMENT_END,
			  NULL);

  text_widget = XtVaCreateManagedWidget("field_text",
					xmTextFieldWidgetClass, form,
					XmNtraversalOn,      True,
					XmNleftAttachment,   XmATTACH_POSITION,
					XmNleftPosition,     4,
					XmNcolumns, numColumns,
					NULL);
  switch( type ) {

  case PHONE_FIELD:
    callback = field_check_phone;
    break;
    
  case UPPERCASE_FIELD:
    callback = field_check_allcaps;
    break;

  case INTERGER_FIELD:
    callback = field_check_float; /* zip;*/
    break;

  case PASSWORD_FIELD:
    callback = field_check_passwd;
    break;

  default:
    XtRemoveAllCallbacks(text_widget, XmNactivateCallback);
    XtAddCallback(text_widget, XmNactivateCallback, fieldItem_CB, (XtPointer)cd);
    return form;
  }

  XtAddCallback(text_widget,
		XmNactivateCallback, callback, (XtPointer)numColumns);
  XtAddCallback(text_widget,
		XmNmodifyVerifyCallback, callback, (XtPointer)numColumns);
  XtAddCallback(text_widget,
		XmNvalueChangedCallback,field_value_changed,(XtPointer)numColumns);

  return form;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateFileNameField --
 *
 *      This function creates a Field Item width a pushbutton at the
 *      end, to looks for a filename.
 *
 * Results:
 *      Returns the Field Item created.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateFileNameField(Widget parent, unsigned int numColumns, const char *label,
			XtCallbackProc cb, int cd) {

  char name[32];
  Widget form, text_widget, push_button;

  sprintf(name, "%s_form", label);
  form = XtVaCreateManagedWidget(name, xmFormWidgetClass, parent,
				 XmNfractionBase,  10,
				 NULL);

  sprintf(name, "%sFNI", label);

  XtVaCreateManagedWidget(name,
			  xmLabelGadgetClass, form,
			  XmNtopAttachment,    XmATTACH_FORM,
			  XmNbottomAttachment, XmATTACH_FORM,
			  XmNleftAttachment,   XmATTACH_FORM,
			  XmNrightAttachment,  XmATTACH_POSITION,
			  XmNrightPosition,    3,
			  XmNalignment,        XmALIGNMENT_END,
			  NULL);

  text_widget = XtVaCreateManagedWidget("fileName_text",
					xmTextFieldWidgetClass, form,
					XmNtraversalOn,      True,
					XmNleftAttachment,   XmATTACH_POSITION,
					XmNleftPosition,     4,
					XmNcolumns, numColumns,
					NULL);

  sprintf(name, "%sFBI", label);

  push_button = XtVaCreateManagedWidget(name,
					xmPushButtonGadgetClass, form,
					XmNtopAttachment,    XmATTACH_FORM,
					XmNbottomAttachment, XmATTACH_FORM,
					XmNleftAttachment,   XmATTACH_WIDGET,
					XmNleftWidget,       text_widget,
					NULL);
  
  XtAddCallback(push_button, XmNactivateCallback, cb, NULL);
  XtAddCallback(text_widget, XmNactivateCallback, fieldItem_CB, (XtPointer)cd);

  return form;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateDocumentParamsPage --
 *
 *      This function is invoked to create the page which presents
 *      the Document Parameters, in a notebook.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Create_DocumentParamsPage(Widget parent) {
  Widget rowcol, frame, rowColumn;
  

  rowColumn = XtVaCreateManagedWidget("rowColumn",
				      xmRowColumnWidgetClass,
				      parent,
				      XmNnotebookChildType, XmPAGE,
				      XmNpageNumber, 1,
				      XmNmarginWidth, 20,
				      XmNmarginHeight, 20,
				      NULL);
  
  frame = XtVaCreateManagedWidget("documentFrame",
				  xmFrameWidgetClass,
				  rowColumn,
				  NULL);
  
  XtVaCreateManagedWidget("documentLabel",
			  xmLabelGadgetClass,
			  frame,
			  XmNchildType, XmFRAME_TITLE_CHILD,
			  XmNchildVerticalAlignment, XmALIGNMENT_CENTER,
			  NULL);
  
  rowcol = XtVaCreateManagedWidget("documentRC",
				   xmRowColumnWidgetClass,
				   frame,
				   XmNpacking, XmPACK_COLUMN,
				   XmNnumColumns, 2,
				   NULL);
  
  GUI_CreateFieldItem(rowcol, INTERGER_FIELD, 6, "leftMargin", 0);
  GUI_CreateFieldItem(rowcol, INTERGER_FIELD, 6, "rightMargin", 1);
  GUI_CreateFieldItem(rowcol, INTERGER_FIELD, 6, "upMargin", 2);
  GUI_CreateFieldItem(rowcol, INTERGER_FIELD, 6, "downMargin", 3);
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateParagraphParamsPage --
 *
 *      This function is invoked to create the page which presents
 *      the Paragraph Parameters, in a notebook.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Create_ParagraphParamsPage(Widget parent) {
  Widget rowcol, frame, rowColumn;

  rowColumn = XtVaCreateManagedWidget("paragraphRC",
				      xmRowColumnWidgetClass,
				      parent,
				      XmNnotebookChildType, XmPAGE,
				      XmNpageNumber, 2,
				      XmNmarginWidth, 20,
				      XmNmarginHeight, 20,
				      NULL);
  
  frame = XtVaCreateManagedWidget("paragraphFrame",
				  xmFrameWidgetClass,
				  rowColumn,
				  XmNmarginWidth, 10,
				  XmNmarginHeight, 10,
				  NULL);
  
  XtVaCreateManagedWidget("paragraphLabel",
			  xmLabelGadgetClass,
			  frame,
			  XmNchildType, XmFRAME_TITLE_CHILD,
			  XmNchildVerticalAlignment, XmALIGNMENT_CENTER,
			  NULL);
  
  rowcol = XtVaCreateManagedWidget("paragraphRC",
			    xmRowColumnWidgetClass,
			    frame,
			    NULL);
  
  GUI_CreateFieldItem(rowcol, INTERGER_FIELD, 6, "leftSpace", 3);
  GUI_CreateFieldItem(rowcol, INTERGER_FIELD, 6, "rightSpace", 4);
  GUI_CreateFieldItem(rowcol, INTERGER_FIELD, 6, "flineSpace", 5);

  GUI_CreateOptionMenu(rowColumn, 4, "paraParams", paraParams_CB);
}



/*
 *----------------------------------------------------------------------
 *
 * Fill_ParamsList --
 *
 *      This function fills the Parameters list.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Fill_ParamsList(Widget list, char *path) {
  Widget *widTable;
  int num, i;
  XmString label;

  XtVaGetValues(XtNameToWidget(toplevel, path),
		XmNnumChildren, &num,
		XmNchildren, &widTable,
		NULL);

  for(i=0; i<num; i++) {
    XtVaGetValues(widTable[i],
		  XmNlabelString, &label,
		  NULL);
    XmListAddItem(list, label, 0);
  }
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateCharParamsPage --
 *
 *      This function is invoked to create the page which presents
 *      the Character Parameters, in a notebook.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
Create_CharParamsPage(Widget parent) {

  Widget mainRowColumn, rowcol, frame, list;
  XmString string;
  int i;
  char name[32];


  init_XFuncs(toplevel);

  mainRowColumn = XtVaCreateManagedWidget("rowColumn",
					  xmFormWidgetClass,
					  parent,
					  XmNnotebookChildType, XmPAGE,
					  XmNpageNumber, 3,
					  XmNmarginWidth, 20,
					  XmNmarginHeight, 20,
					  XmNfractionBase,  20,
					  XmNhorizontalSpacing, 10,
					  NULL);

  rowcol = XtVaCreateManagedWidget("charFamilyRC",
				   xmRowColumnWidgetClass,
				   mainRowColumn,
				   XmNtopAttachment, XmATTACH_POSITION,
				   XmNtopPosition, 1,
				   XmNleftAttachment, XmATTACH_FORM,
				   NULL);
  
  XtVaCreateManagedWidget("charFamilyLabel",
			  xmLabelGadgetClass,
			  rowcol,
			  NULL);
  
  list = XmCreateScrolledList(rowcol, "charFamilyList", NULL, 0);
  XtAddCallback(list, XmNbrowseSelectionCallback, (XtCallbackProc)fontSelect_CB, (void *)0);
  
  Fill_ParamsList(list, "*familyOM");
  XtManageChild( list );
  
  
  rowcol = XtVaCreateManagedWidget("charStyleRC",
				   xmRowColumnWidgetClass,
				   mainRowColumn,
				   XmNtopAttachment, XmATTACH_POSITION,
				   XmNtopPosition, 1,
				   XmNleftAttachment, XmATTACH_WIDGET,
				   XmNleftWidget, rowcol,
				   NULL);
    
  XtVaCreateManagedWidget("charStyleLabel",
			  xmLabelGadgetClass,
			  rowcol,
			  NULL);
    
  list = XmCreateScrolledList(rowcol, "charStyleList", NULL, 0);

  XtAddCallback(list, XmNbrowseSelectionCallback, (XtCallbackProc)fontSelect_CB, (void *)1);

  for(i=0; i<4; i++) {
    string = XmStringCreateSimple(fontStyle[i]);
    XmListAddItem(list, string, 0);
    XmStringFree( string );
  }
  XtManageChild( list );
  
  rowcol = XtVaCreateManagedWidget("charSizeRC",
				   xmRowColumnWidgetClass,
				   mainRowColumn,
				   XmNtopAttachment, XmATTACH_POSITION,
				   XmNtopPosition, 1,
				   XmNleftAttachment, XmATTACH_WIDGET,
				   XmNleftWidget, rowcol,
				   XmNrightAttachment, XmATTACH_FORM,
				   NULL);
  
  XtVaCreateManagedWidget("charSizeLabel",
			  xmLabelGadgetClass,
			  rowcol,
			  NULL);
  
  list = XmCreateScrolledList(rowcol, "charSizeList", NULL, 0);
  XtAddCallback(list, XmNbrowseSelectionCallback, (XtCallbackProc)fontSelect_CB, (void *)2);

  Fill_ParamsList(list, "*sizeOM");
  XtManageChild( list );    

  
  frame = XtVaCreateManagedWidget("charFrame",
				  xmFrameWidgetClass,
				  mainRowColumn,
				  XmNleftAttachment, XmATTACH_FORM,
				  XmNtopAttachment, XmATTACH_WIDGET,
				  XmNtopWidget, list,
				  XmNbottomAttachment, XmATTACH_FORM,
				  NULL);
  
  XtVaCreateManagedWidget("charLabel",
			  xmLabelGadgetClass,
			  frame,
			  XmNchildType, XmFRAME_TITLE_CHILD,
			  XmNchildVerticalAlignment, XmALIGNMENT_CENTER,
			  NULL);
  
  
  rowcol = XtVaCreateManagedWidget("charExempleRC",
				   xmRowColumnWidgetClass,
				   frame,
				   XmNradioBehavior, True,
				   NULL);
  
  
  for(i=0; i<4; i++) {
    sprintf(name, "charParams_b%d", i);    
    XtVaCreateManagedWidget(name,
			    xmToggleButtonGadgetClass,
			    rowcol,
			    NULL );
  }
  
  
  frame = XtVaCreateManagedWidget("fontFrame",
				  xmFrameWidgetClass,
				  mainRowColumn,
				  XmNleftAttachment, XmATTACH_WIDGET,
				  XmNleftWidget, frame,
				  XmNtopAttachment, XmATTACH_WIDGET,
				  XmNtopWidget, list,
				  XmNbottomAttachment, XmATTACH_FORM,
				  XmNrightAttachment, XmATTACH_FORM,
				  NULL);
  
  
  XtVaCreateManagedWidget("fontLabel",
			  xmLabelGadgetClass,
			  frame,
			  XmNchildType, XmFRAME_TITLE_CHILD,
			  XmNchildVerticalAlignment, XmALIGNMENT_CENTER,
			  NULL);
  
  
  XtVaCreateManagedWidget("fontDrawing",
			  xmDrawingAreaWidgetClass,
			  frame,
			  NULL);
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_RealizeDialog --
 *
 *      This function is invoked to realize a dialog. So when the
 *      dialog will be popup, then it will be at the center of
 *      the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_RealizeDialog(Widget widget) {

  Dimension w, h;
  Widget parent;


  parent = XtParent(widget);
  XtRealizeWidget(widget);

  XtVaGetValues(widget,
		XmNwidth, &w,
		XmNheight, &h,
		NULL);
  
  XtVaSetValues(parent,
		XmNminWidth, w,
		XmNminHeight, h,
		XmNmaxWidth,  w,
		XmNmaxHeight, h,
		XmNdeleteResponse, XmDO_NOTHING,
		NULL);

  XtAddCallback(parent, XmNpopupCallback, (XtCallbackProc)CenterDialog_CB, NULL);
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateMainDrawing --
 *
 *      This function is invoked to create the main drawing area. All
 *      the windows (pages of the document) are in this area.
 *      the screen.
 *
 * Results:
 *      Returns the drawing area.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateMainDrawing(Widget parent, Widget top_widget) {

  Pixel fore;
  Pixel back;
  Pixmap stipple;
  Widget scrollW, widget;
  
  XtVaGetValues(parent,
		XmNborderColor, &fore,
		XmNbackground,  &back,
		NULL);
  
  stipple = XmuCreateStippledPixmap(XtScreen(parent), fore, back,
				    DefaultDepth(XtDisplay(parent), 0));

  scrollW = XtVaCreateManagedWidget("mainDrawing",
				    xmScrolledWindowWidgetClass,
				    parent,

				    XmNscrollingPolicy, XmAUTOMATIC,
				    XmNscrollBarDisplayPolicy, XmAS_NEEDED,

				    XmNtopAttachment, XmATTACH_WIDGET,
				    XmNtopWidget, top_widget,
				    XmNbottomAttachment, XmATTACH_FORM,
				    XmNleftAttachment, XmATTACH_POSITION,
				    XmNleftPosition, 1,
				    XmNrightAttachment, XmATTACH_POSITION,
				    XmNrightPosition, 99,

				    NULL);
    
  XtVaGetValues(scrollW,
		XmNclipWindow, &widget,
		XmNhorizontalScrollBar, &mainDrawing_hsb,
		XmNverticalScrollBar, &mainDrawing_vsb,
		NULL);
  
  XtVaSetValues(widget,
		XmNbackgroundPixmap, stipple,
		NULL);
  return scrollW;
}
  


/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateNewPage --
 *
 *      This function is invoked to create a new page in the drawing
 *      area (main scrolled window). This function is called by the
 *      Object 'Page'.
 *
 * Results:
 *      Returns the created Page.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateNewPage(int n, Dimension width, Dimension height) {

  Widget widget;
  Position y;

  y = 10 + n * (20 + height);
  widget = XtVaCreateManagedWidget("PageWidget",
				   xmDrawingAreaWidgetClass,
				   mainDrawing,
				   XmNx, 10, /* (Dimension) ((int)w-500)/2, */
				   XmNy, y,
				   XmNwidth, width,
				   XmNheight, height,
				   XmNborderWidth, 2,
				   NULL );
  return widget;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateMainMessage --
 *
 *      This function is invoked to create the main message at the
 *      very bottom of the PAPyRUS GUI.
 *
 * Results:
 *      Returns the main message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateMainMessage(Widget parent) {

  XmString label;
  Widget widget;

  label = XmStringCreateSimple(" ");

  widget = XtVaCreateManagedWidget("mainLabel",
				   xmLabelGadgetClass,
				   parent,
				   XmNlabelString, label,
				   NULL);
  XmStringFree( label );

  return widget;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_Set_DocumentTitle --
 *
 *      Given a string, this function sets the main window title.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_Set_DocumentTitle(char *title) {

  if( title == NULL )
    XtVaSetValues(toplevel,
		  XmNtitle, appInfo.new_name,
		  NULL);
  else
    XtVaSetValues(toplevel,
		  XmNtitle, title,
		  NULL);
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_CreateNewDocument --
 *
 *      This function is invoked when a new document is created.
 *
 * Results:
 *      Returns mainDrawing widget.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Widget
GUI_CreateNewDocument() {

  GUI_Set_DocumentTitle(NULL);
  return mainDrawing;
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_App_OptionMenuEntry --
 *
 *      This function is invoked to add an entry in an option menu.
 *      The function is called by the AddEntry Tcl command.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_Add_OptionMenuEntry(char *postfix, char *label, char *cmd) {

  Widget optionMenu, widget;
  int num;
  char name[32], path[32], *cmd_name;

  sprintf(path, "*%sOM", postfix);

  optionMenu = XtNameToWidget(toplevel, path);
  XtVaGetValues(optionMenu,
		XmNnumChildren, &num,
		NULL);
  
  sprintf(name, "%s_b%d", path, num);
  widget = XtVaCreateManagedWidget(name,
				   xmPushButtonGadgetClass,
				   optionMenu,
				   XmNlabelString, XmStringCreateSimple(label),
				   NULL );
  if( num == 0 ) {
    Widget widget;
    strcat(path, "_rc");
    widget = XtNameToWidget(toplevel, path);
    XtManageChild(widget);
    XtVaSetValues(XmOptionButtonGadget(widget),
		  XmNlabelString, XmStringCreateSimple(label),
		  NULL);
  }

  cmd_name = strdup(cmd);
  XtAddCallback(widget, XmNactivateCallback, optionMenu_CB, (XtPointer)cmd_name);
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_Delete_OptionMenuEntry --
 *
 *      This function is invoked to delete an entry in an option menu.
 *      The function is called by the DeleteStyle Tcl command.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_Delete_OptionMenuEntry(char *postfix, char *name) {

  Widget optionMenu;
  WidgetList wlist;
  XmString label;
  int i, num;
  char path[32], *string;

  sprintf(path, "*%sOM", postfix);
  optionMenu = XtNameToWidget(toplevel, path);
  
  XtVaGetValues(optionMenu,
		XmNchildren, &wlist,
		XmNnumChildren, &num,
		NULL);

  for(i=0; i<num; i++) {
    XtVaGetValues(wlist[i], XmNlabelString, &label, NULL);
    XmStringGetLtoR(label, XmSTRING_DEFAULT_CHARSET, &string);

    if( (name[0] == string[0]) && (strcmp(name, string) == 0) ) {
      XtDestroyWidget(wlist[i]);
      break;
    }
  }
}



/*
 *----------------------------------------------------------------------
 *
 * Init_XFuncs --
 *
 *      This function initalialize some datas at the Xlib level.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
init_XFuncs(Widget widget) {

  Display *disp;
  XGCValues gcv;
  int s;

  disp = XtDisplay(widget);
  s = DefaultScreen(disp);
  

  gcv.function = GXcopy;
  gcv.foreground = BlackPixel(disp, s);

  drawA_gc = XCreateGC(disp, XtWindow(widget), GCFunction|GCForeground, &gcv);
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_SetCursor --
 *
 *      This function sets the cursor to the watch (processing) icon.
 *      The function is called by the SetCursor Tcl command.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_SetCursor(Widget w) {
  static Cursor watch = 0;
  
  if( !watch )
    watch = XCreateFontCursor(XtDisplay(w), XC_watch);
  
  XDefineCursor(XtDisplay(w), XtWindow(w), watch);
  XmUpdateDisplay(w);
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_ResetCursor --
 *
 *      This function sets the cursor to default cursor.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_ResetCursor(Widget w) {

  XUndefineCursor(XtDisplay(w), XtWindow (w));
  XmUpdateDisplay(w);
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_MakePageVisible --
 *
 *      This function is invoked to be sure that the cursor is
 *      visible. If it's not, the pages are moved on the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_MakePageVisible(Display *disp, Window win, int x, int y) {

  int value, slider, incr, page;
  int newValue;
  int max;
  Position src_y;

  XmScrollBarGetValues(mainDrawing_hsb, &value, &slider, &incr, &page);

  x += 10;

  if( x < value || x > slider + value ) {
    newValue = (x / slider) * slider;
    if( x > (slider + value) ) {
      newValue += value;
      
      XtVaGetValues(mainDrawing_hsb, XmNmaximum, &max, NULL);
      if( newValue > (max-slider) ) {
	/*
	fprintf(stderr, "max = %d \t newValue = %d \t slider = %d \t value = %d\n",
		max, newValue, slider, value);
		*/
	newValue = max-slider;
      }
    }
    XmScrollBarSetValues(mainDrawing_hsb, newValue, slider, incr, page, True);
  }

  XmScrollBarGetValues(mainDrawing_vsb, &value, &slider, &incr, &page);
  XtVaGetValues(XtWindowToWidget(disp, win), XmNy, &src_y, NULL);

  y += src_y;

  if( y < 0 || y > slider ) {
    if( y < 0 )
      newValue = value - slider;
    else
      newValue = value + slider;
    XmScrollBarSetValues(mainDrawing_vsb, newValue, slider, incr, page, True);
  }
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_Set_QuestionMsg --
 *
 *      This function is invoked to display a question dialog. The
 *      two buttons (ok/cancel) have callbacks.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_Set_QuestionMsg(char *string, XtCallbackProc cb) {

  XmString xmstr;

  xmstr = XmStringCreateLtoR(string, XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(question_dialog, XmNmessageString, xmstr, NULL);

  XtRemoveAllCallbacks(question_dialog, XmNokCallback);
  XtRemoveAllCallbacks(question_dialog, XmNcancelCallback);

  XtAddCallback(question_dialog, XmNokCallback, cb, (XtPointer)True);
  XtAddCallback(question_dialog, XmNcancelCallback, cb, (XtPointer)False);

  XtManageChild(question_dialog);
  XmStringFree( xmstr );
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_Add_ErrorMessage --
 *
 *      This function is invoked to had an error message in the
 *      error history dialog.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_Add_ErrorMessage(char *message) {

  extern Widget error_dialog;

  XmTextInsert(error_dialog, XmTextGetLastPosition(error_dialog), message);
  XmTextInsert(error_dialog, XmTextGetLastPosition(error_dialog), "\n");
}



/*
 *----------------------------------------------------------------------
 *
 * GUI_SetOMEntry --
 *
 *      This function is invoked to set the visible entry of an
 *      option menu.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
GUI_SetOMEntry(char *postfix, char *string) {

  int num, i;
  WidgetList wlist;
  Widget choice;
  XmString label;
  char *name, nameOM[32], nameRC[32];

  sprintf(nameOM, "*%sOM", postfix);
  XtVaGetValues(XtNameToWidget(toplevel, nameOM),
		XmNnumChildren, &num,
		XmNchildren, &wlist,
		NULL);

  for(i=0; i<num; i++) {
    XtVaGetValues(wlist[i], XmNlabelString, &label, NULL);
    XmStringGetLtoR(label, XmSTRING_DEFAULT_CHARSET, &name);
    if( (name[0] == string[0]) && (strcmp(name, string) == 0) ) {
      sprintf(nameRC, "%s_rc", nameOM);
      choice = XtNameToWidget(toplevel, nameRC);
      XtVaSetValues(choice,
		    XmNmenuHistory, wlist[i],
		    NULL);
      XmUpdateDisplay(choice);
      break;
    }
  }
}
