/* 
 * callbacks.cc --
 *
 *      This file contains fonctions all the callbacks of the
 *      PAPyRUS GUI.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <Xm/FileSB.h>
#include <Xm/PushBP.h>
#include <Xm/PushBG.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/ToggleBG.h>
#include <Xm/List.h>

#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/LabelG.h>

#include "callbacks.h"
#include "globals.h"
#include "widgets.h"
#include "dialogs.h"

#include "../tcl/mytcl.h"



extern BOOL Allow_Balloons;
extern BOOL interactive_flag;
extern char *fontFamily[], *fontSize[];
extern GC drawA_gc;
extern Dimension appWidth, appHeight;


static char string[] = "This is PaPyRuS";
static int family=0, size=0;
static char weight[]="medium", slant='r';


/*
 *----------------------------------------------------------------------
 *
 * command_CB --
 *
 *      This callback is invoked when <Enter> key in entered in the
 *      field of the langage interpret.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
command_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  Tcl_MyEval( XmTextGetString(widget) );
}



/*
 *----------------------------------------------------------------------
 *
 * saveFormat_CB --
 *
 *      This callback is invoked when the output format is changed by
 *      using the option menu of the 'print' dialog.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
saveFormat_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  int num = (int)clientData;

  Tcl_SetVar(theInterp, "outputFormat",
	     num == 0 ? "papyrus" :
	     num == 1 ? "ascii" :
	     "postscript", TCL_GLOBAL_ONLY);
}



/*
 *----------------------------------------------------------------------
 *
 * optionMenu_CB --
 *
 *      This callback is invoked when an entry of an option menu is
 *      selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
optionMenu_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  int num;

  if( Tcl_GetInt(theInterp, (char *)clientData, &num) != TCL_OK )
    Tcl_MyEval((char *)clientData);
/*
  else {
    fprintf(stderr, "OptionMenu: %d, ", num/100);
    fprintf(stderr, "PushButton; %d\n", num%100);
  }
*/
}



/*
 *----------------------------------------------------------------------
 *
 * iconBar_CB --
 *
 *      This callback is invoked when an icon in the iconbar is pushed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
iconBar_CB(Widget w, XtPointer clientData, XtPointer callData) {

  XtCallActionProc(w, "balloon-off", NULL, NULL, 0);

  switch( (int)clientData ) {

  case 0:
    menu_p0_b0_CB(w, clientData, callData);
    break;
    
  case 1:
    menu_p0_b1_CB(w, clientData, callData);
    break;

  case 2:
    menu_p0_b4_CB(w, clientData, callData);
    break;

  case 3:
    menu_p0_b5_CB(w, clientData, callData);
    break;

  case 4:
    menu_p3_b0_CB(w, clientData, callData);
    break;

  case 10:
    Tcl_MyEval("ParaConfigure -style Liste1");
    break;

  case 11:
    Tcl_MyEval("Incr_Left_Margin -10");
    break;

  case 12:
    Tcl_MyEval("Incr_Left_Margin +10");
    break;

  default:
/*    fprintf(stderr, "PushButton: %d\n", (int)clientData); */
    break;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p0_b0_CB --
 *
 *      This callback is invoked when the entry number 0, in the pane
 *      number 0 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p0_b0_CB(Widget w, XtPointer clientData, XtPointer callData) {

  Tcl_MyEval("NewDocument");
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p0_b1_CB --
 *
 *      This callback is invoked when the entry number 1, in the pane
 *      number 0 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p0_b1_CB(Widget w, XtPointer clientData, XtPointer callData) {
  
  XtManageChild( GUI_CreateFileSelection_Dialog("open", open_CB) );
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p0_b2_CB --
 *
 *      This callback is invoked when the entry number 2, in the pane
 *      number 0 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p0_b2_CB(Widget w, XtPointer clientData, XtPointer callData) {
  
  Tcl_MyEval("CloseDocument");
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p0_b4_CB --
 *
 *      This callback is invoked when the entry number 4, in the pane
 *      number 0 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p0_b4_CB(Widget w, XtPointer clientData, XtPointer callData) {

  Widget fsb, frame;
  
  fsb = GUI_CreateFileSelection_Dialog("save", save_CB);

  if( fsb == NULL )
    return;

  frame = XtVaCreateManagedWidget( "Frame",
				  xmFrameWidgetClass,
				  fsb,
				  NULL);
  GUI_CreateOptionMenu(frame, 3, "saveFormat", saveFormat_CB);
  XtManageChild(fsb);
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p0_b5_CB --
 *
 *      This callback is invoked when the entry number 5, in the pane
 *      number 0 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p0_b5_CB(Widget w, XtPointer clientData, XtPointer callData) {

  GUI_SetCursor(toplevel);

  if( print_dialog == NULL ) {
    print_dialog = GUI_CreatePrint_Dialog(toplevel);
    XtManageChild(print_dialog);
  }

  GUI_ResetCursor(toplevel);
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p0_b6_CB --
 *
 *      This callback is invoked when the entry number 6, in the pane
 *      number 0 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p0_b6_CB(Widget w, XtPointer clientData, XtPointer callData) {

  GUI_Set_QuestionMsg(appInfo.quit_msg, quit_CB);
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p2_b0_CB --
 *
 *      This callback is invoked when the entry number 0, in the pane
 *      number 2 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p2_b0_CB(Widget w, XtPointer clientData, XtPointer callData) {

  GUI_SetCursor(toplevel);

  if( params_dialog == NULL ) {
    params_dialog = GUI_CreateParams_Dialog(toplevel);
    XtManageChild(params_dialog);
  }

  XtVaSetValues(XtNameToWidget(toplevel, "*params_popup*notebook"),
		XmNcurrentPageNumber, (int)clientData,
		NULL);

  GUI_ResetCursor(toplevel);
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p3_b0_CB --
 *
 *      This callback is invoked when the entry number 0, in the pane
 *      number 3 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p3_b0_CB(Widget w, XtPointer clientData, XtPointer callData) {

  Widget widget;

  widget = GUI_CreateIspell_Dialog(toplevel);
  XtSetSensitive(XtNameToWidget(widget, "*button1"), False);
  XtSetSensitive(XtNameToWidget(widget, "*button2"), False);
  XtSetSensitive(XtNameToWidget(widget, "*button3"), True);
  XtManageChild(widget);
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p3_b1_CB --
 *
 *      This callback is invoked when the entry number 1, in the pane
 *      number 3 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p3_b1_CB(Widget w, XtPointer clientData, XtPointer callData) {

  XtManageChild( GUI_CreateSpecialChars_Dialog(toplevel) );
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p3_b3_CB --
 *
 *      This callback is invoked when the entry number 3, in the pane
 *      number 3 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p3_b3_CB(Widget w, XtPointer clientData, XtPointer callData) {

  XtManageChild( GUI_CreateDocumentInfo_Dialog(toplevel) );
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p4_b0_CB --
 *
 *      This callback is invoked when the entry number 0, in the pane
 *      number 4 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p4_b0_CB(Widget w, XtPointer clientData, XtPointer callData) {
  
  if( XtIsManaged(mainToolBar) == True ) {
    XtUnmanageChild(mainToolBar);

    if( XtIsManaged(mainFormatBar) == True ) {

      XtVaSetValues(mainFormatBar,
		    XmNtopAttachment, XmATTACH_FORM,
		    NULL);
      XtVaSetValues(mainDrawing,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, mainFormatBar,
		    NULL);
    } else
      XtVaSetValues(mainDrawing,
		    XmNtopAttachment, XmATTACH_POSITION,
		    XmNtopPosition, 1,
		    NULL);
  } else {
    XtManageChild(mainToolBar);

    if( XtIsManaged(mainFormatBar) == True ) {
      XtVaSetValues(mainFormatBar,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, mainToolBar,
		    NULL);
    } else
      XtVaSetValues(mainDrawing,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, mainToolBar,
		    NULL);
  }
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p4_b1_CB --
 *
 *      This callback is invoked when the entry number 1, in the pane
 *      number 4 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p4_b1_CB(Widget w, XtPointer clientData, XtPointer callData) {
  
  if( XtIsManaged(mainFormatBar) == True ) {
    XtUnmanageChild(mainFormatBar);
    
    if( XtIsManaged(mainToolBar) == True ) {

      XtVaSetValues(mainDrawing,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, mainToolBar,
		    NULL);
    } else
      XtVaSetValues(mainDrawing,
		    XmNtopAttachment, XmATTACH_POSITION,
		    XmNtopPosition, 1,
		    NULL);
  } else {
    XtManageChild(mainFormatBar);

    if( XtIsManaged(mainToolBar) == True ) {
      XtVaSetValues(mainFormatBar,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, mainToolBar,
		    NULL);

      XtVaSetValues(mainDrawing,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, mainFormatBar,
		    NULL);
    } else {
      XtVaSetValues(mainFormatBar,
		    XmNtopAttachment, XmATTACH_FORM,
		    NULL);
      XtVaSetValues(mainDrawing,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, mainFormatBar,
		    NULL);
    }
  }
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p4_b3_CB --
 *
 *      This callback is invoked when the entry number 3, in the pane
 *      number 4 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p4_b3_CB(Widget w, XtPointer clientData, XtPointer callData) {
  
  if( XtIsManaged(mainLabel) == True ) {
    XtUnmanageChild(mainLabel);
    if( XtIsManaged(mainCommand) == False )
      XtVaSetValues(mainDrawing,
		    XmNbottomAttachment, XmATTACH_POSITION,
		    XmNbottomPosition, 99,
		    NULL);
  } else {
    XtManageChild(mainLabel);
    if( XtIsManaged(mainCommand) == False )
      XtVaSetValues(mainDrawing,
		    XmNbottomAttachment, XmATTACH_FORM,
		    NULL);
    else
      XtVaSetValues(mainDrawing,
		    XmNbottomAttachment, XmATTACH_POSITION,
		    XmNbottomPosition, 99,
		    NULL);      
  }
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p4_b4_CB --
 *
 *      This callback is invoked when the entry number 4, in the pane
 *      number 4 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p4_b4_CB(Widget w, XtPointer clientData, XtPointer callData) {
  
  if( XtIsManaged(mainCommand) == True ) {
    XtUnmanageChild(mainCommand);
    XtCallActionProc(XtNameToWidget(toplevel, "*button_b9"),
		     "Disarm", NULL, NULL, 0);
    if( XtIsManaged(mainLabel) == False )
      XtVaSetValues(mainDrawing,
		    XmNbottomAttachment, XmATTACH_POSITION,
		    XmNbottomPosition, 99,
		    NULL);
    else
      XtVaSetValues(mainDrawing,
		    XmNbottomAttachment, XmATTACH_FORM,
		    NULL);

    /*
     * Because this function can be invoked by pressing
     * a button in the iconbar, we have to insure that
     * the menu item has a right value.
     */
    if( XmToggleButtonGadgetGetState( w ) == True )
      XmToggleButtonGadgetSetState( w, False, False );
  } else {
    XtCallActionProc(XtNameToWidget(toplevel, "*button_b9"),
				    "Arm", NULL, NULL, 0);
    XtManageChild(mainCommand);
    XtVaSetValues(mainDrawing,
		  XmNbottomAttachment, XmATTACH_POSITION,
		  XmNbottomPosition, 99,
		  NULL);

    /*
     * Because this function can be invoked by pressing
     * a button in the iconbar, we have to insure that
     * the menu item has a right value.
     */
    if( XmToggleButtonGadgetGetState( w ) == False )
      XmToggleButtonGadgetSetState( w, True, False );
  }
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p4_b5_CB --
 *
 *      This callback is invoked when the entry number 5, in the pane
 *      number 4 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p4_b5_CB(Widget w, XtPointer clientData, XtPointer callData) {

  Allow_Balloons = (BOOL)(Allow_Balloons ^ TRUE);
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p4_b6_CB --
 *
 *      This callback is invoked when the entry number 6, in the pane
 *      number 4 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p4_b6_CB(Widget w, XtPointer clientData, XtPointer callData) {

  Widget dialog;

  dialog = XtNameToWidget(toplevel, "*error");

  if( XtIsManaged(dialog) == True )
    XtUnmanageChild(dialog);
  else
    XtManageChild(dialog);
}



/*
 *----------------------------------------------------------------------
 *
 * menu_p5_b3_CB --
 *
 *      This callback is invoked when the entry number 3, in the pane
 *      number 5 is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
menu_p5_b3_CB(Widget w, XtPointer clientData, XtPointer callData) {

  Widget dialog;

  dialog = XtNameToWidget(toplevel, "*infos");
  
  if( dialog == NULL ) {
    dialog = GUI_CreateInfos_Dialog(toplevel);
    XtManageChild(dialog);
  }
}



/*
 *----------------------------------------------------------------------
 *
 * open_CB --
 *
 *      This callback is invoked for openning a document (icon & menu).
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
open_CB(Widget widget, XtPointer clientData, XtPointer callData) {
  XmFileSelectionBoxCallbackStruct* fs = (XmFileSelectionBoxCallbackStruct*)callData;
  char *filename;
  char string[128];

  XtUnmanageChild(widget);
  
  if( (int)clientData == TRUE ) {
    XmStringGetLtoR(fs->dir, "", &filename);
    chdir(filename);

    XmStringGetLtoR(fs->value, XmSTRING_DEFAULT_CHARSET, &filename);

    GUI_SetCursor(toplevel);

    Tcl_MyEval("NewDocument");

    interactive_flag = FALSE;
    sprintf(string, "source %s", filename);
    Tcl_MyEval(string);
    interactive_flag = TRUE;

    XtFree(filename);

    GUI_ResetCursor(toplevel);
  }  
  XtDestroyWidget(widget);
}



/*
 *----------------------------------------------------------------------
 *
 * open_CB --
 *
 *      This callback is invoked for saving a document (icon & menu).
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
save_CB(Widget widget, XtPointer clientData, XtPointer callData) {
  XmFileSelectionBoxCallbackStruct *fs = (XmFileSelectionBoxCallbackStruct*)callData;
  char *filename;
  char string[128];

  if( (int)clientData == True ) {
    XmStringGetLtoR(fs->dir, "", &filename);
    chdir(filename);

    XmStringGetLtoR(fs->value, XmSTRING_DEFAULT_CHARSET, &filename);
    sprintf(string, "Save_As %s", filename);
    Tcl_MyEval(string);
  }
  XtDestroyWidget(widget);
}



/*
 *----------------------------------------------------------------------
 *
 * search_CB --
 *
 *      This callback is invoked when a FileSeclection box is used to
 *      find a file.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
search_CB(Widget widget, XtPointer clientData, XtPointer callData) {
  char *filename;
  XmFileSelectionBoxCallbackStruct* fs = (XmFileSelectionBoxCallbackStruct*)callData;

  if( (int)clientData == TRUE ) {
    XmStringGetLtoR(fs->dir, "", &filename);
    chdir(filename);

    XmStringGetLtoR(fs->value, XmSTRING_DEFAULT_CHARSET, &filename);
    Tcl_SetVar2(theInterp, "print", "file", filename, TCL_GLOBAL_ONLY);

    XmTextFieldSetString(XtNameToWidget(print_dialog, "*file_form.fileName_text"),
			 filename);
  }
  XtDestroyWidget(widget);
  
}



/*
 *----------------------------------------------------------------------
 *
 * Info_CB --
 *
 *      This callback is invoked when the ok button of the 'Info'
 *      dialog is actived.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
infos_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  XtDestroyWidget( widget );
}



/*
 *----------------------------------------------------------------------
 *
 * quit_CB --
 *
 *      This callback is invoked for the two buttons of the 'Quit'
 *      dialog. If clientData == True -> Ok, else Cancel.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
quit_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  if( (int)clientData == True )
    exit(0);
  else
    XtUnmanageChild(widget);
}



/*
 *----------------------------------------------------------------------
 *
 * params_CB --
 *
 *      This callback is invoked to map or unmap the Parameters dialog.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
params_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  char cmd[64];
  int margins[4], fline;
  char *string, *string2, *align;
  XmString xmstring;
  Widget w;
  char style[32];


  if( (int)clientData == True ) {

    /*
     * Document params
     */

    Tcl_MyEval("Get_DocParams");

    string  = Tcl_GetVar2(theInterp, "common", "top", TCL_GLOBAL_ONLY);
    string2 = XmTextFieldGetString(XtNameToWidget(widget, "*upMargin_form.field_text"));
    
    if( atoi(string) != atoi(string2) )
      margins[0] = atoi(string2);
    else
      margins[0] = atoi(string);

    string  = Tcl_GetVar2(theInterp, "common", "bottom", TCL_GLOBAL_ONLY);
    string2 = XmTextFieldGetString(XtNameToWidget(widget, "*downMargin_form.field_text"));
    
    if( atoi(string) != atoi(string2) )
      margins[1] = atoi(string2);
    else
      margins[1] = atoi(string);

    string  = Tcl_GetVar2(theInterp, "common", "left", TCL_GLOBAL_ONLY);
    string2 = XmTextFieldGetString(XtNameToWidget(widget, "*leftMargin_form.field_text"));
    
    if( atoi(string) != atoi(string2) )
      margins[2] = atoi(string2);
    else
      margins[2] = atoi(string);

    string  = Tcl_GetVar2(theInterp, "common", "right", TCL_GLOBAL_ONLY);
    string2 = XmTextFieldGetString(XtNameToWidget(widget, "*rightMargin_form.field_text"));
    
    if( atoi(string) != atoi(string2) )
      margins[3] = atoi(string2);
    else
      margins[3] = atoi(string);

    sprintf(cmd, "DocConfigure -margins {%d %d %d %d}", 
	    margins[0], margins[1], margins[2], margins[3]);
    Tcl_MyEval(cmd);
    Tcl_MyEval("RefreshScreen on");

    /*
     * Paragraph params
     */

    Tcl_MyEval("Get_ParaParams");
    
    if( atoi(string) != atoi(string2) )
      margins[1] = atoi(string2);
    else
      margins[1] = atoi(string);

    string  = Tcl_GetVar2(theInterp, "common", "top", TCL_GLOBAL_ONLY);
    margins[0] = atoi(string);
    string  = Tcl_GetVar2(theInterp, "common", "bottom", TCL_GLOBAL_ONLY);
    margins[1] = atoi(string);

    string  = Tcl_GetVar2(theInterp, "common", "left", TCL_GLOBAL_ONLY);
    string2 = XmTextFieldGetString(XtNameToWidget(widget, "*leftSpace_form.field_text"));
    
    if( atoi(string) != atoi(string2) )
      margins[2] = atoi(string2);
    else
      margins[2] = atoi(string);

    string  = Tcl_GetVar2(theInterp, "common", "right", TCL_GLOBAL_ONLY);
    string2 = XmTextFieldGetString(XtNameToWidget(widget, "*rightSpace_form.field_text"));
    
    if( atoi(string) != atoi(string2) )
      margins[3] = atoi(string2);
    else
      margins[3] = atoi(string);

    string  = Tcl_GetVar2(theInterp, "common", "fline", TCL_GLOBAL_ONLY);
    string2 = XmTextFieldGetString(XtNameToWidget(widget, "*flineSpace_form.field_text"));
    
    if( atoi(string) != atoi(string2) )
      fline = atoi(string2);
    else
      fline = atoi(string);

    string  = Tcl_GetVar2(theInterp, "common", "align", TCL_GLOBAL_ONLY);

    XtVaGetValues(XtNameToWidget(widget, "*paraParamsOM_rc"),
		  XmNmenuHistory, &w,
		  NULL);
    XtVaGetValues(w, XmNlabelString, &xmstring, NULL);
    XmStringGetLtoR(xmstring, XmSTRING_DEFAULT_CHARSET, &align);

    switch( align[0] ) {
    case 'G':
      strcpy(align, "left");
	break;
    case 'D':
      strcpy(align, "right");
      break;
    case 'C':
      strcpy(align, "center");
      break;
    case 'J':
      strcpy(align, "full");
      break;
    }
    if( strcmp(string, align) == 0 )
      align = string;

    sprintf(cmd, "ParaConfigure -margins {%d %d %d %d} -fline %d -align %s",
	    margins[0], margins[1], margins[2], margins[3], fline, align);
    Tcl_MyEval(cmd);
/*    Tcl_MyEval("RefreshScreen on"); */

    /*
     * Paragraph params
     */
    switch( weight[0] ) {
    case 'm':
      if( slant == 'r' )
	strcpy(style, "normal");
      else
	strcpy(style, "italic");
      break;

    case 'b':
      if( slant == 'r' )
	strcpy(style, "bold");
      else
	strcpy(style, "bold_italic");
      break;
    }
    sprintf(cmd, "WordConfigure -size %s -style %s -family %s",
	    fontSize[size], style, fontFamily[family]);
    Tcl_MyEval(cmd);
  }

  XtDestroyWidget(widget);
  params_dialog = NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * print_CB --
 *
 *      This callback is invoked for the two buttons of the 'Print'
 *      dialog.
 *      using the option menu of the 'print' dialog.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
print_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  char *string, *format, *device;
  char cmd[64];

  if( (int)clientData == True ) {

    format = Tcl_GetVar(theInterp, "outputFormat", TCL_GLOBAL_ONLY);
    Tcl_SetVar(theInterp, "outputFormat", "postscript", TCL_GLOBAL_ONLY); 
    device = Tcl_GetVar2(theInterp, "print", "device", TCL_GLOBAL_ONLY);

    if( strcmp(device, "file") == 0 ) {  /* Print in a file */

      string = XmTextFieldGetString(XtNameToWidget(widget, "*file_form.fileName_text"));
      if( string != NULL )
	Tcl_SetVar2(theInterp, "print", "file", string, TCL_GLOBAL_ONLY);
      else
	string = Tcl_GetVar2(theInterp, "print", "file", TCL_GLOBAL_ONLY);
      
      sprintf(cmd, "Save_As %s", string);
      
    } else {                             /* Print to the printer */

      string = XmTextFieldGetString(XtNameToWidget(widget, "*cmd_form.field_text"));

      if( string != NULL )
	Tcl_SetVar2(theInterp, "print", "command", string, TCL_GLOBAL_ONLY);
      else
	string = Tcl_GetVar2(theInterp, "print", "command", TCL_GLOBAL_ONLY);

      sprintf(cmd, "Save_As \"|%s\"", string);
    }
    Tcl_MyEval(cmd);
    Tcl_SetVar(theInterp, "outputFormat", format, TCL_GLOBAL_ONLY);
  }
  XtDestroyWidget(widget);

  print_dialog = NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * paraParams_CB --
 *
 *      This callback is invoked for the Paragraph Parameters dialog.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
paraParams_CB(Widget widget, XtPointer clientData, XtPointer callData) {
}



/*
 *----------------------------------------------------------------------
 *
 * CenterDialog_CB --
 *
 *      This callback is invoked when any dialog is popup. The
 *      function centers the dialog on the screen.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
CenterDialog_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  Position x_offset, y_offset;
  Dimension dialogWidth, dialogHeight;


  XtVaGetValues(widget,                      /* Find dialog shell size */
		XmNwidth, &dialogWidth,
		XmNheight, &dialogHeight, 
		NULL);
  
  x_offset = (appWidth  - dialogWidth)  / 2;  /* Calculate dialog offset */
  y_offset = (appHeight - dialogHeight) / 2;  /* from appl shell origin */
  
  XtVaSetValues(widget,                      /* Set dialog shell X & Y */
		XmNx, x_offset,
		XmNy, y_offset, 
		NULL);
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * printOrient_CB --
 *
 *      This callback is invoked when the paper orientation toggle
 *      of the 'Print' dialog are modified.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The Tcl 'print(orient)' variable is updated.
 *
 *----------------------------------------------------------------------
 */

void
printOrient_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  Widget toggle, label;
  extern Pixmap paperOrient_xpm[2];

  toggle = XtNameToWidget(toplevel, "*print*paperOrient_b0");
  label  = XtNameToWidget(toplevel, "*print*paperBitmap");

  if( XmToggleButtonGadgetGetState(widget) == True )
    return;

  if( widget == toggle )
    Tcl_SetVar2(theInterp, "print", "orient", "portrait", TCL_GLOBAL_ONLY);
  else
    Tcl_SetVar2(theInterp, "print", "orient", "landscape", TCL_GLOBAL_ONLY);

  XtVaSetValues(label, XmNlabelPixmap,
		(Pixmap)paperOrient_xpm[(int)clientData], NULL);
}



/*
 *----------------------------------------------------------------------
 *
 * printFormat_CB --
 *
 *      This callback is invoked when the paper format toggle of
 *      the 'Print' dialog are modified.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The Tcl 'print(format)' variable is updated.
 *
 *----------------------------------------------------------------------
 */

void
printFormat_CB(Widget widget, XtPointer clientData, XtPointer callData) {
  
  switch( (int)clientData ) {
  case 0:
    Tcl_SetVar2(theInterp, "print", "format", "letter", TCL_GLOBAL_ONLY);
    break;
  case 1:
    Tcl_SetVar2(theInterp, "print", "format", "legal", TCL_GLOBAL_ONLY);
    break;
  case 2:
    Tcl_SetVar2(theInterp, "print", "format", "executive", TCL_GLOBAL_ONLY);
    break;
  case 3:
    Tcl_SetVar2(theInterp, "print", "format", "a4", TCL_GLOBAL_ONLY);
    break;
  }
}



/*
 *----------------------------------------------------------------------
 *
 * saveFormat_CB --
 *
 *      This callback is invoked when the paper device text item of
 *      the 'Print' dialog are modified.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The Tcl 'print(device)' variable is updated.
 *
 *----------------------------------------------------------------------
 */

void
printDevice_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  if( (int)clientData == 0 ) {       /* Printer */
    Tcl_SetVar2(theInterp, "print", "device", "printer", TCL_GLOBAL_ONLY);
    XtSetSensitive(XtNameToWidget(toplevel, "*file_form"), False);
    XtSetSensitive(XtNameToWidget(toplevel, "*prn_form"), True);
    XtSetSensitive(XtNameToWidget(toplevel, "*cmd_form"), True);
  } else {                           /* Print in a file */
    Tcl_SetVar2(theInterp, "print", "device", "file", TCL_GLOBAL_ONLY);
    XtSetSensitive(XtNameToWidget(toplevel, "*file_form"), True);
    XtSetSensitive(XtNameToWidget(toplevel, "*prn_form"), False);
    XtSetSensitive(XtNameToWidget(toplevel, "*cmd_form"), False);
    XmProcessTraversal(XtNameToWidget(toplevel,
				      "*file_form.fileName_text"), XmTRAVERSE_CURRENT);
  }
}



/*
 *----------------------------------------------------------------------
 *
 * saveFormat_CB --
 *
 *      This callback is invoked to manage the FileSectionBox
 *      of the 'Print' dialog.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
filePrintSearch_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  XtManageChild( GUI_CreateFileSelection_Dialog("file", search_CB) );
}



/*
 *----------------------------------------------------------------------
 *
 * saveFormat_CB --
 *
 *      This callback is invoked when an item is selected in on of
 *      the tree lists of the 'Character Parameters' dialog.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
fontSelect_CB(Widget widget, XtPointer clientData, XtPointer callData) {

  int pos, *item, tw;
  static XFontStruct *xfs = NULL;
  char fontName[128];
  Widget drawA;
  Dimension height, width;


  GUI_SetCursor(widget);

  XmListGetSelectedPos(widget, &item, &pos);

  switch( (int)clientData ) {

  case 0:    /* Family change (Times, Courier, ...) */
    family = *item-1;
    if( slant != 'r' )
      if( family == 0 || family == 1 ) slant = 'o';
      else slant = 'i';
    break;
    
  case 1:    /* Weight and/or Slant change (bold, italic, ...) */
    switch( *item-1 ) {

    case 0:  /* normal */
      strcpy(weight, "medium");
      slant = 'r';
      break;

    case 1:  /* Italic */
      strcpy(weight, "medium");
      if( family == 0 || family == 1 ) /* Because the 'courier' and 'helvetica families   */
	slant = 'o';                   /*  haven't the 'italic' slant so we use 'oblique' */
      else
	slant = 'i';
      break;

    case 3:  /* Bold-Italic */
      strcpy(weight, "bold");
      if( family == 0 || family == 1 )    /* see below */
	slant = 'o';
      else
	slant = 'i';
      break;

    case 2:  /* Bold */
      strcpy(weight, "bold");
      slant = 'r';
      break;
    }
    break;
    
  case 2:    /* Size change (2, 4, ...) */
    size = *item-1;
    break;
  }

  if( xfs != NULL )
    XFreeFont(XtDisplay(widget), xfs);

  sprintf(fontName, "-adobe-%s-%s-%c-normal--%s-*-*-*-*-*-iso8859-1",
	  (char *)fontFamily[family], weight, slant, fontSize[size]);

  xfs = XLoadQueryFont(XtDisplay(widget), fontName);
  tw = XTextWidth(xfs, string, strlen(string));

  drawA = XtNameToWidget(toplevel, "*fontFrame.fontDrawing");
  XtVaGetValues(drawA,
		XmNwidth, &width,
		XmNheight, &height,
		NULL);

  XClearArea(XtDisplay(drawA), XtWindow(drawA), 0, 0, width, height, 0);

  XSetFont(XtDisplay(widget), drawA_gc, xfs->fid);
  XDrawString(XtDisplay(drawA), XtWindow(drawA), drawA_gc,
	      (width-tw)/2, height/2,
	      string, strlen(string));

  GUI_ResetCursor(widget);
}



/*
 *----------------------------------------------------------------------
 *
 * specialChars_CB --
 *
 *      This callback is invoked when a character is selected in
 *      the 'Special Characters' dialog.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
specialChars_select_CB(Widget w, XtPointer client_data, XtPointer rs) {

  Position x, y, root_x, root_y;
  Dimension width, height;
  XmString label;
  
  XtPopdown(zoomCharWidget);

  XtVaGetValues(w,
                XmNx, &x,
                XmNy, &y,
                XmNwidth, &width,
                XmNheight, &height,
		XmNlabelString, &label,
                NULL );

  /*
   * Convert these coordinates to parent window coordinates
   */

  XtTranslateCoords(XtNameToWidget(toplevel, "sChars_popup*toggleBox"),
		    x, y,
		    &root_x, &root_y);

  XtVaSetValues(zoomCharWidget,
                XmNx,      root_x - width/2,
                XmNy,      root_y - height/2,
                XmNwidth,  width*2,
                XmNheight, height*2,
                NULL );

  XtVaSetValues(XtNameToWidget(zoomCharWidget, "zoomLabel"),
		XmNlabelString, label,
		NULL);

  zoomCharNumber = (int)client_data;
  XtPopup(zoomCharWidget, XtGrabNone);
}



/*
 *----------------------------------------------------------------------
 *
 * specialChars_cancel_CB --
 *
 *      This callback is invoked when the cancel button of the 
 *      'Special Characters' dialog is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
specialChars_cancel_CB(Widget w, XtPointer client_data, XtPointer rs) {

  if( zoomCharNumber != -1 )
    XtDestroyWidget(zoomCharWidget);

  XtDestroyWidget(XtNameToWidget(toplevel, "*sChars"));
  zoomCharNumber = -1;
}



/*
 *----------------------------------------------------------------------
 *
 * specialChars_insert_CB --
 *
 *      This callback is invoked when the insert button of the 
 *      'Special Characters' dialog is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
specialChars_insert_CB(Widget w, XtPointer client_data, XtPointer rs) {

  char cmd[32];

  if( zoomCharNumber == -1 )
    return;

  sprintf(cmd, "Insert_SpecialChar \"%c\"", (char)zoomCharNumber);
  Tcl_MyEval(cmd);
}



/*
 *----------------------------------------------------------------------
 *
 * field_Item_CB --
 *
 *      This callback is invoked when the <Enter> key is entered
 *      in a text field item.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
fieldItem_CB(Widget w, XtPointer client_data, XtPointer rs) {

  char *string;

  string =  XmTextGetString(w);

  switch( (int)client_data ) {
  case 0:  /* document left margin */
    break;
  case 1:  /* document right margin */
    break;
  case 2:  /* document right margin */
    break;
  case 3:  /* document right margin */
    break;
  case 4:  /* paragraph left margin */
    break;
  case 5:  /* paragraph right margin */
    break;
  case 6:  /* paragraph first line margin */
    break;
  case 7:  /* printer name */
    Tcl_SetVar2(theInterp, "print", "printer", string, TCL_GLOBAL_ONLY);
    break;
  case 8:  /* printer command */
    Tcl_SetVar2(theInterp, "print", "command", string, TCL_GLOBAL_ONLY);
    break;
  case 9:  /* printer file */
    Tcl_SetVar2(theInterp, "print", "file", string, TCL_GLOBAL_ONLY);
    break;
  case 10:  /* ispell */
    break;
  case 11: /* document title */
    break;
  case 12: /* document owner */
    break;
  default:
    fprintf(stderr, "Error: no callback registered\n");
  }
  XtFree(string);
  XmProcessTraversal(w, XmTRAVERSE_NEXT_TAB_GROUP);
}



/*
 *----------------------------------------------------------------------
 *
 * error_CB --
 *
 *      This callback is invoked when one of the two buttons of
 *      the 'Error History' dialog is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
error_CB(Widget w, XtPointer client_data, XtPointer rs) {

  if( (int)client_data == 0 )
    XmToggleButtonGadgetSetState(XtNameToWidget(toplevel, "*menu_p4_b6"), False, True);
  else
    XmTextReplace(error_dialog, 0, XmTextGetLastPosition(error_dialog), "");
}



/*
 *----------------------------------------------------------------------
 *
 * ispell_CB --
 *
 *      This callback is invoked when one of the buttons of the
 *      'Ispell' dialog is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
ispell_CB(Widget w, XtPointer client_data, XtPointer rs) {

  Widget list, field;
  char **tablePtr, string[64];
  int num, i, idx;
  extern int docOpen;
  XmString *items;

  list = XtNameToWidget(toplevel, "*ispell*list");
  field = XtNameToWidget(toplevel, "*ispell*field_text");

  switch( (int)client_data ) {
  case 0: /* Ignore */
    Tcl_MyEval("Ispell_Ignore");
    break;

  case 1: /* Replace */
    sprintf(string, "Replace_Word %s", XmTextFieldGetString(field));
    Tcl_MyEval(string);
    Tcl_MyEval("Ispell_Buffer");
    break;

  case 2: /* Start */
    XtSetSensitive(XtNameToWidget(XtParent(w), "*button1"), True);
    XtSetSensitive(XtNameToWidget(XtParent(w), "*button2"), True);
    XtSetSensitive(w, False);

    /*
     * We simulate that document is closed
     * to disable toolbar refresh.
     */
    docOpen = 0;
    Tcl_MyEval("Ispell_Document");
    break;

  default:
    break;
    /* dummy */
  }
  if( Tcl_SplitList(theInterp, theInterp->result, &num, &tablePtr) != TCL_OK )
    return;

  if( num == 0 ) /* An error occured ? */
    return;

  XmListDeleteAllItems(list);

  switch( tablePtr[0][0] ) {
  case '*': /* spelling process is finished */
    XtSetSensitive(XtNameToWidget(XtParent(w), "*button1"), False);
    XtSetSensitive(XtNameToWidget(XtParent(w), "*button2"), False);
    XtSetSensitive(XtNameToWidget(XtParent(w), "*button3"), True);
    XmTextFieldSetString(field, "");
    break;

  case '#': /* submitted word is unknown */
    XmTextFieldSetString(field, tablePtr[1]);
    break;

  default:  /* ispell gave some words */
    XmTextFieldSetString(field, tablePtr[1]);
    items = (XmString *)XtMalloc(sizeof(XmString)*num-4);
    for(i=4; i<num; i++) {
      idx = strlen(tablePtr[i])-1;
      if( tablePtr[i][idx] == ',' )
	tablePtr[i][idx] = '\0';
      items[i-4] = XmStringCreateSimple(tablePtr[i]);
    }
    XmListAddItems(list, items, num-4, 0);
  }
}



/*
 *----------------------------------------------------------------------
 *
 * ispellList_CB --
 *
 *      This callback is invoked when an item in the Ispell dialog
 *      list of words is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
ispellList_CB(Widget w, XtPointer client_data, XtPointer rs) {

  XmListCallbackStruct *cbs = (XmListCallbackStruct *)rs;
  char *string;

  XmStringGetLtoR(cbs->item, XmSTRING_DEFAULT_CHARSET, &string);
  XmTextFieldSetString(client_data, string);
}



/*
 *----------------------------------------------------------------------
 *
 * ispellClose_CB --
 *
 *      This callback is invoked for closing the connection width
 *      ispell. It's called by the <cancel> button of the 'Ispell
 *      dialog'.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
ispellClose_CB(Widget w, XtPointer client_data, XtPointer rs) {
  Tcl_MyEval("Ispell_Close");
}



/*
 *----------------------------------------------------------------------
 *
 * documentInfo_CB --
 *
 *      This callback is invoked when the <Cancel> button of the
 *      'Document Info' is selected.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
documentInfo_CB(Widget w, XtPointer client_data, XtPointer rs) {

  char *owner, *title;
  char cmd[64];

  if( (int)client_data == 0 ) {
    title = XmTextFieldGetString(XtNameToWidget(w, "*title_form.field_text"));
    owner = XmTextFieldGetString(XtNameToWidget(w, "*owner_form.field_text"));

    sprintf(cmd, "DocConfigure -owner %s -title %s", owner, title);
    Tcl_MyEval(cmd);
  }
  XtDestroyWidget(w);
}
