/* 
 * actions.cc --
 *
 *      This file contains fonctions that manage Intrinsics
 *      actions.
 *
 * Copyright (C) 1996-1997  Carlos Nunes - loscar@mime.univ-paris8.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <stdlib.h>

#include <Xm/PushBP.h>


#include <stdio.h>

#include "globals.h"
#include "actions.h"

#include "../tcl/mytcl.h"
#include "callbacks.h"

/*
 * Registered actions
 */
static XtActionsRec actions[]={
  { "n-of-many",       N_of_Many       },
  { "one-of-many",     One_of_Many     },
  { "balloon-on",      Balloon_On      },
  { "balloon-off",     Balloon_Off     },
  { "toggle-activate", Toggle_Activate },
};


extern BOOL Allow_Balloons;
Widget toggle_table[] = { NULL, NULL };
static XtIntervalId id = (XtIntervalId)0;



/*
 *----------------------------------------------------------------------
 *
 * the_timeout --
 *
 *      This action is invoked to display a ballon under a pushbutton,
 *      after a some time.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
the_timeout(XtPointer cl, XtIntervalId *id) {

  Window win_return;
  int x_return, int_return;
  unsigned int mask_return;

  XQueryPointer(XtDisplay(toplevel), 
		XtWindow(toplevel), 
		&win_return, &win_return,
		&x_return, &int_return,
		&int_return, &int_return,
		&mask_return);

  XtVaSetValues(balloon,
                XmNx, x_return,
                NULL );

  XtPopup(balloon, XtGrabNone);
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * toggle_activate --
 *
 *      This action is invoked to process actions associated with
 *      toggle buttons.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

void
Toggle_Activate(Widget w, XEvent *e, String *vector, Cardinal *count) {

  switch( atoi(vector[0]) ) {

  case 0:
    menu_p4_b4_CB(XtNameToWidget(toplevel, "*menu_p4_b4") , NULL, NULL);
    break;

  case 1:
    Tcl_Eval(theInterp, "Toggle_Mask");
    break;

  case 2:
    Tcl_Eval(theInterp, "Toggle_Bold_Attr");
    break;

  case 3:
    Tcl_Eval(theInterp, "Toggle_Italic_Attr");
    break;

  case 4:
    break;

  case 5:
    Tcl_Eval(theInterp, "ParaConfigure -align left");
    break;

  case 6:
    Tcl_Eval(theInterp, "ParaConfigure -align right");
    break;

  case 7:
    Tcl_Eval(theInterp, "ParaConfigure -align center");
    break;

  case 8:
    Tcl_Eval(theInterp, "ParaConfigure -align full");
    break;
    
  default:
/*    fprintf(stderr, "ToggleButton: %d\n", atoi(vector[0])); */
    break;
  }    
}



/*
 *----------------------------------------------------------------------
 *
 * Balloon_On --
 *
 *      This action is invoked when the cursor enters a pushbutton
 *      which gots a balloon. So the counting time starts.
 *      toggle buttons.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

void
Balloon_On(Widget w, XEvent *e, String *vector, Cardinal *count) {

  Position y, x2, y2, h, mh;
  XmString label;

  if( id != (XtIntervalId)0 )
    return;

  if( Allow_Balloons == FALSE )
    return;

  XtVaGetValues(w,
		XmNy, &y,
		XmNheight, &h,
		XmNmarginHeight, &mh,
		NULL);
  
  XtTranslateCoords(w,
		    (Position) 0,       /* The x value is defined later */
		    (Position) y,
		    &x2, &y2);

  XtVaSetValues(balloon,
                XmNy, y2 + h-mh,
                NULL );

  label = XmStringCreateSimple(vector[0]);
  XtVaSetValues(XtNameToWidget(balloon, "balloonLabel"),
		XmNlabelString, label,
		NULL);
  XmStringFree( label );

  label = XmStringCreateSimple(vector[1]);
  XtVaSetValues(mainLabel,
		XmNlabelString, label,
		NULL);
  XmStringFree( label );

  id = XtAppAddTimeOut(XtWidgetToApplicationContext(toplevel),
		       500, (XtTimerCallbackProc)the_timeout, NULL);
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * Balloon_Off --
 *
 *      This action is invoked when the cursor leaves a pushbutton
 *      which gots a balloon. So the counter is stopped.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

void
Balloon_Off(Widget w, XEvent *e, String *vector, Cardinal *count) {

  XmString label;

  XtPopdown( balloon );

  label = XmStringCreateSimple(" ");

  XtVaSetValues(mainLabel,
	      XmNlabelString, label,
	      NULL);

  XmStringFree( label );

  if( id != (XtIntervalId)0 ) {
    XtRemoveTimeOut( id );
    id = (XtIntervalId)0;
  }
  return;
}



/*
 *----------------------------------------------------------------------
 *
 * N_of_Many --
 *
 *      This action is invoked to turn push buttons into toggle
 *      buttons. This kind of toggle buttons can be select in
 *      the same time as the others.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

void
N_of_Many(Widget w, XEvent *e, String *vector, Cardinal *count) {
  
  if( ((XmPushButtonWidget)w)->pushbutton.armed == 0 )
    XtCallActionProc(w, "Arm", NULL, NULL, 0);
  else
    XtCallActionProc(w, "Disarm", NULL, NULL, 0);
}



/*
 *----------------------------------------------------------------------
 *
 * One_of_Many --
 *
 *      This action is invoked to turn push buttons into toggle
 *      buttons. The selection of this kind of toggle buttons is
 *      exclusive, so if there're another selected toggle button,
 *      then this one will be released
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

void
One_of_Many(Widget w, XEvent *e, String *vector, Cardinal *count) {

  int offset = 0;

  if( toggle_table[ offset ] != NULL )
    XtCallActionProc(toggle_table[ offset ], "Disarm", NULL, NULL, 0);
  
  if( toggle_table[ offset ] != w ) {
    XtCallActionProc(w, "Arm", NULL, NULL, 0);
    toggle_table[ offset ] = w;    
  }
}



/*
 *----------------------------------------------------------------------
 *
 * Set_Actions --
 *
 *      This action is invoked to turn on the actions.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

void
Set_Actions(XtAppContext context) {

  XtAppAddActions(context, actions, XtNumber(actions));
}


/*
 *----------------------------------------------------------------------
 *
 * Arm_ToggleButton --
 *
 *      This action is invoked to force a toggle button to be armed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

void
Arm_ToggleButton(char *path) {

  Widget button;

  button = XtNameToWidget(toplevel, path);
  if( ((XmPushButtonWidget)button)->pushbutton.armed == 0 )
    XtCallActionProc(button, "Arm", NULL, NULL, 0);
}


/*
 *----------------------------------------------------------------------
 *
 * Disarm_ToggleButton --
 *
 *      This action is invoked to force a toggle button to be not
 *      armed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

void
Disarm_ToggleButton(char *path) {

  Widget button;

  button = XtNameToWidget(toplevel, path);
  if( ((XmPushButtonWidget)button)->pushbutton.armed != 0 )
    XtCallActionProc(button, "Disarm", NULL, NULL, 0);
}
