/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE : mx_sc_col.C
 *
 * AUTHOR : David Miller
 *
 * This file 
 *
 * DESCRIPTION: 
 * Module mx_sc_col.C 
 * 
 *
 *
 */

#include <mx_sc_device.h>

mx_hash mx_screen_device::colour_dict;
static mx_hash colour_name_dict;
static mx_hash colour_val_dict;

/*-------------------------------------------------
 * FUNCTION: mx_screen_device::registerColour
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_screen_device::registerColour(int       &err,
                                      mx_colour *colour,
                                      LUT_VALUE *pixel) 
{
    XColor screenColour,exactColour ;
    XColor *nextColour ;
    bool   gotPixel = FALSE ;
    bool   found = FALSE ;
    int    dist, newdist ;
    
    err = MX_ERROR_OK ;
    *pixel = BlackPixel(display,DefaultScreen(display)) ;
    
    // Try by name first
    if(colour->name != NULL) 
    {
        *pixel = (LUT_VALUE)colour_name_dict.get(err, colour->name);
        if (err != MX_HASH_NOT_FOUND)
        {
            MX_ERROR_CHECK(err);
            return;
        }
        else
        {
            MX_ERROR_CLEAR(err);
            
            if(XLookupColor(display,
                            DefaultColormap(display,DefaultScreen(display)),
                            colour->name,&screenColour,&exactColour)) 
            {
                gotPixel = XAllocNamedColor(
                    display,DefaultColormap(display,DefaultScreen(display)),
                    colour->name,&screenColour,&exactColour) ;

                if (gotPixel)
                {
                    colour_name_dict.add(err, colour->name, 
                                         (void *)screenColour.pixel);
                    MX_ERROR_CHECK(err);
                }
            }
        }
    }

    if(!gotPixel) 
    {
        uint32 dict_lu = ((((uint32)colour->red) << 16 & 0x00ff0000) |
                          (((uint32)colour->green) << 8 & 0x0000ff00) |
                          (((uint32)colour->blue) & 0x000000ff));
        
        *pixel = (LUT_VALUE)colour_val_dict.get(err, dict_lu);
        if (err != MX_HASH_NOT_FOUND)
        {
            MX_ERROR_CHECK(err);
            return;
        }
        else
        {
            // Try the pixel value 
            
            screenColour.red   = (uint16)colour->red << 8;
            screenColour.green = (uint16)colour->green << 8;
            screenColour.blue  = (uint16)colour->blue << 8;
            screenColour.flags = DoRed | DoGreen | DoBlue;

            gotPixel = XAllocColor(
                display, DefaultColormap(display,DefaultScreen(display)),
                &screenColour) ;

            if (gotPixel)
            {
                colour_val_dict.add(err, dict_lu, (void *)screenColour.pixel);
                MX_ERROR_CHECK(err);
            }
        }
    }
    
    if(!gotPixel) 
    {
        // Havent got a pixel value yet - map to the nearest for which 
        // we have read access 
        colour_dict.iterate_start(err) ;
        MX_ERROR_CHECK(err) ;
        
        dist = 65536*4;
        bool foundPixel = FALSE ;
        while((nextColour = (XColor *)colour_dict.iterate_next(err))) 
        {
            MX_ERROR_CHECK(err) ;
            
            newdist = abs(colour->red   - nextColour->red)   +
                             abs(colour->green - nextColour->green) +
                      abs(colour->blue  - nextColour->blue) ;
            if(newdist < dist) 
            {
	      foundPixel = TRUE ;
	      *pixel = nextColour->pixel ;
	      dist = newdist; 
            }
        }
        MX_ERROR_CHECK(err) ;
        
        // No colour - use the foreground colour 
      if(!foundPixel) *pixel = BlackPixel(display,DefaultScreen(display)) ;
    }
    else
    {
        *pixel = screenColour.pixel;
        
        // Have we already got the pixel value 
        
        colour_dict.iterate_start(err) ;
        MX_ERROR_CHECK(err) ;
        
        found = FALSE ;
        while((nextColour = (XColor *)colour_dict.iterate_next(err)))
        {
            MX_ERROR_CHECK(err) ;
            if((nextColour->red   == screenColour.red) &&
               (nextColour->green == screenColour.green) &&
               (nextColour->blue  == screenColour.blue)) 
            {
                found = TRUE ;
                break ;
            }
        }
        MX_ERROR_CHECK(err) ;
        
        if(!found) 
        {
            nextColour = new XColor ;
            
            *nextColour = screenColour ;
            colour_dict.add(err,*pixel,nextColour) ;        
            MX_ERROR_CHECK(err) ;
        }
    }
    
    return ;
  abort:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: setForegroundColour
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_screen_device::setForegroundColour(int &err, LUT_VALUE pixel)
{
    err = MX_ERROR_OK ;

    XSetForeground(display, gc, pixel) ;

    foreground = pixel;

    return ;
}

/*-------------------------------------------------
 * FUNCTION: setBackgroundColour
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_screen_device::setBackgroundColour(int &err, LUT_VALUE pixel)
{
    err = MX_ERROR_OK ;

    XSetBackground(display, gc, pixel) ;

    background = pixel;

    return ;
}

void mx_screen_device::setForegroundColour(int &err, mx_colour &c)
{
    if (!c.pixel_value_set)
    {
        registerColour(err, &c, &c.pixel_value);
        MX_ERROR_CHECK(err);

        c.pixel_value_set = TRUE;
    }

    setForegroundColour(err, c.pixel_value);
    MX_ERROR_CHECK(err);
abort:;
}

void mx_screen_device::setBackgroundColour(int &err, mx_colour &c)
{
    if (!c.pixel_value_set)
    {
        registerColour(err, &c, &c.pixel_value);
        MX_ERROR_CHECK(err);

        c.pixel_value_set = TRUE;
    }

    setBackgroundColour(err, c.pixel_value);
    MX_ERROR_CHECK(err);
abort:;
}
