/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_type1
 *
 * AUTHOR : Andrew Haisley
 *
 * 
 * 
 * Description routines for manipulating type 1 fonts in clever and
 * interesting ways.
 *
 *
 *
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <mx.h>
#include <mx_type1.h>

#define PFB_MARKER   128
#define PFB_ASCII    1
#define PFB_BINARY   2
#define PFB_DONE     3

static void output_hex(int b, FILE *f)
{
    static int col = 0;

    if (col >= 32)
    {
        fputc('\n', f);
        col = 0;
    }
    else
    {
        col++;
    }

    fprintf(f, "%02x", b);
}

static int get_block_length(FILE *f)
{
  int length;

  length = fgetc(f);
  length |= fgetc(f) << 8;
  length |= fgetc(f) << 16;
  length |= fgetc(f) << 24;

  return length;
}

static void mx_pfb_translate_blocks(int &err, FILE *in_file, FILE *out_file)
{
    int length, c, last_type = PFB_ASCII;

    err = MX_ERROR_OK;

    while (TRUE)
    {
        c = fgetc(in_file);
        if (c == EOF) 
        {
            break;
        }

        if (c != PFB_MARKER) 
        {
            return;
        }

        c = fgetc(in_file);

        switch (c)
        {
            case PFB_ASCII:
                if (last_type != PFB_ASCII)
                {
                    fputc('\n', out_file);
                }
                last_type = PFB_ASCII;
                for (length = get_block_length(in_file); length > 0; length--)
                {
                    c = fgetc(in_file);
                    if (c == '\r')
                    {
                        fputc('\n', out_file);
                    }
                    else
                    {
                        fputc(c, out_file);
                    }
                }
                break;

            case PFB_BINARY:
                last_type = PFB_BINARY;
                for (length = get_block_length(in_file); length > 0; length--)
                {
                    output_hex(fgetc(in_file), out_file);  
                }
 
            default:
            case PFB_DONE:
                break;
        }
    }
}

// translates a type 1 pfb font into a type 1 pfa font
void mx_pfb_2_pfa(int &err, const char *in, const char *out)
{
    FILE *in_file;
    FILE *out_file;

    in_file = fopen(in, "r");
    if (in_file == NULL)
    {
        err = mx_translate_file_error(errno);
        return;
    }

    out_file = fopen(out, "w");
    if (out_file == NULL)
    {
        err = mx_translate_file_error(errno);
        fclose(in_file);
        return;
    }

    mx_pfb_translate_blocks(err, in_file, out_file);
    MX_ERROR_CHECK(err);

    fclose(in_file);
    fclose(out_file);

abort:;
}

// gets the font name out of a pfa
char *mx_pfa_name(int &err, char *in)
{
    FILE *f;
    static char res[200];
    static char line[200];
    int i, j;

    f = fopen(in ,"r");
    if (f == NULL)
    {
        err = mx_translate_file_error(errno);
        return NULL;
    }

    while (TRUE)
    {
        if (fgets(line, 199, f) == NULL)
        {
            break;
        }
        if (strncmp(line, "/FullName", 9) == 0)
        {
            i = 0;
            while (line[i] != '(' && line[i] != 0) 
            {
                i++;
            }
            if (line[i] == 0)
            {
                fclose(f);
                return "Corrupt font";
            }
            i++;
            j = 0;
            while (line[i] != ')' && line[i] != 0) 
            {
                res[j++] = line[i++];
            }
            if (line[i] == 0 || j == 0)
            {
                fclose(f);
                return "Corrupt font";
            }
            else
            {
                res[j - 1] = 0;
                fclose(f);
                return res;
            }
        }
    }
    fclose(f);
    return "Corrupt font";
}

// gets the font name out of a pfb
char *mx_pfb_name(int &err, char *in)
{
    char *pfa, *res;

    pfa = mx_tmpnam(NULL);

    mx_pfb_2_pfa(err, in, pfa);
    MX_ERROR_CHECK(err);

    res = mx_pfa_name(err, pfa);
    MX_ERROR_CHECK(err);

    return res;

abort:;
    return NULL;
}
