/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef MX_FONT_METRICS_H
#define MX_FONT_METRICS_H
/*
 * MODULE/CLASS : mx_font_metrics & mx_font_metrics_store
 *
 * AUTHOR : Tom Newton
 *
 * 
 *
 * DESCRIPTION:
 *
 * The mx_font_metrics class represents the metrics for a given font. It is
 * used by mx_font to get its characters' widths etc. The mx_font_metrics_store
 * class is used as a nice interface to a static hash table for a load of
 * instances of mx_font_metrics.
 *
 *
 *
 * 
 */

#include <mx.h>
#include <mx_font.h>

class mx_list;
class mx_font_family;
class mx_font_metrics;
class mx_font_metrics_store;


// This class describes a family of fonts. It is a very simple class which
// simple holds pointers to four font metrics objects, one for each style,
// normal, bold, italic, and bold-italic.
class mx_font_family
{
public:
    mx_font_family();
    ~mx_font_family();

    // gets a pointer to some font metrics for the given style. Returns NULL if
    // there are none for a given style.
    mx_font_metrics *get_font_metrics(mx_font_style_t st) const;

    // stores the font metrics class in the relevant slot;
    void set_font_metrics(mx_font_metrics *fm);

    // Tells us whether the font family has bold or italics defined for it
    void family_has_B_or_I(bool &b, bool &it);

    // returns the name of the font family
    const char *get_name();

private:

    friend uint32 mx_font::em_width(char character) const;

    mx_font_metrics *styles[4];
};

// This class is a storage mechanism for keeping rtrack of font families. It is
// basically a wrapper for a hash table of font families, keyed by the family
// name.
class mx_font_metrics_store
{
public:

    mx_font_metrics_store();
    ~mx_font_metrics_store();

    // gets the named font family. Returns error if the named font family does
    // not exist
    mx_font_family *get_font_family(int &err, const char *font_family_name);

    // gets a family whose named contains the given string (case insensitive)
    mx_font_family *get_nearest_font_family(int &err, const char *name);

    // gets the metrics for a particular family and style
    mx_font_metrics *get_font_metrics(int &err, const char *font_family_name,
                                      mx_font_style_t style);

    // returns the number of font families stored here
    uint32 get_num_font_families();

    // gets a list of font family names 
    void get_font_family_names(mx_list &list);

private:

    void add_fm(mx_font_metrics *fm);
    
    void init_type1_fm(int &err);

    // a hash table of font_familys. Each family contains up to four pointers to
    // mx_font_metrics instances, according to the bold/italic options. The
    // pointers are NULL if a particular bold/italic style does not exist.
    static mx_hash font_families;
};

#define MX_FM_CHARSET_SIZE 256

// This defines a class for storing the metrics information for a given
// font. There is functionality for reading postscript font metrics information
// and also reading/writing the maxwell font metrics files.
class mx_font_metrics
{
public:
    mx_font_metrics();
    ~mx_font_metrics();

    // The maxwell font names are in this format:
    char *get_name() const;
    char *get_x_name() const;
    char *get_ps_name() const;
    char *get_full_name() const;
    char *get_file_name() const;
    mx_font_style_t get_style() const;
    int32 get_em_ascender() const;
    int32 get_em_descender() const;

    // This function gets called a lot by mx_font. It returns the wdith (in
    // thousands of an em-square) of a particular character, or zero if the
    // character does not exist for this font.
    uint32 get_em_width(uint16 c) const;

    // set methods to be used by mx_font_metrics_store only
    void set_name(const char *name);
    void set_x_name(const char *x_name);
    void set_full_name(const char *ps_name);
    void set_ps_name(const char *ps_name);
    void set_file_name(const char *filename);
    void set_style(mx_font_style_t st);
    void set_em_ascender(int32 a);
    void set_em_descender(int32 d);

    void set_em_width(uint16 character, uint32 em_width);

    // This outputs a maxwell font metrics (.mfm) file for the given font
    void write_mfm(int &err, char *file_name);

    // This reads a maxwell font_metrics (.mfm) file into txhis instance.
    void read_mfm(int &err, char *file_name);

    // This decrypts and decodes a type 1 postscript font. The info should be
    // saved to an mfm after this is called, because they can be read more
    // quickly. 
    void read_type1(int &err, const char *file_name, const char *x_name);

private:

    // private variables

    char            *family_name;        // name of the font family
    char            *x_name;             // the name used by X to reference
                                         // this font
    char            *file_name;          // The fonts full pathname
    char            *full_name;          // the full name
    char            *ps_name;            // the type1 name
    mx_font_style_t style;               // the style of the font

    uint32           widths_arr[MX_FM_CHARSET_SIZE];
                                         // an array of font widths
    int32           em_ascender;         // height of the ascender 
    int32           em_descender;        // height of the descender

    mx_hash         *encoding_to_type1;  // encoding lookup tables
    mx_hash         *type1_to_encoding;
    
    // private methods for reading type1 fonts

    void get_type1_family_name(int &, const char *, uint32);
    void get_type1_ps_name(int &, const char *, uint32);
    void get_type1_full_name(int &, const char *, uint32);
    void get_type1_style(int &, const char *, uint32);
    void get_type1_bounding_box(int &, const char *, uint32);
    void get_type1_encoding(int &, const char *, uint32);
    void get_type1_char_widths(int &, const char *, uint32);
    const char *get_type1_char_width(int &, const char *);

    // private methods for accessing the map between iso numbers and type1
    // character names

    void get_iso_lookup(int &err);
    uint16 get_iso(int &err, const char *type1_name);
    char *get_type1_name(int &err, uint16 iso_num);

    // private hash tables for the iso num to type 1 name map

    static mx_hash iso_to_type1;
    static mx_hash type1_to_iso;

    // private variable indicating the number of instances of this class
    static uint32 num_instances;
};

#endif // MX_FONT_METRICS_H
