/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE : mx_help_file.C
 *
 * AUTHOR : David Miller
 *
 * DESCRIPTION: 
 * Module mx_help_file.C 
 * 
 *
 */

#include "mx_help.h"
#include <memory.h>
#include <ctype.h>


/*-------------------------------------------------
 * FUNCTION: mx_help_file::mx_help_file
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_file::mx_help_file(int &err,
                           char *iFullFilename,
                           char *ifileName)
:mx_rtf(err,iFullFilename),
fileSize(0,0)
{
  MX_ERROR_CHECK(err);

  processingHidden = FALSE;
  gettingLinkText  = FALSE ;
  
  paraBufferSize      = 0 ;
  allocParaBufferSize = 0 ;
  paraBuffer          = NULL;
  
  hiddenBufferSize      = 0 ;
  allocHiddenBufferSize = 0 ;
  hiddenBuffer          = NULL ;
  
  // the number of lines in the file
  currentLine           = 0 ;
  nlines                = 0 ;
  nallocLines           = 0 ;
  lines                 = NULL ;
  
  // the links in the file
  currentLink           = 0 ;
  nlinks                = 0 ;
  nallocLinks           = 0 ;
  links                 = NULL ;
  
  fullFileName = new char[strlen(iFullFilename)+1] ;
  strcpy(fullFileName,iFullFilename) ;
  
  fileName = new char[strlen(ifileName)+1] ;
  strcpy(fileName,ifileName) ;

  return ;
  
 abort:
  return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::~mx_help_file
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_file::~mx_help_file()
{
    int iline ;
    
    delete [] paraBuffer ;
    delete [] hiddenBuffer ;
    delete [] fileName ;
    delete [] fullFileName ;
    delete [] links ;
    
    for(iline=0;iline<nlines;iline++) 
    {
        delete lines[iline] ;
    }
    
    delete [] lines ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_line::mx_help_line
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_line::mx_help_line() 
{
   line           = NULL ;
   lineSize       = 0 ;
   justification  = mx_line_left_justify_e;
   nlinks         = 0 ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_line::~mx_help_line
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_line::~mx_help_line()
{
    // Delete the line text 
    delete [] line ;
    
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_line::setLine
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_line::setLine(int               &err, 
                           int               ilineSize, 
                           char              *lineStart,
                           double            iindentSize,
                           double            fileOffset,
                           mx_line_justify_t ijustification,
                           int               inlinks,
                           int               ilinkOffset)
{
   err = MX_ERROR_OK ;

   lineSize        = ilineSize ;
   linePos.p.x     = iindentSize ;
   linePos.p.y     = fileOffset ;
   justification   = ijustification;
   nlinks          = inlinks ;
   linkOffset      = ilinkOffset ;

   // copy the text for this line
   if(lineSize == 0) 
   {
      line = NULL ;
   }
   else
   {
      line = new char[lineSize+1] ;
      memcpy(line,lineStart,lineSize) ;
      line[lineSize] = 0 ;
   }

   return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::getIndentSize
 *
 * DESCRIPTION: Get the indent associated with a 
 * paragraph
 *
 */

mx_point mx_help_file::getIndentSize()
{
    mx_point z(0,0);
    
    mx_file_link *nextLink ;
    
    // get the next link to be processed - we want the link to 
    // be at the beginning of the paragraph and to 
    // to have a pixmap associated with it 

    if(currentLink < nlinks) 
    {
        nextLink = links + currentLink ;
        if( (nextLink->start == 0) && (nextLink->hasPixmap)) 
        {
            z    = helpTarget->getPixmapSize() ;
            z.x += helpTarget->getLeftMargin() ;        
        }
    }

    z.x += helpTarget->getLeftMargin() ;
    
    // this paragraph isnt indented
    return z;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::getLineLinks
 *
 * DESCRIPTION: Split the characters in a paragraph 
 * into lines
 *
 */

void mx_help_file::getLineLinks(int  lineStartOffset,
                                int   &startLink,
                                int   &nlineLinks,
                                int   &lineSize) 
{
    mx_file_link *nextLink ;
    int endOffset  = lineStartOffset + lineSize - 1 ;
    
    nlineLinks = 0 ;
    startLink  = currentLink ;
    
    // processed all links already
    if(currentLink == nlinks) return ;
    
    nextLink = links + currentLink ;

    while( (nextLink != NULL) && (nextLink->end <= endOffset))
    {
        // this link lives on this line 
        nlineLinks++ ;
        currentLink++ ;

        if(currentLink >= nlinks) 
        {
            nextLink = NULL ;
        }
        else
        {
            nextLink = links + currentLink ;
        }
    }

    // now check that the next link is not broken by the
    // line break
    if(nextLink != NULL) 
    {
        if(nextLink->start <= endOffset) 
        {
            /* The link is split by the line */
            lineSize -= (endOffset - nextLink->start + 1) ;
        }
    }
}


/*-------------------------------------------------
 * FUNCTION: mx_help_file::processLineLinks
 *
 * DESCRIPTION: Split the characters in a paragraph 
 * into lines
 *
 */

void mx_help_file::processLineLinks(int  lineStartOffset,
                                    int  startLink,
                                    int  nlineLinks,
                                    char *line,
                                    double xlineStart,
                                    double ylineStart) 
{
    int ilink ;
    
    mx_file_link *nextLink ;

    for(ilink=0;ilink<nlineLinks;ilink++) 
    {
        nextLink = links + startLink + ilink ;

        // get start and end relative to the line 
        nextLink->start -= lineStartOffset ;
        nextLink->end   -= lineStartOffset ;

        if(nextLink->hasPixmap)
        {
            mx_point psz = helpTarget->getPixmapSize() ;


            nextLink->extent.xb = helpTarget->getLeftMargin() ;
            nextLink->extent.yb = ylineStart - helpTarget->getLineHeight() ;
            nextLink->extent.xt = nextLink->extent.xb + psz.x ;
            nextLink->extent.yt = nextLink->extent.yb + psz.y ;
        }
        else
        {
            nextLink->extent.xb = helpTarget->lineMMLength(line,nextLink->start) + xlineStart ;
            nextLink->extent.yb = ylineStart - helpTarget->getLineAscender() ;
            nextLink->extent.xt = helpTarget->lineMMLength(line,nextLink->end+1)   + xlineStart ;
            nextLink->extent.yt = ylineStart ;
        }
        
    }
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::splitParagraphLines
 *
 * DESCRIPTION: Split the characters in a paragraph 
 * into lines
 *
 */

void mx_help_file::splitParagraphLines(int &err,
                                       mx_line_justify_t justification) 
{
    char         *paraBufferEnd ;
    char         *lineStart ;
    char         *lineEnd ;
    int          lineStartOffset ;
    mx_point     indentSize ;
    double       maxMMsize ;
    double       minMMsize ;
    int          idealCharLength ;
    int          lineCharLength ;
    int          lineInc ;
    int          paraBufferLeft ;
    int          firstLink ;
    int          numberLineLinks ;
    mx_help_line *nextLine ;
    double       lineMMLength ;
    int          lineSize ;
    int          nlines = 0 ;
    double       thisLineLength,fileAddOn ;
    double       lineHeight ;
    double       lineStartXOffset ;
    
    err  = MX_ERROR_OK  ;
    
    if (paraBufferSize <= 0) 
    {
            fileSize.y += lineHeight; 
            return;
    }
 
    paraBufferEnd     = paraBuffer + paraBufferSize - 1 ;
    lineStart         = paraBuffer ;
    lineStartOffset   = 0 ;
    lineHeight        = helpTarget->getLineHeight() + helpTarget->getLineSeparator();

    // get the indent size for this paragraph
    indentSize = getIndentSize() ;
    
    // put two lines at the top
    if(fileSize.y == 0) fileSize.y = 2.0*lineHeight ;
    
    // get the limits for the size of the line to iterate over 
    
    maxMMsize       = helpTarget->getMaxMMLineLength() - indentSize.x ;
    minMMsize       = helpTarget->getMinMMLineLength() - indentSize.x ;
    idealCharLength = helpTarget->getLineIdealCharacterLength() ;

    // while not at end of the bufffer continue
    while (lineStart <= paraBufferEnd) 
    {
        // guess a length for the string which will fit - this is the
        // start estimate for an iteration 
        
        lineCharLength = idealCharLength;
        lineInc        = (lineCharLength - 1) / 2;
        
        paraBufferLeft = paraBufferEnd - lineStart + 1;

        if (lineCharLength > paraBufferLeft) 
        {
            lineCharLength = paraBufferLeft ;
        }
        
        while (TRUE)
        {
            lineMMLength = helpTarget->lineMMLength(lineStart, lineCharLength);

            if (lineMMLength > maxMMsize)
            {
                if (lineCharLength > 1)
                {
                    lineCharLength--;
                }
                break;
            }
            lineCharLength++;
            if (lineCharLength > paraBufferLeft)
            {
                lineCharLength--;
                break;
            }
        }
        
        lineEnd = lineStart + lineCharLength - 1;
          
        // Go back to next white space so dont split word unless we are at the end of the
        // paragraph

        if (lineEnd != paraBufferEnd) 
        {
            while ((lineEnd >= lineStart) && (!isspace(*lineEnd))) 
            {
                lineEnd--;
            }
        }

        // if right justification now eat the white space too 
        if( (lineEnd != paraBufferEnd) && (justification == mx_line_right_justify_e)) 
        {
            while ((lineEnd >= lineStart) && (isspace(*lineEnd))) 
            {
                lineEnd--;
            }
        }

        // process the links in this line - a link 
        // cannot be split over a line so the line might end up shorter 
        // than before 
        lineSize  = lineEnd - lineStart + 1; 

        // get the number of links in the line -
        // note that links do not get split so 
        // the line length might be decreased 
        getLineLinks(lineStartOffset,
                     firstLink,
                     numberLineLinks,
                     lineSize) ;
        
        if(lineSize == 0) 
        {
            goto abort;
        }

        // get the new length 
        lineMMLength = helpTarget->lineMMLength(lineStart, lineSize);

        switch(justification) 
        {
        case mx_line_right_justify_e:
            lineStartXOffset = indentSize.x + (maxMMsize - lineMMLength);
            break;
        case mx_line_centre_justify_e: 
            lineStartXOffset = indentSize.x + ((maxMMsize - lineMMLength) * 0.5);
            break;
        default:
            lineStartXOffset = indentSize.x;
            break;                   
        }
        


        // get the new length
        {
            char sv ;
            
            sv = lineStart[lineSize] ;
            lineStart[lineSize] = 0 ;

            lineStart[lineSize] = sv ;
        }
        

        processLineLinks(lineStartOffset,
                         firstLink,
                         numberLineLinks,
                         lineStart,
                         lineStartXOffset,
                         fileSize.y + lineHeight*nlines) ;

        thisLineLength = lineMMLength + lineStartXOffset ;
        if(thisLineLength > fileSize.x) fileSize.x = thisLineLength ;

        // get memory for the next line 
        nextLine = nextHelpLine() ;
        
        // set the fields for the line
        nextLine->setLine(err,
                          lineSize,
                          lineStart,
                          lineStartXOffset,
                          fileSize.y + lineHeight*nlines,
                          justification,
                          numberLineLinks,
                          firstLink) ;
        MX_ERROR_CHECK(err);
        
        lineStart       += lineSize ;
        lineStartOffset += lineSize ;

        nlines++ ;
        
    }

    //fileAddOn = DMAX(nlines*lineHeight,indentSize.y) + lineHeight ;
    fileAddOn = nlines * lineHeight;

    fileSize.y += fileAddOn ; 
    return ;
  abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::processNewParagraph
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_file::processNewParagraph(int                 &err,
                                       RTFSgroup           *group,
                                       rtfDestinationType  destination) 

{
   mx_line_justify_t just;

   err = MX_ERROR_OK ;

   // get the formatting for this paragraph style 

   switch (group->formatting.paragraph.fmt.alignment)
   {
     case RTFqr : 
       just = mx_line_right_justify_e;
       break;
     case RTFqj : 
       just = mx_line_full_justify_e;
       break;
     case RTFqc : 
       just = mx_line_centre_justify_e;
       break;
     default:
       just = mx_line_left_justify_e;
       break;       
   }

   // If processing a hidden buffer at the end of the paragraph
   // get any links

   processHiddenBuffer(err) ;
   MX_ERROR_CHECK(err) ;

   // split the paragraoh into lines 
   splitParagraphLines(err,just) ;
   MX_ERROR_CHECK(err) ; 
   
   paraBufferSize       = 0 ;
   currentLine          = nlines ;
   currentLink          = nlinks ;

   goto exit;
 abort:
 exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::processSpecialParagraph
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_file::processSpecialCharacter(int                 &err,
                                           RTFSgroup           *group,
                                           char                special,
                                           rtfDestinationType  destination) 
{
   err = MX_ERROR_OK ;

}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::processNewTab
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_file::processNewTab(int                 &err,
                                 RTFSgroup           *group,
                                 rtfDestinationType  destination) 

{
   err = MX_ERROR_OK ;

   return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::processMainOutput
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_file::processMainOutput(int                 &err,
                                     RTFSgroup           *group) 
{
   err = MX_ERROR_OK ;

   if(textBufferSize == 0) return ;

   if(group->formatting.character.fmt.v) 
   {
      addToHidden(err,textBuffer,textBufferSize);
      MX_ERROR_CHECK(err);
   }
   else
   {
       processHiddenBuffer(err) ;
       MX_ERROR_CHECK(err) ;

       addToParagraph(err,textBuffer,textBufferSize);
       MX_ERROR_CHECK(err);
   }
   
   goto exit;
  abort:
  exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: createFileName
 *
 * DESCRIPTION: Get the file name from the hidden text
 * 
 *
 */

void createFileName(int &err,
                    char *inBuffer,
                    int  inBufferLen,
                    char outBuffer[]) 
{
    char *offsetPtr ;
    int nbytesStr ;
    
    err = MX_ERROR_OK ;
    
    // Search through for closing bracket 
    
    offsetPtr = (char *) memchr((void *)inBuffer,')',inBufferLen) ;
    
    if(offsetPtr != NULL) 
    {
        // The length of the string is equal to 
        // the pointer difference
        nbytesStr = offsetPtr - inBuffer ;
        if(nbytesStr > 127) 
        {
            MX_ERROR_THROW(err,MX_FILE_LINK_TOO_LONG);
        }
        else
        {
            // Save the string
            memcpy(outBuffer,inBuffer,nbytesStr) ;
            outBuffer[nbytesStr] = 0 ;
        }
    }
    else
    {
        MX_ERROR_THROW(err,MX_FILE_LINK_TOO_LONG);
    }
    
    return ;
  abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::nextFileLink
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_file_link *mx_help_file::nextFileLink() 
{
    mx_file_link *thisFileLink ;
    
    if(nlinks == nallocLinks) 
    {
        // store old links 
        mx_file_link *oldLinks = links ;
        
        nallocLinks += 4 ;
        
        links = new mx_file_link[nallocLinks] ;
        
        memcpy(links,oldLinks,sizeof(mx_file_link)*nlinks) ;
        
        delete [] oldLinks ;
    }

    thisFileLink = links + nlinks ;
    nlinks++ ;
    
    return thisFileLink ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::nextHelpLine
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_line *mx_help_file::nextHelpLine() 
{
    mx_help_line *thisFileLine ;
    
    if(nlines == nallocLines) 
    {
        // store old lines 
        mx_help_line **oldLines = lines ;
        
        nallocLines += 32 ;
        
        lines = new mx_help_line_ptr[nallocLines] ;
        
        memcpy(lines,oldLines,sizeof(mx_help_line_ptr)*nlines) ;
        
        delete [] oldLines ;
    }

    thisFileLine  = new mx_help_line ;

    lines[nlines] = thisFileLine ; 
    nlines++ ;
    
    return thisFileLine ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::processHiddenBuffer
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_file::processHiddenBuffer(int &err)
{
    mx_file_link *currentFileLink ;
    
    err = MX_ERROR_OK ;
    
    
    if( (!processingHidden) || (hiddenBufferSize == 0)) 
    {
        processingHidden = FALSE ;
        return ;
    }
        
    if( (hiddenBufferSize > 11) && (memcmp(hiddenBuffer,"(LINKSTART-",11)==0)) 
    {
        // It is the start of a link
        if(gettingLinkText) 
        {
            // Cant already be getting a link
            MX_ERROR_THROW(err,MX_LINK_FORMAT_ERROR);
        }
        
        gettingLinkText = TRUE ;
        
        // Store the offset in the paragraph buffer
        // where the current button text starts 

        currentFileLink = nextFileLink() ;
        MX_ERROR_CHECK(err);
        
        currentFileLink->start = paraBufferSize ;
        currentFileLink->hasPixmap = FALSE ;

        // Get the filename associated with the link
        createFileName(err,hiddenBuffer+11,hiddenBufferSize-11,
                       currentFileLink->fileName) ;
        MX_ERROR_CHECK(err) ;
    }
    else if( (hiddenBufferSize == 9) && (memcmp(hiddenBuffer,"(LINKEND)",9)==0))
    {
        if(!gettingLinkText) 
        {
            MX_ERROR_THROW(err,MX_LINK_FORMAT_ERROR);
        }

        currentFileLink = links + nlinks - 1 ;
        
        // Store the end of the buffer text 
        currentFileLink->end = paraBufferSize - 1 ;
        
        // No longer getting button text
        gettingLinkText = FALSE ;
    }
    else if( (hiddenBufferSize > 7) && (memcmp(hiddenBuffer,"(BLINK-",7)==0)) 
    {
        currentFileLink = nextFileLink() ;
        MX_ERROR_CHECK(err);

        // Attach push button for whole paragraph
        currentFileLink->start = paraBufferSize ;
        currentFileLink->end   = paraBufferSize ;
        
        currentFileLink->hasPixmap = TRUE ;
        createFileName(err,hiddenBuffer+7,hiddenBufferSize-7,
                       currentFileLink->fileName) ;
        MX_ERROR_CHECK(err) ;
        
    }
    
  abort:
    
    processingHidden = FALSE ;
    hiddenBufferSize = 0 ;
    
   return; 
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::addToParagraph
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_file::addToParagraph(int  &err,
                                  char *chars,
                                  int  charLen) 
{
   char *newBuffer ;
   int newParaBufferSize ;

   err = MX_ERROR_OK ;
   
   newParaBufferSize = paraBufferSize + charLen ;

   if(newParaBufferSize > allocParaBufferSize) 
   {
      allocParaBufferSize += DMAX(1024,newParaBufferSize) ; 
      newBuffer = new char[allocParaBufferSize] ;
      
      memcpy(newBuffer,paraBuffer,paraBufferSize);
      
      delete [] paraBuffer ;
      paraBuffer = newBuffer ;
   } 

   memcpy(paraBuffer+paraBufferSize,chars,charLen) ;
   paraBufferSize = newParaBufferSize ;

   if(paraBufferSize > 1000000) 
   {
       MX_ERROR_THROW(err,MX_HELP_BUFFER_TOO_SMALL);
   }

   goto exit;
  abort:
  exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::addToHidden
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_file::addToHidden(int  &err,
                               char *chars,
                               int  charLen)

{
   char *newBuffer ;
   int newHiddenBufferSize ;

   err = MX_ERROR_OK ;
   
   processingHidden = TRUE ;

   newHiddenBufferSize = hiddenBufferSize + charLen ;

   if(newHiddenBufferSize > allocHiddenBufferSize) 
   {
      allocHiddenBufferSize += DMAX(1024,newHiddenBufferSize) ; 
      newBuffer = new char[allocHiddenBufferSize] ;
      
      memcpy(newBuffer,hiddenBuffer,hiddenBufferSize);
      
      delete [] hiddenBuffer ;
      hiddenBuffer = newBuffer ;
   } 

   memcpy(hiddenBuffer+hiddenBufferSize,chars,charLen) ;
   hiddenBufferSize = newHiddenBufferSize ;

   if(hiddenBufferSize > 1000000) 
   {
       MX_ERROR_THROW(err,MX_HELP_BUFFER_TOO_SMALL);
   }

   goto exit;
  abort:
  exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_file::getLine
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_line *mx_help_file::getLine(int &err,
                                    int iline)
{
    err = MX_ERROR_OK ;
    
    if( (iline < 0) || (iline >= nlines)) 
    {
        MX_ERROR_THROW(err,1) ;
    }
    
    return lines[iline];

abort:
    return NULL ;
    
}






















