/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE : maxwell.C
 *
 * AUTHOR : Andrew Haisley
 *
 * This file 
 *
 * DESCRIPTION: 
 *
 *
 *
 */

#include <signal.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <X11/Intrinsic.h>
#include <locale.h>

#include <mx.h>
#include <mx_db.h>
//#include <mx_lm.h>

#include <maxwell.h>
#include <mx_font.h>

#include <mx_config.h>
#include <mx_app.h>
//#include <mx_install.h>
#include <mx_path_exp.h>
#include <new.h>
#include <mx_support_d.h>

// the one and only app
static mx_app *app = NULL;

// the name of the file used to lock the display sort of
static char lock_file_name[MAX_PATH_LEN];


Widget       global_top_level;
XtAppContext global_app_context;
mx_config    *global_user_config;
char         *global_maxhome;

/*
 * PROCEDURE : error_exit
 * ACTION    : called on an error which forces maxwell to exit. tries to
 *           : clean up neatly.
 * ARGS      : none
 *
 */
static void error_exit()
{
    int err = MX_ERROR_OK;

    if (app != NULL)
    {
        app->stop(err);
        app->unlock();
    }

    unlink(lock_file_name);
    exit(1);
}


/*
 * PROCEDURE : mx_handle_warn
 * ACTION    : handles warning generated by the Xt stuff
 * ARGS      : a string with a description in it
 *
 */
static void mx_handle_warn(String m)
{
#ifndef BUILD_RELEASE
    fprintf(stderr, "WARNING:%s\n", m);
#endif
}

/*
 * PROCEDURE : mx_handle_error
 * ACTION    : handles Xt errors
 *
 */
static void mx_handle_error(String m)
{
#ifndef BUILD_RELEASE
    fprintf(stderr, "ERROR:%s\n", m);
#endif
}

/*
 * PROCEDURE : child_sigproc
 * ACTION    : repsonds to SIG_CHLD - only action is to stop zombie
 *           : processes from building up in a long run
 * ARGS      : none
 *
 */
static int child_sigproc()
{
    int s;

    // get rid of zombies
    (void)wait(&s);

    signal(SIGCLD, (mx_sighandler_t)child_sigproc);
    return 0;
}


/*
 * PROCEDURE : segv_sigproc
 * ACTION    : prints an apology for the shiteness of our code before dying
 * ARGS      : none
 *
 */
static int segv_sigproc()
{
    fprintf(stderr, "ERROR:sorry, maxwell has crashed\n");
    error_exit();
    return 0;
}


/*
 * PROCEDURE : get_display_name
 * ACTION    : figures out the X display name
 * ARGS      : command line args used to figure the display.
 *
 */
static void get_display_name(int argc, char *argv[], char *dn)
{
    char hn[100];
    int  i;

    if (gethostname(hn, 99) < 0)
    {
        strcpy(hn, "localhost");
    }
    else
    {
        hn[99] = 0;
    }

    strcpy(dn, "");

    // is display in the arglist
    for (i = 1; i < argc; i++)
    {
        if (strcmp(argv[i], "-display") == 0)
        {
            if (i != (argc - 1))
            {
                strcpy(dn, argv[i + 1]);
                break;
            }
        }
    }

    if (strlen(dn) == 0)
    {
        // next, is DISPLAY in the environment ?
        if (getenv("DISPLAY") != NULL)
        {
            strcpy(dn, getenv("DISPLAY"));
        }
        else
        {
            // no use hostname:0.0
            sprintf(dn, "%s:0.0", hn);
        }
    }
}

/*
 * PROCEDURE : ensure_unique
 * ACTION    : attemps to make sure this is the only copy of Maxwell running
 *           : on this server. A nice job for someone would be to re-write this
 *           : to use its own X protocol to communicate between clients....on
 *           : the other hand, this one doens't need to open the display to do
 *           : its work.....on the other hand XOpenDisplay isn;t exactly hard..
 * ARGS      : command line args used to figure the display.
 *
 */
static void ensure_unique(int argc, char *argv[])
{
    /* don't bother with this stuff for now - it's causing more problems than
       its worth

    char  dn[100];
    char  hn[100];
    FILE  *f;
    char  line[100];
    pid_t pid;
    int   i;

    get_display_name(argc, argv, dn);

    if (gethostname(hn, 99) < 0)
    {
        strcpy(hn, "localhost");
    }
    else
    {
        hn[99] = 0;
    }

    sprintf(lock_file_name, "/tmp/mx-%s-%s", hn, dn);

    // is there another copy of maxwell running on this host with the 
    // same display ?
    if ((f = fopen(lock_file_name, "r")) != NULL)
    {
        // maybe
        fgets(line, 99, f);
        line[99] = 0;

        pid = atoi(line);
        if (pid != 0)
        {
            if (kill(pid, 0) != -1)
            {
                // it's still alive
                fprintf(stderr, "WARNING:maxwell already active on this display\n");
                exit(1);
            }
        }
        fclose(f);
    }

    // no - we're the only one
    if ((f = fopen(lock_file_name, "w")) == NULL)
    {
        // can't write to /tmp
        fprintf(stderr, "ERROR:unable to write to /tmp (%s)\n", sys_errlist[errno]);
        error_exit();
    }

#ifdef sun
    fprintf(f, "%ld", getpid());
#else
    fprintf(f, "%d", getpid());
#endif
    fclose(f);
    */
}

/*
 * PROCEDURE : check_display
 * ACTION    : makes sure that the display can be opened. Also, prints a simple
 *           : message if it it can't.
 * ARGS      : command line args used to figure the display.
 *
 */
static void check_display(int argc, char *argv[])
{
    char dn[100];
    char hn[100];
    Display *d;
    int i;

    get_display_name(argc, argv, dn);

    d = XOpenDisplay(dn);
    if (d == NULL)
    {
        printf( "ERROR:Maxwell was unable to open the display %s.\n", dn);
        error_exit();
    }
    else
    {
        XCloseDisplay(d);
    }
}


/*
 * PROCEDURE : merge_resources
 * ACTION    : merges in the maxwell resource file
 *
 */
void merge_resources()
{
    XrmDatabase db = NULL ;
    char file_name[MAX_PATH_LEN];

    db = XtDatabase(XtDisplay(global_top_level)) ;
    if(db == NULL) 
    {
        fprintf(stderr, "ERROR:couldn't get resource DB\n");
        error_exit();
    }

    sprintf(file_name, "%s/resource/maxwell.rsc", global_maxhome);

    XrmCombineFileDatabase(file_name, &db, True);
}


/*
 * PROCEDURE : set_font_path
 * ACTION    : makes sure that the maxwell type 1 directory is on the font path
 *           : (it may already be there)
 *
 */
static void set_font_path()
{
    /* don't bother doing this for now, since it stops people using maxwell on
       a remote display.  Maybe this should be configurable at compile time, so
       it can be enforced on users?

    char **dirs;
    int n, i;
    char *new_dirs[20];
    char mfd[MAX_PATH_LEN];

    dirs = XGetFontPath(XtDisplay(global_top_level), &n);

    sprintf(mfd, "%s/fonts/Type1", global_maxhome);

    for (i = 0; i < n; i++)
    {
        if (strcmp(dirs[i], mfd) == 0)
        {
            XFreeFontPath(dirs);
            return;
        }
    }

    for (i = 0; i < n; i++)
    {
        new_dirs[i] = dirs[i];
    }
    new_dirs[i] = mfd;
    
    XSetFontPath(XtDisplay(global_top_level), new_dirs, n + 1);
    XFreeFontPath(dirs);
    */
}

/*
 * PROCEDURE : start_lm
 * ACTION    : starts a new lock manager if one isn't already running
 * ARGS      : none
 *
 */
/*
static void start_lm()
{
    int  err = MX_ERROR_OK;
    char temp[MAX_PATH_LEN];

    // try to connect to a lock server
    mx_lm_client_connect(err);
    if (err != MX_ERROR_OK)
    {
        switch (err)
        {
            case MX_LM_STANDALONE : 
                    break;
            case MX_LM_CLIENT_TIMEOUT :
                    // try to start a lock manager
                    struct stat fs;

                    sprintf(temp, "%s/bin/mxlm", global_maxhome);
                    if (stat(temp, &fs) == 0)
                    {
                        if (vfork() == 0)
                        {
                            execl(temp, "mxlm", "-n", NULL);
                            error_exit();
                        }
                        else
                        {
                            sleep(10);
                            mx_lm_client_connect(err);
                            if (err != MX_ERROR_OK)
                            {
                                fprintf(stderr, "WARNING:Couldn't start a lock manager\n");
                            }
                        }
                    }
                    else
                    {
                        fprintf(stderr, "WARNING:Couldn't start a lock manager\n");
                    }
                    break;
            default :
            case MX_LM_SYSTEM_ERROR :
            case MX_LM_CORRUPT_PACKET :
                    break;
        }
        MX_ERROR_CLEAR(err);
    }
}
*/

/*
 * PROCEDURE : open_config
 * ACTION    : opens the users config file - if there isn't one, create one from the
 *           : system default config
 * ARGS      : error 
 *
 */
static void open_config(int &err)
{
    char temp[MAX_PATH_LEN];
    char temp2[MAX_PATH_LEN];
    uid_t uid;

    struct passwd *pw;

    uid = getuid();
    pw = getpwuid(uid);

    sprintf(temp, "%s/%s", pw->pw_dir, ".maxwellrc");
    global_user_config = new mx_config(err, temp);
    if (err != MX_ERROR_OK)
    {
        sprintf(temp2, "%s/%s", global_maxhome, ".maxwellrc");
        mx_copy_file(err, temp2, temp);
        chown(temp, getuid(), getgid());
        chmod(temp, S_IRUSR | S_IWUSR);

        if (err != MX_ERROR_OK)
        {
            fprintf(stderr, "ERROR:Couldn't create user defaults file %s\n", temp);
            MX_ERROR_CHECK(err);
        }
        else
        {
            global_user_config = new mx_config(err, temp);
            MX_ERROR_CLEAR(err);
        }
    }
abort:;
}

/*
 * PROCEDURE : open_config
 * ACTION    : adds the maxwell bitmaps directory into the X server bitmap search path
 *
 */
static void set_bitmap_path()
{
    static char temp[MAX_PATH_LEN];
    char *lp;

    // set the bitmap path
    lp = getenv("XBMLANGPATH");
#ifdef sun
    if (lp == NULL)
    {
        sprintf(temp, "XBMLANGPATH=%s/bitmaps/%%B", global_maxhome);
    }
    else
    {    
        sprintf(temp, "XBMLANGPATH=%s/bitmaps/%%B:%s", global_maxhome, lp);
    }
    putenv(temp);
#else
    if (lp == NULL)
    {
        sprintf(temp, "%s/bitmaps/%%B", global_maxhome);
    }
    else
    {    
        sprintf(temp, "%s/bitmaps/%%B:%s", global_maxhome, lp);
    }
    setenv("XBMLANGPATH", temp, TRUE);
    unsetenv("XAPPLRESDIR");
#endif
}

/*
 * PROCEDURE : open_wp_file
 * ACTION    : attempts to open a named document file. prints errors if it can't
 * RETURNS   : NULL on failure, a document pointer otherwise
 *
 */
static mx_wp_document *open_wp_file(char *name)
{
    char *locked_host, *locked_by;
    pid_t locked_pid;

    mx_wp_document *res;

    int err = MX_ERROR_OK;

    // try to open it - set recover flag, so never fail for that reason
    res = mx_db_client_open_wp_doc(err, name, TRUE, &locked_host, &locked_by, locked_pid);
    if (err != MX_ERROR_OK)
    {
        if (err == MX_LM_ALREADY_LOCKED)
        {
            printf("WARNING:sorry, can't open file %s as it is locked by user %s on host %s\n",
                        name, locked_host, locked_by);
            return NULL;
        }

        if (err == MX_FILE_UNRECOVERABLE)
        {
            printf("WARNING:sorry, file %s is damaged beyond recovery\n", name);
            MX_ERROR_CLEAR(err);

            return NULL;
        }

        else
        {
            MX_ERROR_CLEAR(err);
            return NULL;
        }
    }
    else
    {
        return res;
    }
}

static void handle_args(int argc, char *argv[])
{
    int err = MX_ERROR_OK;
    mx_wp_document *doc;

    // were there any files on the command line? Locking is handled a bit
    // differently here as the windows haven't been realised yet.....
    if (argc > 1)
    {
        for (int i = 1; i < argc; i++)
        {
            if (mx_is_file_type(argv[i], mx_document_file_e))
            {
                if (!app->is_doc_open(argv[i]))
                {
                    doc = open_wp_file(argv[i]);
                    if (doc != NULL)
                    {
                        app->new_wp_editor(err, doc);
                        MX_ERROR_CHECK(err);
                    }
                }
            }
            else
            {
                printf("WARNING:sorry, can't open non-Maxwell files from command line\n");
            }
        }
    } 
    return;
abort:
    MX_ERROR_CLEAR(err);
}


/*
 * PROCEDURE : set_signals
 * ACTION    : sets up signal handlers for various things
 *
 */
static void set_signals()
{
    signal(SIGCLD, (mx_sighandler_t)child_sigproc);
    signal(SIGCLD, SIG_IGN);

#ifdef BUILD_RELEASE
    signal(SIGINT, SIG_IGN);
    signal(SIGQUIT, SIG_IGN);
    signal(SIGTERM, SIG_IGN);
#endif
    signal(SIGSEGV, (mx_sighandler_t)segv_sigproc);
    signal(SIGFPE, (mx_sighandler_t)segv_sigproc);
    signal(SIGBUS, (mx_sighandler_t)segv_sigproc);
}

//
// standard gcc out of memory handler. 
//
void out_of_memory()
{
    printf("ERROR:sorry, Maxwell ran out of memory\n");
    exit(-1);
}

// the maxwell top level. Creates the main window with start
// buttons for each variety of maxwell app. currently just the
// WP app of course.
int main(int argc, char *argv[])
{
    int    err = MX_ERROR_OK;
    uid_t uid;

    struct passwd *pw;

    setlocale(LC_ALL, "");
    set_new_handler(out_of_memory);

    strcpy(lock_file_name, "");

    XtAppSetWarningHandler(global_app_context, mx_handle_warn);
    XtAppSetErrorHandler(global_app_context, mx_handle_error);

    set_signals();

    // make sure the thing's been installled correctly
    // (void)mx_install_sn();

    // make sure some basic services are working
    uid = getuid();
    pw = getpwuid(uid);
    if (pw == NULL)
    {
        fprintf(stderr, "ERROR:couldn't get user ID\n");
        error_exit();
    }

    global_maxhome = getenv("MAXHOME");

    if (global_maxhome == NULL)
    {
        fprintf(stderr, "WARNING:MAXHOME environment variable unset, defaulting to /usr/local/maxwell\n");
        global_maxhome = strdup("/usr/local/maxwell");
    }
    else
    {
        global_maxhome = strdup(global_maxhome);
    }

    // start_lm();

    open_config(err);
    MX_ERROR_CHECK(err);

    ensure_unique(argc, argv);

    mx_db_client_login(err);
    MX_ERROR_CHECK(err);

    // get the bitmaps into the path
    set_bitmap_path();

    check_display(argc, argv);

    global_top_level = XtVaAppInitialize(&global_app_context, "Maxwell",
                 NULL, 0, &argc, argv,
                 NULL, NULL);
    
    // set up the X resources
    merge_resources();

#ifdef BUILD_RELEASE
    global_error_trace->init_error_d(global_top_level);
#endif
    global_error_trace->init_exit_proc(error_exit);

    // set the font path if necessary
    set_font_path();

    // get the app ready to open files and stuff
    app = mx_app::init();

    handle_args(argc, argv);
    MX_ERROR_CHECK(err);

    // instantiate a font to force the font files to be read
    {
        mx_font f;
        (void)f.get_ps_fontname();
    }

    {
        // print up a copyright dialog
        char copyright_file_name[MAX_PATH_LEN];
        mx_support_d d(app->get_widget());

        sprintf(copyright_file_name, "%s/text/copyrght.txt", global_maxhome);
        d.run(copyright_file_name);
    }

    // off it goes..........
    app->run(err);
    MX_ERROR_CHECK(err);

    app->stop(err);
    MX_ERROR_CHECK(err);

    return 0;

abort:
    global_error_trace->print();
    return -1;
}
