/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <stdio.h>
#include <mx.h>
#include <mx_config.h>
#include <mx_list.h>
#include <mx_path_exp.h>
#include <mx_file.h>

void insert_test(int &err)
{
    mx_file *f;
    int i, n;

    unlink("grot.doc");
    f = new mx_file;

    printf("Number to insert:");
    scanf("%d", &n);

    f->open(err, "grot.doc", TRUE);
    MX_ERROR_CHECK(err);

    f->start_transaction(err);
    MX_ERROR_CHECK(err);

    printf("START\n");

    /* insert n objects */
    for (i = 0; i < n; i++)
    {
        f->add_object(err, i);
        MX_ERROR_CHECK(err);
    }

    f->commit(err);
    MX_ERROR_CHECK(err);

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    printf("CLOSE FILE\n");
    f->close(err);
    MX_ERROR_CHECK(err);

    printf("END\n");

abort:;
}

void update_test(int &err)
{
    mx_file *f;
    int n;

    f = new mx_file;

    f->open(err, "grot.doc", TRUE);
    MX_ERROR_CHECK(err);

    f->check_tree(err);
    MX_ERROR_CHECK(err);

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    printf("Object number to update:");
    scanf("%d", &n);

    f->start_transaction(err);
    MX_ERROR_CHECK(err);

    f->add_object(err, n);
    MX_ERROR_CHECK(err);

    f->commit(err);
    MX_ERROR_CHECK(err);

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    f->close(err);
    MX_ERROR_CHECK(err);

abort:;
}

void single_delete_test(int &err)
{
    mx_file *f;
    int n;

    f = new mx_file;

    f->open(err, "grot.doc", TRUE);
    MX_ERROR_CHECK(err);

    f->check_tree(err);
    MX_ERROR_CHECK(err);

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    printf("Object number to delete:");
    scanf("%d", &n);

    f->start_transaction(err);
    MX_ERROR_CHECK(err);

    f->del_object(err, n);
    MX_ERROR_CHECK(err);

    f->commit(err);
    MX_ERROR_CHECK(err);

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    f->close(err);
    MX_ERROR_CHECK(err);

abort:;
}


void add_objects(int &err, mx_file *f, int n)
{
    int i; 
    for (i = 0; i < n; i++)
    {
        f->add_object(err, i);
        MX_ERROR_CHECK(err);
    }

abort:;
}

void blob_test(int &err)
{
    mx_file *f;
    int i, j, n;
    char buf[MX_FILE_BLOCK_SIZE];

    unlink("grot.doc");

    f = new mx_file;

    printf("Number of bytes:");
    scanf("%d", &n);

    f->open(err, "grot.doc", TRUE);
    MX_ERROR_CHECK(err);

    f->start_transaction(err);
    MX_ERROR_CHECK(err);

    /* add an object */
    f->add_object(err, 100);
    MX_ERROR_CHECK(err);

    /* add a blob */
    f->create_blob(err, 100, "DATA", n);
    MX_ERROR_CHECK(err);

    /* set the data */
    for (i = 0; i <= (n / MX_FILE_BLOCK_SIZE); i++)
    {
        printf("SETTING SEG %d\n", i);

        for (j = 0; j < MX_FILE_BLOCK_SIZE; j++)
        {
            buf[j] = (j + i) % 256;
        }
        f->set_blob_data(err, 100, "DATA", buf, i);
        MX_ERROR_CHECK(err);
    }

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    /* check the data */
    for (i = 0; i < (n / MX_FILE_BLOCK_SIZE); i++)
    {
        f->get_blob_data(err, 100, "DATA", buf, i);
        MX_ERROR_CHECK(err);

        for (j = 0; j < MX_FILE_BLOCK_SIZE; j++)
        {
            if (buf[j] != ((j + i) % 256))
            {
                printf("ERROR segment %d byte %d is wrong\n", i, j);
                break;
            }
        }
    }

    f->commit(err);
    MX_ERROR_CHECK(err);

    f->close(err);
    MX_ERROR_CHECK(err);

abort:;
}
void delete_test(int &err)
{
    mx_file *f;
    int i, j, n;

    unlink("grot.doc");

    f = new mx_file;

    printf("Number to delete:");
    scanf("%d", &n);

    f->open(err, "grot.doc", TRUE);
    MX_ERROR_CHECK(err);

    f->start_transaction(err);
    MX_ERROR_CHECK(err);

    /* add some objects */
    add_objects(err, f, n);
    MX_ERROR_CHECK(err);

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    /* now, delete them */
    for (i = n - 1; i >= 0; i--)
    {
        f->del_object(err, i);
        MX_ERROR_CHECK(err);

        f->check_tree(err);
        MX_ERROR_CHECK(err);

        for (j = 0; j < i; j++)
        {
            if (!f->object_exists(err, j))
            {
                printf("ERROR OBJECT %d NOT THERE\n", j);
            }
            MX_ERROR_CHECK(err);
        }
    }

    printf("COMMIT\n");

    f->commit(err);
    MX_ERROR_CHECK(err);

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    f->close(err);
    MX_ERROR_CHECK(err);

    return;

abort:;
}

void vector_test(int &err)
{
    mx_file *f;
    mx_attribute       a("TEST", mx_int_array);
    const mx_attribute *b;
    mx_attribute_value v;
    int i, n, m;
    int32 ia[10];

    unlink("grot.doc");

    f = new mx_file;

    f->open(err, "grot.doc", TRUE);
    MX_ERROR_CHECK(err);

    f->start_transaction(err);
    MX_ERROR_CHECK(err);

    printf("how many :");
    scanf("%d", &m);

    for (n = 1; n <= m; n++)
    {
        f->add_object(err, n);
        MX_ERROR_CHECK(err);

        v.ia = ia;
        for (i = 0; i < 10; i++)
        {
            ia[i] = i;
        }
        a.set_value(err, v, mx_int_array, 10);
        MX_ERROR_CHECK(err);

        f->write_attribute(err, n, &a);
        MX_ERROR_CHECK(err);
    }

    f->print_tree(err);
    MX_ERROR_CHECK(err);

    f->commit(err);
    MX_ERROR_CHECK(err);

    for (n = 1; n <= m; n++)
    {
        b = f->read_attribute(err, n, "TEST");
        MX_ERROR_CHECK(err);

        if (b->type != mx_int_array)
        {
            printf("ERROR, DIDN'T GET BACK AN INTEGER ARRAY\n");
            break;
        }
        else
        {
            if (b->length != 10)
            {
                printf("ERROR GOT BACK LENGTH OF %ld\n", b->length);
                break;
            }
            else
            {
                for (i = 0; i < 10; i++)
                {
                    if (b->value.ia[i] != i)
                    {
                        printf("ERROR ENTRY %d HAS VALUE %ld\n", i, b->value.ia[i]);
                        return;
                    }
                }
            }
        }
    }

abort:;
}

void allocation_test(int &err)
{
    mx_file *f;

    int i, j, bn[MX_FILE_BLOCK_SIZE], offset[MX_FILE_BLOCK_SIZE];
    unsigned char data[MX_FILE_BLOCK_SIZE];

    unlink("grot.doc");

    f = new mx_file;

    f->open(err, "alloctest.doc", TRUE);
    MX_ERROR_CHECK(err);

    f->start_transaction(err);
    MX_ERROR_CHECK(err);

    for (i = 32; i < 100; i++)
    {
        f->block_cache->allocate_data(err, i, bn[i], offset[i]);
        MX_ERROR_CHECK(err);
        printf("ALLOCATE %d %d %d\n", i, bn[i], offset[i]);

        for (j = 0; j < i; j++)
        {
            data[j] = j;
        }
        f->block_cache->write_data(err, (char *)data, i, bn[i], offset[i]);
    }

    for (i = 32; i < 100; i++)
    {
        printf("READ %d %d %d\n", i, bn[i], offset[i]);

        f->block_cache->read_data(err, data, i, bn[i], offset[i]);
        MX_ERROR_CHECK(err);

        for (j = 0; j < i; j++)
        {
            if (data[j] != j)
            {
                printf("DATA ERROR %d %d\n", i, j);
            }
        }
    }

    f->commit(err);
    MX_ERROR_CHECK(err);

    f->close(err);
    MX_ERROR_CHECK(err);

abort:;
}

void thrash_test()
{
    mx_file *f;
    int i, j, n, inta[1000], stra[1000], s1;
    int err = MX_ERROR_OK;
    mx_attribute a;
    char s[100];
    char blobs[10][102400], buffer[4096];
    int sizes[10];

    unlink("grot.doc");

    for (i = 0; i < 10; i++)
    {
        memset(blobs[i], 0, 102400);
        sizes[i] = 0;
    }

    f = new mx_file;

    f->open(err, "grot.doc", TRUE);
    MX_ERROR_CHECK(err);

    f->start_transaction(err);
    MX_ERROR_CHECK(err);

    srand(100);
    for (i = 0; i < 1000; i++)
    {
        inta[i] = -1;
        stra[i] = -1;
    }

    while (TRUE)
    {
        switch (rand() % 5)
        {
            case 0 :
                // delete normal attribute
                n = rand() % 1000;
                if (inta[n] != -1)
                {
                    f->delete_attribute(err, n + 1, "fred");
                    MX_ERROR_CHECK(err);
                    inta[n] = -1;
                }
                break;
            case 1 :
                // blob data
                // first, set size
                n = rand() % 10;

                sizes[n] = (rand() % 102300) + 50;
//printf("1:sizes[n]=%d\n", sizes[n]);
                f->create_blob(err, n + 1, "blobby", sizes[n]);
                MX_ERROR_CLEAR(err);

//printf("2:sizes[n]=%d\n", sizes[n]);

if ((n + 1) == 7) {
printf("-----------------------------------------before set blob size\n");
                f->dump_tree(err);
                MX_ERROR_CHECK(err);
}

                f->set_blob_size(err, n + 1, "blobby", sizes[n]);
                MX_ERROR_CHECK(err);
//printf("3:sizes[n]=%d\n", sizes[n]);
if ((n + 1) == 7) {
printf("-----------------------------------------after set blob size\n");
                f->dump_tree(err);
                MX_ERROR_CHECK(err);
}

                s1 = (rand() % sizes[n]) / 4096;
                memset(blobs[n] + (s1 * 4096), (rand() % 200) + 1, 4096);

                f->set_blob_data(err, n + 1, "blobby", blobs[n] + s1 * 4096,
                    s1);
                MX_ERROR_CHECK(err);

                break;
            case 2 :
                // normal attribute
                strcpy(a.name,"fred");
                a.type = mx_int;
                n = rand() % 1000;
                a.value.i = inta[n] = rand();

                f->write_attribute(err, n + 1, &a);
                MX_ERROR_CHECK(err);

                break;
            case 3 :
                // vector attribute
                strcpy(a.name,"zog");
                a.type = mx_string;
                n = rand() % 1000;
                stra[n] = rand();
                sprintf(s, "this is %d string", stra[n]);
                a.length = strlen(s);
                a.value.s = s;

                f->write_attribute(err, n + 1, &a);
                MX_ERROR_CHECK(err);

                break;
            case 4 :
                // delete vector attribute
                n = rand() % 1000;
                if (stra[n] != -1)
                {
                    f->delete_attribute(err, n + 1, "zog");
                    MX_ERROR_CHECK(err);
                    stra[n] = -1;
                }
                break;
        }
        for (i = 0; i < 1000; i++)
        {
            const mx_attribute *x;

            if (inta[i] != -1)
            {
                x = f->read_attribute(err, i + 1, "fred");
                MX_ERROR_CHECK(err);

                if (x->type != mx_int)
                {
                    printf("int %d type wrong\n", i);
                    return;
                }
                if (x->value.i != inta[i])
                {
                    printf("int %d value wrong (%d,%ld)\n", i, inta[i], x->value.i);
                    return;
                }
            }
            if (stra[i] != -1)
            {
                x = f->read_attribute(err, i + 1, "zog");
                MX_ERROR_CHECK(err);

                if (x->type != mx_string)
                {
                    printf("string %d type wrong\n", i);
                    return;
                }
                sprintf(s, "this is %d string", stra[i]);
                if (strcmp(x->value.s, s) != 0)
                {
                    printf("string %d value wrong (%s,%s)\n", i, 
                        s, x->value.s);
                    return;
                }
            }
            else
            {
                x = f->read_attribute(err, i + 1, "zog");
                if (err == MX_ERROR_OK)
                {
                    printf("string attribute %d is there when it shouldn't be\n", i);
                    return;
                }
                else
                {
                    MX_ERROR_CLEAR(err);
                }
            }
        }
        for (i = 0; i < 10; i++)
        {
            // check size
            if (sizes[i] != 0)
            {
                s1 = f->get_blob_size(err, i + 1, "blobby");
                MX_ERROR_CHECK(err);

                if (s1 != sizes[i])
                {
                    printf("blob %d size wrong - should be %d, was %d\n", i,
                        sizes[i], s1);
                    return;
                }

                for (s1 = 0; s1 <= (sizes[i] / 4096); s1++)
                {
                    if (blobs[i][s1 * 4096] != 0)
                    {
                        f->get_blob_data(err, i + 1, "blobby", buffer, s1);
                        MX_ERROR_CHECK(err);

                        for (j = 0; j < 4096; j++)
                        {
                            if (buffer[j] != blobs[i][s1 * 4096 + j])
                            {
                                printf("blob %d byte %d wrong, was %d should be %d\n",
                                    i, j, buffer[j], blobs[i][s1 * 4096 + j]);
                                return;
                            }
                        }
                    }
                }
            }
        }

        if ((rand() % 50) == 30)
        {
        printf("commit & re-open\n");
        f->commit(err);
        MX_ERROR_CHECK(err);

        f->close(err);
        MX_ERROR_CHECK(err);

        f->open(err, "grot.doc", TRUE);
        MX_ERROR_CHECK(err);

        f->start_transaction(err);
        MX_ERROR_CHECK(err);
        printf("done\n");
        }

    }
    return;
abort:
    global_error_trace->print();
}

main( int argc, char *argv[] )
{
    int i, err = MX_ERROR_OK;
    mx_file *f;
    
    printf("sizeof(file_header_t)=%d sizeof(entry_t)=%d\n",
        sizeof(file_header_t), sizeof(entry_t));
    printf("sizeof(area_size_t) = %d\n", sizeof(area_size_t));
    printf("sizeof(entry_t) = %d\n", sizeof(entry_t));
    while (1) 
    {
        printf("insert test.............1\n");
        printf("delete test.............2\n");
        printf("vector attributes.......3\n");
        printf("allocation test.........4\n");
        printf("blob test...............5\n");
        printf("update test.............6\n");
        printf("single delete test......7\n");
        printf("break file..............8\n");
        printf("recover file............9\n");
        printf("thrash file.............10\n");

        printf("which>");
        scanf("%d", &i);

        switch( i ) {
            case 1 : insert_test(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 2 : delete_test(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 3 : vector_test(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 4 : allocation_test(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 5 : blob_test(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 6 : update_test(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 7 : single_delete_test(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 8 : f = new mx_file;
                     f->open(err, "grot.doc", TRUE);
                     MX_ERROR_CHECK(err);
                     f->start_transaction(err);
                     MX_ERROR_CHECK(err);
                     exit(1);
            case 9 : f = new mx_file;
                     f->open(err, "grot.doc", TRUE);
                     MX_ERROR_CHECK(err);
            case 10 : thrash_test();
        }
    }
abort:
    global_error_trace->print();
}
