/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_block
 *
 * AUTHOR : Andrew Haisley
 *
 * 
 *
 * DESCRIPTION:
 * block cache 
 *
 *
 *
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <netinet/in.h>
#include <mx.h>
#include <mx_attribute.h>
#include <mx_block.h>

#define MX_FILE_SIGNATURE       0x5432abcd
#define MX_FREE_BLOCK_SIGNATURE 0x123214af
#define MX_FIRST_BLOCK_OFFSET   1024

#define ROOT_TO_USE_OFFSET           12
#define DIRTY_OFFSET                 16
#define ROOT_ARRAY_OFFSET            20
#define FIRST_FREE_DATA_OFFSET       8
#define FIRST_FREE_DATA_INDEX_OFFSET 12

typedef struct
{
    area_size_t as;
    int32 next_bn;
    int32 next_offset;
    int32 prev_bn;
    int32 prev_offset;
}
free_data_header_t;

mx_block::mx_block(int &err, int fd)
{
    int i;

    mx_block::fd = fd;

    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        cache[i] = new unsigned char[MX_FILE_BLOCK_SIZE];
		memset(cache[i], 0, MX_FILE_BLOCK_SIZE);
        numbers[i] = -1;
        touch[i] = 0;
        dirty[i] = FALSE;
        locked[i] = FALSE;
    }

    next_touch = 0;
    memset(dummy, 0, MX_FILE_BLOCK_SIZE);

    // is it an empty file ?
    if ((i = lseek(fd, 0, SEEK_END)) ==  0)
    {
        size_in_blocks = 0;
        // yes
        ::write(fd, &dummy, MX_FIRST_BLOCK_OFFSET);

		memset(&header, 0, sizeof(header)) ;

        header.sig = MX_FILE_SIGNATURE;
        header.first_free = -1;
        header.first_free_data = -1;
        header.first_free_data_index = 0;
        header.root_to_use = 0;
        header.dirty_flag = FALSE;
        header.next_id = 2;
        header.root_block[0] = header.root_block[1] = 0;
        header.num_spare_blocks = 0;
        

        write_header(err);
        MX_ERROR_CHECK(err);

        err = MX_ERROR_OK;
        return;
    }
    else
    {
        lseek(fd, 0, SEEK_SET);

        if (::read(fd, &header, sizeof(file_header_t)) != sizeof(file_header_t))
        {
            err = MX_BLOCK_NOT_MX_FILE;
            return;
        }

        header.sig = ntohl(header.sig);
        header.first_free = ntohl(header.first_free);
        header.first_free_data = ntohl(header.first_free_data);
        header.first_free_data_index = ntohl(header.first_free_data_index);
        header.root_to_use = ntohl(header.root_to_use);
        header.dirty_flag = ntohl(header.dirty_flag);
        header.root_block[0] = ntohl(header.root_block[0]);
        header.root_block[1] = ntohl(header.root_block[1]);
        header.next_id = ntohl(header.next_id);
        header.num_spare_blocks = ntohl(header.num_spare_blocks);

        if (header.sig != MX_FILE_SIGNATURE)
        {
            err = MX_BLOCK_NOT_MX_FILE;
            return;
        }

        size_in_blocks = (i - MX_FIRST_BLOCK_OFFSET) / MX_FILE_BLOCK_SIZE;
        err = MX_ERROR_OK;
        return;
    }
abort:;
}

mx_block::~mx_block()
{
    int i;

    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        delete [] cache[i];
    }
}

void mx_block::seek_to_block(int &err, int bn)
{
    if (lseek(fd, (bn * MX_FILE_BLOCK_SIZE) + MX_FIRST_BLOCK_OFFSET, SEEK_SET) == -1)
    {
        err = translate_error(errno);
    }
    else
    {
        err = MX_ERROR_OK;
    }
}

int mx_block::get_free_cache_position(int &err)
{
    int i, lru_index, touch_num;

    // is there a free one ?
    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        if (numbers[i] == -1)
        {
            break;
        }
    }

    // did we find one ?
    if (i >= MX_BLOCK_CACHE_SIZE)
    {
        /* no, find the lru block */
        lru_index = -1;
        touch_num = 0x7fffffff;

        for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
        {
            if (locked[i])
            {
                continue;
            }
            if (touch[i] < touch_num)
            {
                lru_index = i;
                touch_num = touch[i];
            }
        }

        i = lru_index;
        if (i > MX_BLOCK_CACHE_SIZE)
        {
            err = MX_BLOCK_OUT_OF_BLOCKS;
            return -1;
        }
        if (dirty[i])
        {
            write(err, numbers[i]);
            MX_ERROR_CHECK(err);
        }
    }
    return i;
abort:
    return -1;
}

const unsigned char *mx_block::read(int &err, int bn, bool is_blob = FALSE)
{
    int i;

    if (bn >= size_in_blocks)
    {
        err = MX_BLOCK_NO_SUCH_BLOCK;
        return NULL;
    }

    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        if (numbers[i] == bn)
        {
            mx_block::is_blob[i] = is_blob;
            touch[i] = next_touch++;
            return cache[i];
        }
    }


    i = get_free_cache_position(err);
    MX_ERROR_CHECK(err);

    touch[i] = next_touch++;
    numbers[i] = bn;
    dirty[i] = FALSE;
    mx_block::is_blob[i] = is_blob;

    seek_to_block(err, bn);
    MX_ERROR_CHECK(err);

    if (::read(fd, cache[i], MX_FILE_BLOCK_SIZE) != MX_FILE_BLOCK_SIZE)
    {
        err = translate_error(errno);
        return NULL;
    }
    else
    {
        block_t *b = (block_t *)cache[i];

        if (!is_blob)
        {
            // translate data into something understandable
            network_block_to_host(b);
        }

        err = MX_ERROR_OK;
        return cache[i];
    }
abort:
    return NULL;
}

void mx_block::write(int &err, int bn)
{
    int i;
    block_t *b;
    unsigned char temp[MX_FILE_BLOCK_SIZE];

    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        if (numbers[i] == bn)
        {
            break;
        }
    }

	memset(temp, 0, MX_FILE_BLOCK_SIZE);

    seek_to_block(err, bn);
    MX_ERROR_CHECK(err);

    touch[i] = next_touch++;
    dirty[i] = FALSE;

    if (is_blob[i])
    {
        b = (block_t *)cache[i];
    }
    else
    {
        memcpy(temp, cache[i], MX_FILE_BLOCK_SIZE);

        b = (block_t *)temp;
        host_block_to_network(b);
    }

    if (::write(fd, b, MX_FILE_BLOCK_SIZE) == -1)
    {
        err = translate_error(errno);
    }
    else
    {
        err = MX_ERROR_OK;
    }

abort:;
}

int mx_block::allocate(int &err, bool reserve = TRUE)
{
    int i, allocated;

    if (header.num_spare_blocks > 0)
    {
        header.num_spare_blocks--;
    }

    if (header.first_free != -1)
    {
        allocated = header.first_free;

        // down the free chain
        seek_to_block(err, header.first_free);
        MX_ERROR_CHECK(err);

        if (::read(fd, &header.first_free, 4) != 4)
        {
            err = translate_error(errno);
            return -1;
        }

        header.first_free = ntohl(header.first_free);

        if (reserve)
        {
            i = get_free_cache_position(err);
            MX_ERROR_CHECK(err);

            touch[i] = next_touch++;
            numbers[i] = allocated;
            dirty[i] = FALSE;
            memset(cache[i], 0, MX_FILE_BLOCK_SIZE);
        }

        return allocated;
    }

    if (reserve)
    {
        i = get_free_cache_position(err);
        MX_ERROR_CHECK(err);

        touch[i] = next_touch++;
        numbers[i] = size_in_blocks;
        dirty[i] = FALSE;
        memset(cache[i], 0, MX_FILE_BLOCK_SIZE);
    }

    seek_to_block(err, size_in_blocks);
    MX_ERROR_CHECK(err);
    
    static char temp[MX_FILE_BLOCK_SIZE];

    if (::write(fd, &temp, MX_FILE_BLOCK_SIZE) == -1)
    {
        err = translate_error(errno);
        return -1;
    }

    return size_in_blocks++;

abort:;
    return -1;
}

void mx_block::free(int &err, int bn)
{
    int i;
    uint32 temp;

    header.num_spare_blocks++;

    seek_to_block(err, bn);
    MX_ERROR_CHECK(err);

    temp = htonl(header.first_free);

    if (::write(fd, &temp, 4) == -1)
    {
        err = translate_error(errno);
        return;
    }

    // dump it from the cache
    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        if (numbers[i] == bn)
        {
            numbers[i] = -1;
            break;
        }
    }

    header.first_free = bn;

abort:;
}

void mx_block::reset_free()
{
    header.first_free = -1;
}

int mx_block::num_blocks()
{
    return size_in_blocks;
}

void mx_block::set_dirty(int &err, int bn)
{
    int i;

    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        if (numbers[i] == bn)
        {
            dirty[i] = TRUE;
            err = MX_ERROR_OK;
            return;
        }
    }
    err = MX_BLOCK_NO_SUCH_BLOCK;
    return;
}

// set one of the root block pointers
void mx_block::set_root(int &err, int root_to_set, int bn)
{
    header.root_block[root_to_set] = bn;

    write_header(err);
    MX_ERROR_CHECK(err);
abort:;
}

// get one of the root block pointers
int mx_block::get_root(int &err, int root_to_get)
{
    err = MX_ERROR_OK;
    return header.root_block[root_to_get];
}

// set or clear dirty flag
void mx_block::set_dirty_flag(int &err, bool dirty)
{
    header.dirty_flag = dirty;
    write_header(err);
    MX_ERROR_CHECK(err);
abort:;
}

// set root_to_use
void mx_block::set_root_to_use(int &err, int root_to_use)
{
    header.root_to_use = root_to_use;
    write_header(err);
    MX_ERROR_CHECK(err);
abort:;
}

// get root_to_use
int mx_block::get_root_to_use(int &err)
{
    err = MX_ERROR_OK;
    return header.root_to_use;
}

// get dirty flag
bool mx_block::get_dirty_flag(int &err)
{
    err = MX_ERROR_OK;
    return header.dirty_flag;
}

void mx_block::flush(int &err)
{
    int i;

    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        if ((numbers[i] != -1) && dirty[i])
        {
            write(err, numbers[i]);
            MX_ERROR_CHECK(err);
        }

        locked[i] = FALSE;
        dirty[i] = FALSE;
        numbers[i] = -1;
    }

    write_header(err);
    MX_ERROR_CHECK(err);

abort:;
}

// allocate from fresh block
void mx_block::allocate_from_new_block(int &err, int size, int &bn, int &offset)
{
    int  i;
    char *data;

    area_size_t as;

    // allocate new block
    bn = allocate(err);
    MX_ERROR_CHECK(err);

    data = (char *)read(err, bn);
    MX_ERROR_CHECK(err);

    set_dirty(err, bn);
    MX_ERROR_CHECK(err);

    as.size = size;
    as.type = MX_BLOCK_USED_TYPE;
    memcpy(data, &as, 4);

    // set the size marker for the rest of the block
    i = MX_FILE_BLOCK_SIZE - size - 8;

    as.type = MX_BLOCK_FREE_TYPE;
    as.size = i;
    memcpy(data + size + 4, &as, 4);

    offset = 4;
        
    // add the remaining bit of the block to the free chain
    free_data(err, bn, size + 8);
    MX_ERROR_CHECK(err);

    err = MX_ERROR_OK;

abort:;
}

// allocate from the data chain. size <= BLOCK_SIZE
void mx_block::allocate_data(int &err, int size, int &bn, int &offset)
{
    char *data;
    int i;
    area_size_t as;
    
    if (size < 32)
    {
        size = 32;
    }

    if (header.first_free_data == -1)
    {
        allocate_from_new_block(err, size, bn, offset);
        MX_ERROR_CHECK(err);

        return;
    }
    else
    {
        // go down the free chain and look for space
        int search_block;
        int search_index;

        free_data_header_t h;

        search_block = header.first_free_data;
        search_index = header.first_free_data_index;

        while (TRUE)
        {
            data = (char *)read(err, search_block);
            MX_ERROR_CHECK(err);

            memcpy(&h, data + search_index, sizeof(free_data_header_t));

            if (h.as.size < size)
            {
                // too small
                search_block = h.next_bn;
                search_index = h.next_offset;

                if (search_block == -1)
                {
                    // off the end
                    allocate_from_new_block(err, size, bn, offset);
                    MX_ERROR_CHECK(err);
    
                    return;
                }
            }
            else
            {
                set_dirty(err, search_block);
                MX_ERROR_CHECK(err);

                // need to split ?
                if (size >= (h.as.size / 2) || (h.as.size < 64))
                {
                    // greater than half or the available chunk is
                    // smaller than 64 -> keep it together
                    remove_from_free_chain(err, search_block, search_index);
                    MX_ERROR_CHECK(err);
                }
                else
                {
                    // split it up
                    remove_from_free_chain(err, search_block, search_index);
                    MX_ERROR_CHECK(err);

                    as.size = size;
                    as.type = MX_BLOCK_USED_TYPE;
                    memcpy(data + search_index , &as, 4);

                    i = h.as.size - size - 4;
                    as.size = i;
                    as.type = MX_BLOCK_USED_TYPE;
                    memcpy(data + search_index + size + 4, &as, 4);
                    free_data(err, search_block, search_index + size + 8);
                    MX_ERROR_CHECK(err);
                }

                bn = search_block;
                offset = search_index + 4;
                return;
            }
        }
    }

abort:;
}

// write data 
void mx_block::write_data(int &err, char *data, int size, int bn, int offset)
{
    char *block_data;

    block_data = (char *)read(err, bn);
    MX_ERROR_CHECK(err);

    set_dirty(err, bn);
    MX_ERROR_CHECK(err);
    
    memcpy(block_data + offset, data, size);
abort:;
}

// free data - put pack in free chain
void mx_block::free_data(int &err, int bn, int offset)
{
    free_data_header_t h, h1;

    char *data;

    data = (char *)read(err, bn);
    MX_ERROR_CHECK(err);

    set_dirty(err, bn);
    MX_ERROR_CHECK(err);

    memcpy(&(h.as), data + offset - 4, 4);
    h.as.type = MX_BLOCK_FREE_TYPE;

    h.next_bn = header.first_free_data;
    h.next_offset = header.first_free_data_index;
    h.prev_bn = -1;     // points to start of chain
    h.prev_offset = -1; 
    memcpy(data + offset - 4, &h, sizeof(free_data_header_t));

    // now, go to the next free item and set its prev pointers
    if (h.next_bn != -1)
    {
        data = (char *)read(err, h.next_bn);

        set_dirty(err, h.next_bn);
        MX_ERROR_CHECK(err);

        memcpy(&h1, data + h.next_offset, sizeof(free_data_header_t));
        h1.prev_bn = bn;
        h1.prev_offset = offset - 4;

        memcpy(data + h.next_offset, &h1, sizeof(free_data_header_t));
    }

    // relink the header
    header.first_free_data = bn;
    header.first_free_data_index = offset - 4;

abort:;
}

void mx_block::remove_from_free_chain(int &err, int bn, int offset)
{
    free_data_header_t h, prev_h, next_h;

    char *data, *prev_data, *next_data;

    data = (char *)read(err, bn);
    MX_ERROR_CHECK(err);

    set_dirty(err, bn);
    MX_ERROR_CHECK(err);

    memcpy(&h, data + offset, sizeof(free_data_header_t));
    h.as.type = MX_BLOCK_USED_TYPE;
    memcpy(data + offset, &h, sizeof(free_data_header_t));

    // relink next block
    if (h.next_bn != -1)
    {
        next_data = (char *)read(err, h.next_bn);
        MX_ERROR_CHECK(err);

        set_dirty(err, h.next_bn);
        MX_ERROR_CHECK(err);

        memcpy(&next_h, next_data + h.next_offset, sizeof(free_data_header_t));
        next_h.prev_bn = h.prev_bn;
        next_h.prev_offset = h.prev_offset;

        memcpy(next_data + h.next_offset, &next_h, sizeof(free_data_header_t));
    }

    // relink previous
    if (h.prev_bn == -1)
    {
        // header pointer, not block
        header.first_free_data = h.next_bn;
        header.first_free_data_index = h.next_offset;
    }
    else
    {
        prev_data = (char *)read(err, h.prev_bn);
        MX_ERROR_CHECK(err);

        set_dirty(err, h.prev_bn);
        MX_ERROR_CHECK(err);

        memcpy(&prev_h, prev_data + h.prev_offset, sizeof(free_data_header_t));
        prev_h.next_bn = h.next_bn;
        prev_h.next_offset = h.next_offset;

        memcpy(prev_data + h.prev_offset, &prev_h, sizeof(free_data_header_t));
    }
abort:;
}

void mx_block::write_header(int &err)
{
    file_header_t temp_header;

    if (htonl(0x12345678) == 0x12345678 &&
        htons(0x1234) == 0x1234)
    {
        // network representation is same as host - do nothing
        memcpy(&temp_header, &header, sizeof(file_header_t));
    }
    else
    {
        temp_header.sig = htonl(header.sig);
        temp_header.first_free = htonl(header.first_free);
        temp_header.first_free_data = htonl(header.first_free_data);
        temp_header.first_free_data_index = htonl(header.first_free_data_index);
        temp_header.root_to_use = htonl(header.root_to_use);
        temp_header.dirty_flag = htonl(header.dirty_flag);
        temp_header.root_block[0] = htonl(header.root_block[0]);
        temp_header.root_block[1] = htonl(header.root_block[1]);
        temp_header.next_id = htonl(header.next_id);
        temp_header.num_spare_blocks = htonl(header.num_spare_blocks);
    }

    if (lseek(fd, 0, SEEK_SET) == -1)
    {
        err = translate_error(errno);
    }
    else
    {
        if (::write(fd, &temp_header, sizeof(file_header_t)) == -1)
        {
            err = translate_error(errno);
        }
        else
        {
            err = MX_ERROR_OK;
        }
    }
}

// read data 
void mx_block::read_data(int &err, unsigned char *data, int size, int bn, int offset)
{
    char *block_data;

    block_data = (char *)read(err, bn);
    MX_ERROR_CHECK(err);

    memcpy(data, block_data + offset, size);
abort:;
}

void mx_block::lock(int &err, int bn)
{
    int i;

    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        if (numbers[i] == bn)
        {
            locked[i] = TRUE;
            return;
        }
    }
    err = MX_BLOCK_NO_SUCH_BLOCK;
    return;
}

void mx_block::unlock(int &err, int bn)
{
    int i;

    for (i = 0; i < MX_BLOCK_CACHE_SIZE; i++)
    {
        if (numbers[i] == bn)
        {
            locked[i] = FALSE;
            return;
        }
    }
    err = MX_BLOCK_NO_SUCH_BLOCK;
    return;
}

uint32 mx_block::get_next_id()
{
    return header.next_id++;
}

void mx_block::reset_free_data()
{
    header.first_free_data = -1;
    header.first_free_data_index = 0;
}

void mx_block::set_info(int &err, char *description, char *author, char *version)
{
    strncpy(header.description, description, MX_BLOCK_DESCRIPTION_LENGTH);
    header.description[MX_BLOCK_DESCRIPTION_LENGTH - 1] = 0;
    strncpy(header.author, author, MX_BLOCK_AUTHOR_LENGTH);
    header.author[MX_BLOCK_AUTHOR_LENGTH - 1] = 0;
    strncpy(header.version, version, MX_BLOCK_AUTHOR_LENGTH);
    header.version[MX_BLOCK_AUTHOR_LENGTH - 1] = 0;

    write_header(err);
    MX_ERROR_CHECK(err);
abort:;
}

const char *mx_block::get_description()
{
    return header.description;
}

const char *mx_block::get_author()
{
    return header.author;
}

const char *mx_block::get_version()
{
    return header.version;
}

void mx_block::duplicate(int &err, int file)
{
    char buffer[1024];
    int i;

    if (lseek(fd, 0, SEEK_SET) == -1)
    {
        err = translate_error(errno);
        return;
    }

    if (lseek(file, 0, SEEK_SET) == -1)
    {
        err = translate_error(errno);
        return;
    }

    // copy the file contents
    do
    {
        i = ::read(fd, buffer, 1024);
        if (::write(file, buffer, i) != i)
        {
            err = translate_error(errno);
            return;
        }
    }
    while (i == 1024);

    fd = file;
    return;
}

int mx_block::translate_error( int e )
{
    return mx_translate_file_error(e);
}

void mx_block::host_block_to_network(block_t *b)
{
    uint32 temp;
    int i;
    unsigned char c;

    memcpy(&c, b, 1);

    if (c == MX_BLOCK_FREE_TYPE || c == MX_BLOCK_USED_TYPE)
    {
        // this is a data block - handle differently
        host_data_block_to_network(b);
        return;
    }
    
    b->header.magic = htonl(b->header.magic);
    b->header.next_modified_block = htonl(b->header.next_modified_block);

    b->header.ptr0.ptrA = htonl(b->header.ptr0.ptrA);
    b->header.ptr0.ptrB = htonl(b->header.ptr0.ptrB);

    for (i = 0; i < b->header.num_entries; i++)
    {
        b->entries[i].oid = htonl(b->entries[i].oid);
        b->entries[i].segment = htonl(b->entries[i].segment);
        b->entries[i].ptr.ptrA = htonl(b->entries[i].ptr.ptrA);
        b->entries[i].ptr.ptrB = htonl(b->entries[i].ptr.ptrB);

        switch (b->entries[i].type)
        {
            case value_int_e :
                    memcpy(&temp, &(b->entries[i].value.i), 4); 
                    temp = htonl(temp);
                    memcpy(&(b->entries[i].value.i), &temp, 4); 
                    break;
            case value_float_e :
                    memcpy(&temp, &(b->entries[i].value.f), 4); 
                    temp = htonl(temp);
                    memcpy(&(b->entries[i].value.f), &temp, 4); 
                    break;
            case value_id_e :
                    memcpy(&temp, &(b->entries[i].value.id), 4); 
                    temp = htonl(temp);
                    memcpy(&(b->entries[i].value.id), &temp, 4); 
                    break;

            case value_string_e :
            case value_char_array_e :
            case value_int_array_e :
            case value_id_array_e :
            case value_float_array_e :
            case value_blob_segment_e :

                    temp = b->entries[i].value.ptr.offset;
                    temp += b->entries[i].value.ptr.block * 0x1000;

                    temp = htonl(temp);
                    memcpy(&(b->entries[i].value.ptr), &temp, 4);

                    memcpy(&temp, &b->entries[i].value.ptr.length, 4);
                    temp = htonl(temp);
                    memcpy(&b->entries[i].value.ptr.length, &temp, 4);
                    break;

            case value_blob_e :
                    memcpy(&temp, &(b->entries[i].value.size), 4); 
                    temp = htonl(temp);
                    memcpy(&(b->entries[i].value.size), &temp, 4); 
                    break;

            default :
            case value_object_e :
                    break;
        }
    }
    b->header.num_entries = htons(b->header.num_entries);
}

void mx_block::network_block_to_host(block_t *b)
{
    uint32 temp;
    int i;
    unsigned char c;

    memcpy(&c, b, 1);

    if (c == MX_BLOCK_FREE_TYPE || c == MX_BLOCK_USED_TYPE)
    {
        // this is a data block - handle differently
        network_data_block_to_host(b);
        return;
    }

    b->header.magic = ntohl(b->header.magic);
    b->header.next_modified_block = ntohl(b->header.next_modified_block); 

    b->header.ptr0.ptrB = ntohl(b->header.ptr0.ptrB);
    b->header.ptr0.ptrA = ntohl(b->header.ptr0.ptrA);

    b->header.num_entries = ntohs(b->header.num_entries);

    for (i = 0; i < b->header.num_entries; i++)
    {
        b->entries[i].oid = ntohl(b->entries[i].oid);
        b->entries[i].segment = ntohl(b->entries[i].segment);
        b->entries[i].ptr.ptrA = ntohl(b->entries[i].ptr.ptrA);
        b->entries[i].ptr.ptrB = ntohl(b->entries[i].ptr.ptrB);

        switch (b->entries[i].type)
        {
            case value_int_e :
                    memcpy(&temp, &(b->entries[i].value.id), 4); 
                    temp = ntohl(temp);
                    memcpy(&(b->entries[i].value.id), &temp, 4); 
                    break;
            case value_float_e :
                    memcpy(&temp, &(b->entries[i].value.i), 4); 
                    temp = ntohl(temp);
                    memcpy(&(b->entries[i].value.i), &temp, 4); 
                    break;
            case value_id_e :
                    memcpy(&temp, &(b->entries[i].value.f), 4); 
                    temp = ntohl(temp);
                    memcpy(&(b->entries[i].value.f), &temp, 4); 
                    break;

            case value_string_e :
            case value_char_array_e :
            case value_int_array_e :
            case value_id_array_e :
            case value_float_array_e :
            case value_blob_segment_e :
                    memcpy(&temp, &(b->entries[i].value.ptr), 4);
                    temp = ntohl(temp);

                    b->entries[i].value.ptr.offset = (temp % 0x1000);
                    b->entries[i].value.ptr.block = (temp / 0x1000);

                    memcpy(&temp, &(b->entries[i].value.ptr.length), 4);
                    temp = ntohl(temp);
                    memcpy(&(b->entries[i].value.ptr.length), &temp, 4);
                    break;

            case value_blob_e :
                    memcpy(&temp, &(b->entries[i].value.size), 4); 
                    temp = ntohl(temp);
                    memcpy(&(b->entries[i].value.size), &temp, 4); 
                    break;

            default :
            case value_object_e :
                    break;
        }

    }
}

void mx_block::host_data_block_to_network(block_t *b)
{
    area_size_t as;
    free_data_header_t h;
    uint16 pos = 0, old_pos;

    memcpy(&as, b, sizeof(as));

    while (pos < MX_FILE_BLOCK_SIZE)
    {
        if (as.type == MX_BLOCK_FREE_TYPE)
        {
            memcpy(&h, (char *)b + pos, sizeof(h));

            h.next_bn = htonl(h.next_bn);
            h.next_offset = htonl(h.next_offset);
            h.prev_bn = htonl(h.prev_bn);
            h.prev_offset = htonl(h.prev_offset);

            memcpy((char *)b + pos, &h, sizeof(h));
        }

        old_pos = pos;
        pos += as.size + 4;
        as.size = htons(as.size);
        memcpy((char *)b + old_pos, &as, sizeof(as));

        if (pos + sizeof(as) < MX_FILE_BLOCK_SIZE)
        {
            memcpy(&as, (char *)b + pos, sizeof(as));
        }
    }
}

void mx_block::network_data_block_to_host(block_t *b)
{
    area_size_t as;
    free_data_header_t h;
    uint16 pos = 0;

    memcpy(&as, b, sizeof(as));

    while (pos < MX_FILE_BLOCK_SIZE)
    {
        if (as.type == MX_BLOCK_FREE_TYPE)
        {
            memcpy(&h, (char *)b + pos, sizeof(h));

            h.next_bn = ntohl(h.next_bn);
            h.next_offset = ntohl(h.next_offset);
            h.prev_bn = ntohl(h.prev_bn);
            h.prev_offset = ntohl(h.prev_offset);

            memcpy((char *)b + pos, &h, sizeof(h));
        }

        as.size = ntohs(as.size);
        memcpy((char *)b + pos, &as, sizeof(as));
        pos += as.size + 4;

        if (pos + sizeof(as) < MX_FILE_BLOCK_SIZE)
        {
            memcpy(&as, (char *)b + pos, sizeof(as));
        }
    }
}

int mx_block::get_num_spare_blocks()
{
    return header.num_spare_blocks;
}

void mx_block::extend(int &err, int nb)
{
    int n[100], i;

    if (nb > 100)
    {
        nb = 100;
    }

    for (i = 0; i < nb; i++)
    {
        n[i] = mx_block::allocate(err, FALSE);
        MX_ERROR_CHECK(err);
    }

    for (i = 0; i < nb; i++)
    {
        mx_block::free(err, n[i]);
        MX_ERROR_CHECK(err);
    }

abort:;
}
