/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : tree.h
 *
 * AUTHOR : David Miller
 *
 * 
 *
 * DESCRIPTION: A balanced binary tree using the red-black 
 *              algorithm
 * 
 * 
 * 
 *
 *
 */

#ifndef TREE_H
#define TREE_H

#include "node.h"


/*-------------------------------------------------
 * CLASS: tree
 *
 * DESCRIPTION : The tree
 * 
 *
 */


class tree
{
friend class collection ;
friend class tree_collection ;

private:
   node              *root ;
   bool              key_only ;
   bool              allow_multiple ;
   mx_attribute_type key_type ;
   mx_attribute_type data_type ;
   collection        *collect ;

public:

   // Creator 
   tree(bool              allowmultiple,
    mx_attribute_type datatype,
    mx_attribute_type keytype,
    collection        *col = NULL) ;
   
   // Destructor - make virtual to get rid of compilation warnings 
   virtual ~tree();

   // Check the red-blackness of the tree
   void check() ;
   void print() ;

   void check_node(node *test_node,int   black_depth) ;

   // Calls to use from the top level using keys  
                  
   // Add a new element to the tree                         
   void add( int                &err,
         mx_attribute_value &key_data,
         mx_attribute_value &data,
         int                &data_count) ;

   // Remove an element from the tree 
   void remove(int                &err,
           mx_attribute_value &key_data,
           int                &data_count,
           bool               force_delete = FALSE) ;
   
   // Get an element in the tree (dictionary type only)
   void get( int                &err,
         mx_attribute_value &key_data,
         mx_attribute_value &data,
         int                &data_count) ;

private:
   
   inline bool keyonly() ;

   void print_subtree(int level,node *n) ;

   // Calls used internally to handle nodes 

   // Rebalance a tree after deletion
   void rebalance(node *double_node,
          node *splice_node_parent) ; 

   // Get the node associated with a key value 

   node *get_node(int                &err,
          mx_attribute_value &key_data) ;

   inline void get_node_value(int                &err,
               node               *nn,
               mx_attribute_value &value) ;

   // Given a node find the next node in the tree
   node* next(node *node) ;

   // Given a node find the previous node in the tree
   node* prev(node *node) ;

   // Get the largest node in the tree relative to a node
   inline node* headnode(node *node) ;

   // Get the smallest  node in the tree relative to a node
   inline node* tailnode(node *node) ;

   // Get the largest node in the whole tree
   node* head() ;

   // Get the smallest  node in the whole tree 
   node* tail() ;

   // Delete a specific node from the tree
   void delete_node(int          &err,
            node         *node) ; 
   
   // Delete a subtree from a node
   void delete_subtree(int   &err,
               node  *node) ; 


   // Copy (memorywise) node from one node to another
   void copy_data(node *dst_node,
          node *src_node) ;

   // Insert a nodes data ino the tree
   void insert_node(int                 &err,
            node                *nn,
            mx_attribute_value  &key,
            mx_attribute_value  &data) ;   


   // Free data associated with a node (but not the node itself) 
   void free_node(int    &err,
          node   *node) ; 

// VIrtual calls which cope with complicated key and data types - the
// default will treat ints,reals,strings etc - need to define derived
// classes for other tree types and redefine these functions to cope
// with the complicated data type 

protected:


   // Get the data associated with a node
   virtual void get_data(int                 &err,
             mx_attribute_value  &node_data,
             mx_attribute_value  &data) ;

   // Get the key associated with a node
   virtual void get_key(int                &err,
            mx_attribute_value &node_key,
            mx_attribute_value &key) ;

   // Compare the key on a node 
   virtual int   comp_key(int                &err,
              mx_attribute_value &node_key,
              mx_attribute_value &key) ;

   // Compare the data on a node 
   virtual int   comp_data(int                &err,
               mx_attribute_value &node_data,
               mx_attribute_value &data) ;

   // Set the data on a node
   virtual void set_data(int                &err,
             mx_attribute_value &node_data,
             mx_attribute_value &data) ;

   // Set the key on a node
   virtual void set_key(int                &err,
            mx_attribute_value &node_key,
            mx_attribute_value &key) ;

   // Delete the key on a node
   virtual void delete_key(int                 &err,
               mx_attribute_value  &node_key) ;
   
   // Delete the data on a node
   virtual void delete_data(int                 &err,
                mx_attribute_value  &node_data) ;

} ;


/*-------------------------------------------------
 * FUNCTION: tree::get_node_value
 *
 * DESCRIPTION: Get the value of a node. The value comes
 * from the key for a set/bag but from the data for a
 * dictionary
 */

inline void tree::get_node_value(int                &err,
                 node               *nn,
                 mx_attribute_value &value) 
{
   if(key_only) 
   {
      get_key(err,nn->key,value) ;
   }
   else
   {
      get_data(err,nn->data,value) ;
   }
   MX_ERROR_CHECK(err) ;
  abort:
   return ;
}


/*-------------------------------------------------
 * FUNCTION: keyonly
 *
 * DESCRIPTION: Test if the tree is key only
 * 
 *
 */

inline bool  tree::keyonly()
{
   return key_only ;
}


/*-------------------------------------------------
 * FUNCTION: tree::insert_node
 *
 * DESCRIPTION: Insert a node values in the tree
 *
 */

inline void tree::insert_node(int                 &err,
                  node                *nn,
                  mx_attribute_value  &key_data,
                  mx_attribute_value  &data) 
{
   // Set the key and data fields 
   set_key(err,nn->key,key_data) ;
   MX_ERROR_CHECK(err) ; 
   
   if(!key_only)
   {
      set_data(err,nn->data,data) ;
      MX_ERROR_CHECK(err) ;
   }

  abort:
   return ;
}   


/*-------------------------------------------------
 * FUNCTION: tree::head
 *
 * DESCRIPTION: Get the maximum value in the tree
 *
 */

inline node* tree::head() 
{
   // Go from root  
  return  headnode(root) ;
}


/*-------------------------------------------------
 * FUNCTION: tree::headnode
 *
 * DESCRIPTION: Get the maximum in the tree relative 
 * to a node       
 *
 */

inline node* tree::headnode(node *node)
{
   if(node == NULL) return NULL ;

   while(node->right != NULL) node = node->right ;

   return node ;
}


/*-------------------------------------------------
 * FUNCTION: tree::tail
 *
 * DESCRIPTION: Get the mimimum value in the tree
 *
 */

inline node* tree::tail() 
{
   return tailnode(root) ;
}


/*-------------------------------------------------
 * FUNCTION: tree::headnode
 *
 * DESCRIPTION: Get the mimimum in the tree relative 
 * to a node       
 *
 */

inline node* tree::tailnode(node                *node) 
{
   if(node == NULL) return NULL ;

   while(node->left != NULL) node = node->left ; 

   return node ;
}


/*-------------------------------------------------
 * FUNCTION: tree::next
 *
 * DESCRIPTION: Get the next greatest value in the tree
 *              from node 
 */

inline node* tree::next(node                *testnode) 
{
   node *next_node ;

   if(testnode == NULL) return NULL ;

   if(testnode->right != NULL) 
   {
      return tailnode(testnode->right) ;
   }
   else
   {
      next_node = testnode->parent ;

      while( (next_node != NULL)  && (next_node->right == testnode))  
      {
     testnode = next_node ;
     next_node = next_node->parent ;
      }
      return next_node ;
   }
}


/*-------------------------------------------------
 * FUNCTION: tree::prev
 *
 * DESCRIPTION: Get the next smallest value in the tree
 *              from node 
 */

inline node* tree::prev(node                *testnode) 
{
   node *next_node ;

   if(testnode == NULL) return NULL ;

   if(testnode->left != NULL) 
   {
      return headnode(testnode->left) ;
   }
   else
   {
      next_node = testnode->parent ;

      while( (next_node != NULL)  && (next_node->left == testnode))  
      {
     testnode = next_node ;
     next_node = next_node->parent ;
      }
      return next_node ;
   }
}


/*-------------------------------------------------
 * FUNCTION: tree::copy_data
 *
 * DESCRIPTION: Copy data on a node
 * 
 *
 */

inline void tree::copy_data(node *dst_node,
                            node *src_node) 
{
   dst_node->count = src_node->count ;
   dst_node->data  = src_node->data ;
   dst_node->key   = src_node->key ;
}
 
#endif
