/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : tree.C
 *
 * AUTHOR : David Miller
 *
 * The author should insert the name of whoever should own the
 * copyright
 *
 * This file 
 *
 * DESCRIPTION:
 * Module tree.C
 * 
 *
 *
 */

#include "tree.h"
#include "mx_attribute.h"
#include "collection.h"


/*-------------------------------------------------
 * Function: tree::tree
 *
 * DESCRIPTION: Tree constructor
 * 
 *
 */


tree::tree(bool              allowmultiple,
           mx_attribute_type datatype,
       mx_attribute_type keytype,
       collection        *col) 
{
   root           = NULL ;
   key_only       = (datatype == mx_unknown) ;
   allow_multiple = allowmultiple ;
   data_type      = datatype ;
   key_type       = keytype ;
   collect        = col ;
} 

/*-------------------------------------------------
 * FUNCTION: tree::print
 *
 * DESCRIPTION: print tree 
 * 
 *
 */


void tree::print() 
{
   print_subtree(0,root) ;
}
 
/*-------------------------------------------------
 * FUNCTION: tree::printsubtree
 *
 * DESCRIPTION: print subtree 
 * 
 *
 */


void tree::print_subtree(int level,node *n) 
{
   int i ;

   if(n == NULL) return ;

   for(i=0;i<level;i++) printf("...") ;
   mx_attribute_print("Key",key_type,&n->key) ;

   if(!key_only) 
   {
      for(i=0;i<level;i++) printf("...") ;
      mx_attribute_print("Data",data_type,&n->data) ;
   }

   print_subtree(level+1,n->left) ;
   print_subtree(level+1,n->right) ;
} 

/*-------------------------------------------------
 * FUNCTION: tree::~tree
 *
 * DESCRIPTION: tree destructor
 * 
 *
 */

tree::~tree()
{
   int err = MX_ERROR_OK ;

   delete_subtree(err,root) ;

}

/*-------------------------------------------------
 * FUNCTION: tree::check
 *
 * DESCRIPTION: Check the red/blackness and depth 
 *              of the tree
 *
 */

void tree::check() 
{
   int   black_depth ;
   node  *test_node ;

   black_depth = 1 ;
   test_node = root ;
   
   while(test_node != NULL) 
   {
      if(test_node->colour == NODE_BLACK) 
      {
     black_depth++ ;
      }

      test_node = test_node->left ;
   }

   printf("Black depth = %d\n",black_depth) ;

   test_node = root ;
   check_node(test_node,1) ;
}


/*-------------------------------------------------
 * FUNCTION: tree::check_node
 *
 * DESCRIPTION: Check the red/blackness of a node
 * 
 *
 */

void tree::check_node(node *test_node,int   black_depth) 
{
   if(test_node == NULL) 
   {
      printf("Reached leaf - Black depth = %d\n",black_depth) ;
   }
   else
   {
      if(test_node->colour == NODE_RED) 
      {
     if(test_node->left != NULL)
     {
        if(test_node->left->colour != NODE_BLACK) 
        {
           printf("Red node has red left child\n") ;
        }
     }
     else if(test_node->right != NULL) 
     {
        if(test_node->right->colour != NODE_BLACK) 
        {
           printf("Red node has red right child\n") ;
        }
     }
      }
      else
      {
     black_depth++ ;
      }
      check_node(test_node->right,black_depth) ;
      check_node(test_node->left,black_depth) ;
   }
}


/*-------------------------------------------------
 * FUNCTION: tree::get
 *
 * DESCRIPTION: Get a node data value from the tree
 * 
 *
 */

void tree::get( int                &err,
        mx_attribute_value &key_data,
        mx_attribute_value &data,
        int                &data_count) 
{
   node *nn ;

   err = MX_ERROR_OK ;

   nn = get_node(err,key_data) ;
   MX_ERROR_CHECK(err) ;

   if(nn != NULL) 
   {
      data_count = nn->count ;
      get_node_value(err,nn,data) ;
      MX_ERROR_CHECK(err) ;
      return ;
   }
   else
   {
      data_count = 0 ;
   }
  abort:
   return ;
}


/*-------------------------------------------------
 * FUNCTION: tree::remove
 *
 * DESCRIPTION: Remove a node from the tree
 * 
 *
 */

void tree::remove(int                &err,
          mx_attribute_value &key_data,
          int                &data_count,
          bool               force_delete) 
{
   node *nn ;

   // First get the node associated with the key data 

   err = MX_ERROR_OK ;

   nn = get_node(err,key_data) ;
   MX_ERROR_CHECK(err) ;
     
   if(nn == NULL) 
   {
      data_count = 0 ;
   }
   else
   {
      data_count = nn->count ;

      if(force_delete || nn->count == 1 )
      {
     delete_node(err,nn) ;
     MX_ERROR_CHECK(err) ;
      }
      else
      {
     (nn->count)-- ;
      }
   }
  abort:
   return ;
}   


/*-------------------------------------------------
 * FUNCTION: tree::delete_subtree
 *
 * DESCRIPTION: Delete a subtree of the tree
 * 
 *
 */

void tree::delete_subtree(int &err, node *n)
{
   err = MX_ERROR_OK ;

   if(n == NULL) return ;

   delete_subtree(err,n->right) ;
   MX_ERROR_CHECK(err) ;

   delete_subtree(err,n->left) ;
   MX_ERROR_CHECK(err) ;

   free_node(err,n) ;
   MX_ERROR_CHECK(err) ;

   goto exit;
  abort:
  exit:
   return ;
}


/*-------------------------------------------------
 * FUNCTION: tree::delete_node
 *
 * DESCRIPTION: Delete a node value from the tree
 *  
 *
 */

void tree::delete_node(int              &err,
               node             *nn)
{
   node *splice_node,*child_splice_node ;

   err = MX_ERROR_OK ;

   // Node might be attached to an iterator 

   if(collect != NULL )
   {
      collect->checkIterators(err,nn) ;
      MX_ERROR_CHECK(err) ;
   }

   if( (nn->left == NULL) || (nn->right == NULL) )
   {
      splice_node = nn ;
   }
   else
   {
      /* more complicated case -  need to find next element - 
     as the node has two children the next element of the 
     tree can have at most one child */
      /* This is the which will be deleted - its data values 
     will be transferred to the node to be deleted */
      splice_node = next(nn) ;
   }
    
   /* Get the unique (or NULL) child of the splice node */
   child_splice_node = (splice_node->left != NULL) ? splice_node->left : splice_node->right ;

   /* Set the parent of the child of the node to be spliced out to be the parent 
      of the splice node */
   if(child_splice_node != NULL) child_splice_node->parent = splice_node->parent ;
     
   /* If the splice node is the root then the child becomes the root */
   if(splice_node->parent == NULL)
   {
      root = child_splice_node ;
   }
   else
   {
      /* Else set the parent of the splice node must point back to the 
     child of the splice node */
      
      if(splice_node == splice_node->parent->left) 
      {
     splice_node->parent->left = child_splice_node ;
      }
      else
      {
     splice_node->parent->right = child_splice_node ;
      }
   }
      
   /* now replace the node to be deleted with the splice node */

   if(splice_node != nn) 
   {
      node temp_node ;

      // Here we are swapping node - better ensure that iterators know 
      // about this 

      if(collect != NULL )
      {
     collect->checkIteratorsNodeMove(err,splice_node,nn) ;
     MX_ERROR_CHECK(err) ;
      }

      copy_data(&temp_node,nn) ;
      copy_data(nn,splice_node) ;
      copy_data(splice_node,&temp_node) ;

   }

   /* Now we have to rebalance the tree */

   if(splice_node->colour == NODE_BLACK) 
   {
      rebalance(child_splice_node,splice_node->parent) ;
   }

   free_node(err,splice_node) ;
   MX_ERROR_CHECK(err) ;

  abort:
   return ;
}


/*-------------------------------------------------
 * FUNCTION: tree::rebalance
 *
 * DESCRIPTION: Rebalance the tree after a deletion
 *
 */

void tree::rebalance(node             *double_node,
             node             *splice_node_parent) 
{
   node *parent_double_node,*sibling ;
   bool sibling_left_black,sibling_right_black ;

   
   /* loop while we haven't move up to the root */

   /* The idea is that the spliced node was black and the blackness has been 
      temporarily given to its child which is the double blacked node - 
      not that the child node may be NULL which is why we pass the parent of the
      splice node */

   while( (double_node != root) && ( (double_node == NULL) || (double_node->colour == NODE_BLACK)))
   {
      /* get the parent of the double node  */

      parent_double_node = (double_node == NULL) ? splice_node_parent : double_node->parent ;

      /* Look to left (or NULL) */

      if(parent_double_node->left == double_node) 
      {
     /* get the sibling of the doubly black node - it cant be NULL as the node
        is doubly black */

     sibling = parent_double_node->right ;
     
     if(sibling->colour == NODE_RED)   
     {
        /* Double node has a red sibling */
        /* Do a rotate so that the parent of the double node has two black children */

        sibling->colour = NODE_BLACK ;
        parent_double_node->colour = NODE_RED ;
        parent_double_node->left_rotate(&root) ;

        //  Get the new sibling 
        sibling = parent_double_node->right ;
     }

     /* At this stage the parent of the the double node is black - and 
        both of the children of the double node are black - as the double 
        node has double blackness its sibling cannot be NULL */

     sibling_left_black = (sibling->left == NULL)  || (sibling->left->colour == NODE_BLACK ) ;
     sibling_right_black = (sibling->right == NULL)  || (sibling->right->colour == NODE_BLACK ) ;
 
     if(sibling_left_black && sibling_right_black) 
     {
        /* Sibling has two black children */
        
        /* This is a simple case - make the sibling red and transfer the 
           extra black to the parent */

        sibling->colour =NODE_RED ;
        double_node = parent_double_node ;
     }
     else 
     {
        if(sibling_right_black) 
        {
           /* do a rotation so that the right sibling is red */
           sibling->colour = NODE_RED ;
           sibling->right_rotate(&root) ;
           sibling = parent_double_node->right ;
        }
        
        /* the doubly black node has a sibling with a red right child */
        
        /* we can change some colours and rotate so that some extra blackness 
           is transfered to the doubly black side */
        sibling->colour = parent_double_node->colour ;
        parent_double_node->colour = NODE_BLACK ;
        sibling->right->colour = NODE_BLACK ;
        parent_double_node->left_rotate(&root) ;
        double_node = root ;
     }
      }
      else
      {
     /* get the sibling of the doubly black node - it cant be NULL as the node
        is doubly black */

     sibling = parent_double_node->left ;
     
     if(sibling->colour == NODE_RED)   
     {
        /* Double node has a red sibling */
        /* Do a rotate so that the parent of the double node has two black children */

        sibling->colour = NODE_BLACK ;
        parent_double_node->colour = NODE_RED ;
        parent_double_node->right_rotate(&root) ;

        //  Get the new sibling 
        sibling = parent_double_node->left ;
     }

     /* At this stage the parent of the the double node is black - and 
        both of the children of the double node are black - as the double 
        node has double blackness its sibling cannot be NULL */

     sibling_left_black = (sibling->left == NULL)  || (sibling->left->colour == NODE_BLACK ) ;
     sibling_right_black = (sibling->right == NULL)  || (sibling->right->colour == NODE_BLACK ) ;
 
     if(sibling_left_black && sibling_right_black) 
     {
        /* Sibling has two black children */
        
        /* This is a simple case - make the sibling red and transfer the 
           extra black to the parent */

        sibling->colour =NODE_RED ;
        double_node = parent_double_node ;
     }
     else 
     {
        if(sibling_left_black) 
        {
           /* do a rotation so that the right sibling is red */
           sibling->colour = NODE_RED ;
           sibling->left_rotate(&root) ;
           sibling = parent_double_node->left ;
        }
        
        /* the doubly black node has a sibling with a red right child */
        
        /* we can change some colours and rotate so that some extra blackness 
           is transfered to the doubly black side */
        sibling->colour = parent_double_node->colour ;
        parent_double_node->colour = NODE_BLACK ;
        sibling->left->colour = NODE_BLACK ;
        parent_double_node->right_rotate(&root) ;
        double_node = root ;
     }
      } 
   }

   if(double_node != NULL) double_node->colour = NODE_BLACK ;
      
   return ;
}


/*-------------------------------------------------
 * FUNCTION: tree::add
 *
 * DESCRIPTION: Add a new element to a tree
 * 
 *
 */

void tree::add( int                &err,
        mx_attribute_value &key_data,
        mx_attribute_value &data,
        int                &data_count) 
{
   node  *ntn,*tn,*nn,*np,*npp,*nppd;
   int   test ;

   err = MX_ERROR_OK ;

   nn   = NULL ;

   /* Treat empty tree as special case */

   if(root == NULL) 
   {
      nn = root = new node() ;

      data_count = 0 ;

      insert_node(err,nn,key_data,data) ;
      MX_ERROR_CHECK(err) ;

      nn->colour = NODE_BLACK ;
      return ;
   }

   tn = root ;

   while(tn != NULL) 
   {
      test = comp_key(err,tn->key,key_data) ;
      MX_ERROR_CHECK(err) ;

      if(test == 0) 
      {
     /* test if data value is the same */

     if(!key_only) 
     {
        test = comp_data(err,tn->data,data) ;
        MX_ERROR_CHECK(err) ;
     }   
     else
     {
        test = 0 ;
     }

     /* Found the value - 
        increase count if a bag */
     /* If it is a dictionary and the key is the same
        but the data value different return -count */
        
     data_count = (test == 0) ? tn->count : -(tn->count) ;
     if(allow_multiple) (tn->count)++ ;
     tn = NULL ;
      }
      else if(test > 0) 
      {
     /* Value is bigger - look to right */
     
     ntn = tn->right ;
     if(ntn == NULL) 
     {
        /* Reached leaf */
        nn = new node(tn,TRUE) ;

        data_count = 0 ;
     }
     tn = ntn ;
      }
      else
      {
     /* Value is smaller - look to left */
     ntn = tn->left ;
     if(ntn == NULL) 
     {
        /* Reached leaf */
        nn = new node(tn,FALSE) ;

        data_count = 0 ;
     }
     tn = ntn ;
      }
   }

   /* Found existing node - no need to balance */
   if(nn == NULL) return ;

   insert_node(err,nn,key_data,data) ;
   MX_ERROR_CHECK(err) ;

   /* Now balance */
   
   /* The new node has been inserted as a red node -
      all rules for the red=black tree except possibly
      the rule that red nodes have black nodes as
      children */

   np = nn->parent ;

   /* At the beginning of the loop the 
      new test node its parent are in the
      variables nn and np. The test 
      node is always red. */

   /* stop if the root has been reached 
      or the parent of the test node is
      black (so red parent rule is obeyed) */

   while( (nn != root) && (np->colour == NODE_RED)) 
   {
      
      /* get the grandparent of the test node
    This must exist since the parent of the node
         has red colour (root is always black) 
    The grandparent is black */
 
      npp = np->parent ;

      if(np == npp->left) 
      {
     /* parent is left branch */
    
     nppd = npp->right ;
     if( (nppd != NULL) && (nppd->colour == NODE_RED)) 
     {
        /* The right branch of the grandparent is red */
        /* We can move two steps up the tree keeping
         the black count rule valid */
        np->colour   = NODE_BLACK ;
        nppd->colour = NODE_BLACK ;
        npp->colour  = NODE_RED ;
        nn           = npp ;
     }
     else
     {
        /* right branch of grandparent is black */
        /* We can do at most two rotates to
         get all red nodes with black children */

        /* If test node is a right branch rotate
          test node to the left side by moving the
          test node up the tree and then
          rotating about the new test node */

        if(nn == np->right) 
        {
           nn = np ;
           nn->left_rotate(&root) ;
           np = nn->parent ;
           npp = np->parent ;
        }

        /* Test node is to the left - set parent
          and grandparent colours and rotate about the
          grandparent - this gives a valid red-black tree */

        np->colour  = NODE_BLACK ;
        npp->colour = NODE_RED ;
        npp->right_rotate(&root) ;
     }
      }
      else
      {
     /* parent is right branch */
     nppd = npp->left ;
     if((nppd != NULL) && (nppd->colour == NODE_RED)) 
     {
        /* The left branch of the grandparent is red */
        /* We can move two steps up the tree keeping
         the black count rule valid */
        np->colour   = NODE_BLACK ;
        nppd->colour = NODE_BLACK ;
        npp->colour  = NODE_RED ;
        nn           = npp ;
     }
     else
     {
        /* left branch of grandparent is black */
        /* We can do at most two rotates to
         get all red nodes with black children */

        /* If test node is a left branch rotate
          test node to the right side by moving the
          test node up the tree and then
          rotating about the new test node */

        if(nn == np->left) 
        {
           nn = np ;
           nn->right_rotate(&root) ;
           np = nn->parent ;
           npp = np->parent ;
        }

        /* Test node is to the right - set parent
          and grandparent colours and rotate about the
          grandparent - this gives a valid red-black tree */

        np->colour  = NODE_BLACK ;
        npp->colour = NODE_RED ;
        npp->left_rotate(&root) ;
     }
      }
     
      /* Reset parent */
      np = nn->parent;
   }

   /* ensure root is black */

   root->colour = NODE_BLACK ;
  abort:
   return ;
}



/*-------------------------------------------------
 * FUNCTION: tree::get_node
 *
 * DESCRIPTION: Get a node from a tree
 * 
 *
 */

node* tree::get_node(int                &err,
             mx_attribute_value &key_data) 
{   
    int   test ;
    node  *nn ;

    err = MX_ERROR_OK ;

    nn         = root ;

    while(nn != NULL) 
    {
       test = comp_key(err,nn->key,key_data) ;
       MX_ERROR_CHECK(err) ;

       if(test == 0) 
       {
      /* Found the value - increase count */
      return nn ;
       }
       else if(test > 0) 
       {
      /* Value is bigger - look to right */
      
      nn = nn->right ;
       }
       else
       {
      /* Value is smaller - look to left */
      nn = nn->left ;
       }
    }
    goto exit ;
  abort:
    return NULL ;
  exit:
    return nn ;
}


/*-------------------------------------------------
 * FUNCTION: tree::free_node
 *
 * DESCRIPTION: Free the memory associated with a node
 *              including the node itself               
 *
 */

void tree::free_node(int               &err,
             node              *n) 
{
   // Delete the node as the default action
   

   err = MX_ERROR_OK ;

   if(!key_only) 
   {
      delete_data(err,n->data) ;
      MX_ERROR_CHECK(err) ;
   }

   delete_key(err,n->key) ;
   MX_ERROR_CHECK(err) ;

   delete n ;
  abort:
   return ;
}


/*-------------------------------------------------
 * FUNCTION: tree::comp_data
 *
 * DESCRIPTION: Compare the data on two nodes
 * 
 *
 */

int   tree::comp_data(int                 &err,
              mx_attribute_value  &node_data,
              mx_attribute_value  &new_data) 
{
   int ans ;

   err = MX_ERROR_OK ;

   if(collect != NULL) 
   {
      ans = collect->comp_data(err,node_data,new_data) ;
      MX_ERROR_CHECK(err) ;
   }
   else
   {
      ans =  mx_attribute_cmp(err,data_type,node_data,new_data) ;
      MX_ERROR_CHECK(err) ;
   }

   return ans ;

  abort:
   return 0 ;
      
}

/*-------------------------------------------------
 * FUNCTION: tree::comp_key
 *
 * DESCRIPTION: Compare the keys on two nodes
 * 
 *
 */

int   tree::comp_key(int                 &err,
             mx_attribute_value  &node_key,
             mx_attribute_value  &new_key) 
{
   int ans ;

   err = MX_ERROR_OK ;

   if(collect != NULL) 
   {
      ans = collect->comp_key(err,node_key,new_key) ;
      MX_ERROR_CHECK(err) ;
   }
   else
   {
      ans = mx_attribute_cmp(err,key_type,node_key,new_key) ;
      MX_ERROR_CHECK(err) ;
   }

   return ans ;

  abort:
   return 0 ;
}

/*-------------------------------------------------
 * FUNCTION: tree::delete_data
 *
 * DESCRIPTION: delete the data on a node
 * 
 *
 */

void tree::delete_data(int                 &err,
               mx_attribute_value  &node_data) 
{
   err = MX_ERROR_OK ;

   if(collect != NULL) 
   {
      collect->delete_data(err,node_data) ;
      MX_ERROR_CHECK(err) ;
   }
   else
   {
      mx_attribute_free(data_type,node_data) ;
   }

   return ;

  abort:
   return ;

}

/*-------------------------------------------------
 * FUNCTION: tree::delete_key
 *
 * DESCRIPTION: delete the key on a node
 * 
 *
 */

void tree::delete_key(int                 &err,
              mx_attribute_value  &node_key) 
{
   err = MX_ERROR_OK ;

   if(collect != NULL) 
   {
      collect->delete_key(err,node_key) ;
      MX_ERROR_CHECK(err) ;
   }
   else
   {
      mx_attribute_free(key_type,node_key) ;
   }

   return ;

  abort:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: tree::get_data
 *
 * DESCRIPTION: get the data on a node
 * 
 *
 */

void tree::get_data(int                 &err,
            mx_attribute_value  &node_data,
            mx_attribute_value  &new_data) 
{
   err = MX_ERROR_OK ;


   // Just do an attribute copy 

   if(collect != NULL) 
   {
      collect->get_data(err,node_data,new_data) ;
      MX_ERROR_CHECK(err) ;
   }
   else
   {
      new_data = node_data ;
   }

   goto exit;
  abort:
  exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: tree::get_key
 *
 * DESCRIPTION: get the key on a node
 * 
 *
 */

void tree::get_key(int                &err,
           mx_attribute_value &node_key, 
           mx_attribute_value &new_key) 
{
   err = MX_ERROR_OK ;

   if(collect != NULL) 
   {
      collect->get_key(err,node_key,new_key) ;
      MX_ERROR_CHECK(err) ;
   }
   else
   {
      // Just do an attribute copy 
      new_key = node_key ;
   }

   goto exit;
  abort:
  exit:
   return ;

}

/*-------------------------------------------------
 * FUNCTION: tree::set_data
 *
 * DESCRIPTION: set the data on a node
 * 
 *
 */

void tree::set_data(int                &err,
            mx_attribute_value &node_data, 
            mx_attribute_value &new_data) 
{
   err = MX_ERROR_OK ;

   if(collect != NULL) 
   {
      collect->set_data(err,node_data,new_data) ;
      MX_ERROR_CHECK(err) ;
   }
   else
   {
      mx_attribute_copy(err,data_type,1,node_data,new_data) ;
      MX_ERROR_CHECK(err) ;
   }

   goto exit;
  abort:
  exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: tree::set key
 *
 * DESCRIPTION: set the key on a node
 * 
 *
 */

void tree::set_key(int                 &err,
           mx_attribute_value  &node_key,
           mx_attribute_value  &new_key) 
{
   err = MX_ERROR_OK ;
   
   if(collect != NULL) 
   {
      collect->set_key(err,node_key,new_key) ;
      MX_ERROR_CHECK(err) ;
   }
   else
   {
      mx_attribute_copy(err,key_type,1,node_key,new_key);
      MX_ERROR_CHECK(err) ;
   }

   goto exit;
  abort:
  exit:
   return ;
}
