/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_list2
 *
 * AUTHOR : David Miller
 *
 * 
 *
 *
 * 
 */

#include <mx.h>
#include <mx_error.h>
#include <mx_list2.h>



//////////////////////////// Constructor
mx_list2::mx_list2()
{
    num_items = current_index = 0;
    first_list_elem = current_list_elem = last_list_elem = NULL;
}

//////////////////////////// Destructor
mx_list2::~mx_list2()
{
    delete_list(first_list_elem);
}

//////////////////////////// Copy Constructor
mx_list2::mx_list2(int &err,const mx_list2 &list)
{
  err = MX_ERROR_OK ;

  first_list_elem = NULL ;
  
  this->equate(err,list) ;
  MX_ERROR_CHECK(err) ;
  
  return;
abort:
  return ;
}

//////////////////////////// equals operator
mx_list2& mx_list2::equate(int &err,const mx_list2 &rvalue)
{
  err = MX_ERROR_OK ;
  
  delete_list(first_list_elem);
  
  num_items = current_index = 0;
  first_list_elem = current_list_elem = last_list_elem = NULL;
  
  first_list_elem = copy_list(err, rvalue.first_list_elem, (mx_list2 &)rvalue);
  MX_ERROR_CHECK(err);
  
  return *this;
abort:
  return *this;
}

// inserts an item before the item specified in the index.
void mx_list2::insert( int &err, int32 index, void * item)
{
    elem_t2 * rest_list = first_list_elem;
    elem_t2 * prev_list = NULL ;
    elem_t2 * new_list_elem;
    
    if((index == -1) || (index == num_items))
    {
        append(err, item);
        MX_ERROR_CHECK(err);
    } 
    else 
    {
        err = MX_ERROR_OK;
        new_list_elem = new elem_t2 ;

        range_check(err, index);
        MX_ERROR_CHECK(err);

        if(index == 0)
        {
            first_list_elem->prev = new_list_elem ;
            first_list_elem = new_list_elem;
            current_index++;
        } 
        else 
        {
            goto_index(err, index - 1);
            rest_list = current_list_elem->next;
                prev_list = current_list_elem;
            rest_list->prev = new_list_elem ;
            prev_list->next = new_list_elem;
        }

        new_list_elem->item = item;
        new_list_elem->next = rest_list;
        new_list_elem->prev = prev_list;
        num_items++;
    }
abort:
    return;
}

void mx_list2::append( int &err, void * item)
{
    elem_t2 * new_list_elem;
    
    err = MX_ERROR_OK;
    new_list_elem = new elem_t2 ;

    if(current_index == num_items)
    current_list_elem = new_list_elem;
 
    if(num_items == 0)
    {
        first_list_elem = new_list_elem;
    }
    else
    {
        last_list_elem->next = new_list_elem;
    }

    new_list_elem->item = item;
    new_list_elem->prev = last_list_elem ;
    new_list_elem->next = NULL;

    last_list_elem = new_list_elem;
    num_items++;
}

void * mx_list2::remove( int &err, int32 index )
{
    elem_t2 *item_to_delete = first_list_elem,
            *rest,   
            *tail = first_list_elem;

    void *removed_item;

    range_check(err, index);
    MX_ERROR_CHECK(err);

    if(index == 0)
    {
        item_to_delete = first_list_elem;
        first_list_elem = first_list_elem->next;
        if(current_index == 0)
        {
            current_list_elem = first_list_elem;
        }
        else
        {
            current_index--;
        }
        if(num_items == 1)
        {
            current_index = 0;
            first_list_elem = current_list_elem = last_list_elem = NULL;
        }
        else
        {
            first_list_elem->prev = NULL ;
        }
    } 
    else 
    {
        goto_index(err, index - 1);
        tail = current_list_elem;
        item_to_delete = tail->next;
        rest = item_to_delete->next;
        tail->next = rest;
        if(rest == NULL)
        {
            last_list_elem = tail;
        }
        else
        {
            rest->prev = tail ;
        }
    }

    removed_item = item_to_delete->item;
    item_to_delete->next = NULL;
    delete_list(item_to_delete);
 
    --num_items;
    return removed_item;
abort:
    return NULL;
}

void * mx_list2::overwrite(int &err, int32 index, void * item)
{
    void * overwritten_item;
    goto_index(err, index);
    MX_ERROR_CHECK(err);
    overwritten_item = current_list_elem->item;
    current_list_elem->item = item;

    return overwritten_item;
abort:
    return NULL;
}

void * mx_list2::get(int &err, int32 index)
{
    goto_index(err, index);
    MX_ERROR_CHECK(err);
    return current_list_elem->item;
abort:
    return NULL;
}

void * mx_list2::find_smallest(int &err, int (*comp_func)(void * one, 
                             void * two))
{
    int32 smallest_index = current_index = 0;
    elem_t2 * smallest_list_elem = current_list_elem = first_list_elem;

    range_check(err, smallest_index);
    if(num_items > 1)
    {
        while(current_list_elem->next != NULL)
        {
            if(comp_func(smallest_list_elem->item, 
                 current_list_elem->next->item) > 0)
            {
                smallest_list_elem = current_list_elem->next;
                smallest_index = current_index;
            }
            current_list_elem = current_list_elem->next;
            current_index++;
        }
    }
    current_list_elem = smallest_list_elem;
    current_index = smallest_index;
    return current_list_elem->item;
    return NULL;
}

static (*the_comp_func)(void * one, void * two);

static inline int comp_reverse( void * one, void * two)
{
    return the_comp_func(two, one);
}

void * mx_list2::find_largest(int &err,  int (*comp_func)(void * one, 
                             void * two))
{
    the_comp_func = comp_func;
    return find_smallest(err, &comp_reverse);
}

void mx_list2::iterate(int &err, void (*iter_func)( int &err, uint32 index, 
                           void * item ))
{
    elem_t2 * iter_list_elem, * next_iter_list_elem;
    int32    iter_index;

    iter_index = 0;
    iter_list_elem = first_list_elem;

    while(iter_list_elem != NULL)
    {
        next_iter_list_elem = iter_list_elem->next;
        iter_func(err, iter_index, iter_list_elem->item);
        MX_ERROR_CHECK(err);
        iter_list_elem = next_iter_list_elem;
        iter_index++;
    }
abort:
    return;
}

// These functions can be used to iterate through inserted objects
// you call iterate_start and then repeatedly call iterate_next.
// iterate_next returns NULL when you have completed the iteration.
 
void mx_list2::iterate_start()
{
    current_index = 0;
    current_list_elem = first_list_elem;
}

 
void mx_list2::iterate_end()
{
    current_index = num_items - 1 ;
    current_list_elem = last_list_elem;
}

void * mx_list2::iterate_prev()
{
    elem_t2 * temp_elem = current_list_elem;
    if(current_list_elem == NULL)
    {
        iterate_start();
        return NULL;
    } 
    else 
    {
        current_list_elem = current_list_elem->prev;
        current_index--;
        return temp_elem->item;
    }
}


void * mx_list2::iterate_next()
{
    elem_t2 * temp_elem = current_list_elem;
    if(current_list_elem == NULL)
    {
        iterate_end();
        return NULL;
    } 
    else 
    {
        current_list_elem = current_list_elem->next;
        current_index++;
        return temp_elem->item;
    }
}

int32 mx_list2::find_index(int &err, void * item)
{
    err = MX_ERROR_OK;
    current_index = 0;
    current_list_elem = first_list_elem;

    while(current_list_elem != NULL)
    {
        if(current_list_elem->item == item)
        {
            return current_index;
        }
        current_list_elem = current_list_elem->next;
        current_index++;
    }

    err = MX_LIST_NO_SUCH_POINTER;
    return 0;
}

void mx_list2::goto_index(int &err, int32 index)
{
    int32 i;

    range_check(err, index);
    MX_ERROR_CHECK(err);

    if((current_index > index) || (index == 0))
    {
        current_index = 0;
        current_list_elem = first_list_elem;
    }

    for(i = current_index; i < index; i++)
    {
        current_list_elem = current_list_elem->next;
    }
    current_index = index;

abort:
    return;
}

void mx_list2::range_check(int &err, int32 &index) const
{
    err = MX_ERROR_OK;

    if(num_items == 0)
    {
        err = MX_LIST_EMPTY;
        goto abort;
    }

    if(index == -1)
    index = num_items - 1;
    if((index < 0) || (index >= num_items))
    {
        err = MX_LIST_INDEX;
        goto abort;
    }
abort:
    return;
}

void mx_list2::delete_list(elem_t2 * list)
{
    if(list != NULL)
    {
        delete_list(list->next);
        delete list->next;
    }
}

mx_list2::elem_t2 * mx_list2::copy_list(int &err, elem_t2 * list, mx_list2 &other)
{
    elem_t2 * result;

    err = MX_ERROR_OK;
    if(list != NULL)
    {
        result = new elem_t2;

        result->prev = NULL ;

        result->item = list->item;
        result->next = copy_list(err, list->next, other);
        MX_ERROR_CHECK(err);

        if(result->next != NULL) result->next->prev = result ;
        {
            num_items++;
        }

        if(result->next == NULL)
        {
            last_list_elem = result;
        }

        if(other.current_list_elem == list)
        {
            current_index = other.current_index;
            current_list_elem = result;
        }
        return result;
    }
abort:
    return NULL;
}
