/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : collection.h
 *
 * AUTHOR : David Miller
 *
 * This file 
 *
 * DESCRIPTION:
 * Module collection.h
 * 
 * 
 * 
 *
 * $header$
 * $log$
 *
 */

#ifndef _COLLECTION_H
#define _COLLECTION_H

class collection ;

#include "tree.h"
#include "mx_iterator.h"

class collection
{
private:

// Functions to help iterators */

   friend class iterator ;
   friend class tree ;

   virtual node *getHead() = NULL ;
   virtual node *getTail() = NULL ;
   virtual node *getNext(node *testnode)  = NULL ;
   virtual node *getPrev(node *testnode)  = NULL ;
   virtual node *getPos(int pos)  = NULL ;
   virtual node *getValue(int &err,
              mx_attribute_value &data) = NULL  ;

   void checkIterators(int &err,
               node *n) ;

   void checkIteratorsNodeMove(int &err,
                   node *n,
                   node *newn) ;

   mx_attribute_type      key_type ;
   mx_attribute_type      data_type ;
   int                    niterators ;
   iterator               **iterators ;

   void addIterator(int &err,iterator *it) ;
   void removeIterator(iterator *it) ;

protected:
   int nelements ;

protected:


   // Get the data associated with a node
   virtual void get_data(int                 &err,
             mx_attribute_value  &node_data,
             mx_attribute_value  &data) ;

   // Get the key associated with a node
   virtual void get_key(int                &err,
            mx_attribute_value &node_key,
            mx_attribute_value &key) ;

   // Compare the key on a node 
   virtual int   comp_key(int                &err,
              mx_attribute_value &node_key,
              mx_attribute_value &key) ;

   // Compare the data on a node 
   virtual int   comp_data(int                &err,
               mx_attribute_value &node_data,
               mx_attribute_value &data) ;

   // Set the data on a node
   virtual void set_data(int                &err,
             mx_attribute_value &node_data,
             mx_attribute_value &data) ;

   // Set the key on a node
   virtual void set_key(int                &err,
            mx_attribute_value &node_key,
            mx_attribute_value &key) ;

   // Delete the key on a node
   virtual void delete_key(int                 &err,
               mx_attribute_value  &node_key) ;
   
   // Delete the data on a node
   virtual void delete_data(int                 &err,
                mx_attribute_value  &node_data) ;

public:
   collection(mx_attribute_type keytype=mx_unknown,
          mx_attribute_type datatype=mx_unknown) ;

   virtual ~collection() ;
   int count() ;
   mx_attribute_type keytype() ;
   mx_attribute_type datatype() ;

   virtual bool in(int &err,
           mx_attribute_value &data) = NULL ;
 
} ;  


/*-------------------------------------------------
 * FUNCTION: collection::count
 *
 * DESCRIPTION: Return the number of elements in a collection
 * 
 *
 */

inline int collection::count() 
{
   return nelements ;
}


/*-------------------------------------------------
 * FUNCTION: collection::keytype
 *
 * DESCRIPTION: Return the keytype of the collection
 * 
 *
 */

inline mx_attribute_type collection::keytype() 
{
   return key_type ;
}


/*-------------------------------------------------
 * FUNCTION: collection::datatype
 *
 * DESCRIPTION: Return the keytype of the collection
 * 
 *
 */

inline mx_attribute_type collection::datatype() 
{
   return data_type ;
}



/*-------------------------------------------------
 * CLASS: tree_collection
 *
 * DESCRIPTION: A collection based on a tree
 * 
 *
 */

class tree_collection:public collection 
{
private:
   node *getHead() ;
   node *getTail() ;
   node *getNext(node *testNode) ;
   node *getPrev(node *testNode) ;
   node *getPos(int pos) ;
   node *getValue(int &err,
          mx_attribute_value &data)  ;
protected:
   tree t ;
public:
   void check() ;

   tree_collection(mx_attribute_type keytype,
           bool              allowmultiple,
                   mx_attribute_type datatype=mx_unknown) ;

   bool in(int &err,mx_attribute_value &data) ;   
} ;


/*-------------------------------------------------
 * FUNCTION: tree_collection::check
 *
 * DESCRIPTION: Check a tree is balanced
 * 
 *
 */

inline void tree_collection::check() 
{
   t.check() ;
}


/*-------------------------------------------------
 * FUNCTION: getHead
 *
 * DESCRIPTION: Get the head of the tree
 * 
 *
 */

inline node *tree_collection::getHead()
{
   return t.head() ;

}

/*-------------------------------------------------
 * FUNCTION: getTail
 *
 * DESCRIPTION: Get the tail of the tree
 * 
 *
 */

inline node *tree_collection::getTail()
{
   return t.tail() ;

}

/*-------------------------------------------------
 * FUNCTION: getNext
 *
 * DESCRIPTION: Get the next node in the tree
 * 
 *
 */

inline node *tree_collection::getNext(node *testnode)
{
   return t.next(testnode) ;

}

/*-------------------------------------------------
 * FUNCTION: getPrev
 *
 * DESCRIPTION: Get the previous node in the tree
 * 
 *
 */

inline node *tree_collection::getPrev(node *testnode)
{
   return t.prev(testnode) ;
}


/*-------------------------------------------------
 * FUNCTION: in
 *
 * DESCRIPTION: Test if the key is in the collection
 * 
 *
 */

inline bool tree_collection::in(int                 &err,
                mx_attribute_value  &key) 
{
   node *testnode ;

   err = 0 ;

   testnode = t.get_node(err,key) ;
   MX_ERROR_CHECK(err) ;
   
   return (testnode != NULL) ;

  abort:
   return FALSE ;
}

/*-------------------------------------------------
 * FUNCTION: getValue
 *
 * DESCRIPTION: Get the node associated with a value 
 * 
 *
 */

inline node *tree_collection::getValue(int                 &err,
                       mx_attribute_value  &key) 
{
   node *testnode ;

   err = 0 ;

   testnode = t.get_node(err,key) ;
   MX_ERROR_CHECK(err) ;
   
   return testnode ;
  abort:
   return NULL ;
}

/*-------------------------------------------------
 * FUNCTION: getPos
 *
 * DESCRIPTION: Get the previous node in the tree
 * 
 *
 */

inline node *tree_collection::getPos(int pos) 
{
   int ipos ;

   node *test_node ;

   if(pos < 0) return NULL ;

   test_node = getTail() ;
   
   if(pos == 0) 
   {
      return test_node ;
   }
   else
   {
      for(ipos=0;ipos<pos;ipos++) 
      {
     test_node = getNext(test_node) ;
     if(test_node == NULL) return NULL ;
      }

      return test_node ;
   }
}

   


#endif

