/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : collection.C
 *
 * AUTHOR : David Miller
 *
 * This file 
 *
 * DESCRIPTION:
 * Module collection.C
 * 
 *
 *
 */


#include "mx_std.h"
#include "mx_attribute.h"
#include "collection.h"
#include <memory.h>


/*-------------------------------------------------
 * FUNCTION: collection::collection
 *
 * DESCRIPTION: collection initialisor
 * 
 *
 */

collection::collection(mx_attribute_type keytype,
               mx_attribute_type datatype) 
{
   nelements  = 0 ;
   niterators = 0 ;
   key_type   = keytype ;
   data_type  = datatype ;

   iterators = NULL ;
}


/*-------------------------------------------------
 * FUNCTION: collection::~collection
 *
 * DESCRIPTION: Collection destroyer-need to 
 * clear iterators
 *
 */

collection::~collection()
{
   int      it ;
   int      err ;
   iterator *thisIterator ;

   // Tell each attached iterator that the
   // collection is going

   for(it=0;it<niterators;it++) 
   {
      thisIterator = iterators[it] ;

      if(thisIterator == NULL) continue ;

      // This cant fail with a NULL collection 
      thisIterator->reset(err,NULL) ;
   }

   // Free the memory for the iterators 
   delete [] iterators ;
}


/*-------------------------------------------------
 * FUNCTION: addIterator
 *
 * DESCRIPTION: Add an iterator to a collection
 * 
 *
 */

void collection::addIterator(int &err, iterator *newIterator)
{
   int  slot      = -1 ;
   int  it,oldSize ;
   iterator **oldIterators ;
   int  newSize ;

   err = MX_ERROR_OK ;

   // Loop through iterators to see if it 
   // is already attached and to find an
   // empty slot in iterator array 

   for(it=0;it<niterators;it++) 
   {
      if(iterators[it] == newIterator) 
      {
     // Already added
     return ;
      }
      else if(iterators[it] == NULL) 
      {
     // Here's a slot
     slot = it ;
      }
   }

   if(slot >= 0) 
   {
      // New iterator can be stored in old slot
      iterators[slot] = newIterator ;
   }
   else
   {
      // New iterator but needs new memory 

      // Store old parameters 
      oldIterators = iterators ;
      oldSize      = niterators*sizeof(iterator*) ;

      // Get new parameters 
      niterators += 4 ;
      newSize      = niterators*sizeof(iterator*) ;

      iterators = new IND_PTR[niterators]  ;
      if(iterators == NULL) 
      {
     niterators = 0 ;
      }

      // Set new array to zero 
      memset(iterators,0,newSize) ;
      // copy old array 
      memcpy(iterators,oldIterators,oldSize) ;

      // Set new element 
      iterators[niterators-4] = newIterator ;

      delete [] oldIterators ;
   }
      
      

   goto exit;
  exit:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: removeIterator
 *
 * DESCRIPTION: Remove an iterator from a collection
 * 
 *
 */

void collection::removeIterator(iterator *oldIterator)
{
   // Loop through iterators looking for the iterator
   int it ;

   for(it=0;it<niterators;it++) 
   {
      if(iterators[it] == oldIterator) 
      {
     // Found it 
     // Set the slot to null 
     iterators[it] = NULL ;
     return ;
      }
   }
   return ;
}


/*-------------------------------------------------
 * FUNCTION: checkIterators
 *
 * DESCRIPTION: If a node is being deleted and
 * it is the current node of an iterator move the
 * iterator on 
 */

void collection::checkIterators(int &err,
                node *n) 
{
   // Loop through iterators looking for the iterator
   int it ;

   err = MX_ERROR_OK ;

   for(it=0;it<niterators;it++) 
   {
      if(iterators[it] == NULL) continue ;

      if((iterators[it])->n == n) 
      {
     // The iterator lies on the node to be deleted
     // Move it on 
     iterators[it]->next(err) ;
     MX_ERROR_CHECK(err) ;
      }
   }
   return ;
  abort:
   return ;
}

/*-------------------------------------------------
 * FUNCTION: checkIteratorsNodeMove
 *
 * DESCRIPTION: If a node is being deleted then
 * nodes change their data. 
 * Here node n is about to be splice out and the
 * iterator should be moved 
 */

void collection::checkIteratorsNodeMove(int &err,
                    node *n,
                    node *newn) 
{
   // Loop through iterators looking for the iterator
   int it ;

   err = MX_ERROR_OK ;

   for(it=0;it<niterators;it++) 
   {
      if(iterators[it] == NULL) continue ;

      // Found an iterator on the node to be spliced
      // Move it onto the next node 
      if((iterators[it])->n == n) (iterators[it])->n = newn ; 
   }
   return ;
}

/*-------------------------------------------------
 * FUNCTION: collection::comp_data
 *
 * DESCRIPTION: Compare the data on two nodes
 * 
 *
 */

int   collection::comp_data(int                 &err,
                mx_attribute_value  &node_data,
                mx_attribute_value  &new_data) 
{
   err = MX_ERROR_OK ;

   int ans = mx_attribute_cmp(err,
                  data_type,
                  node_data,
                  new_data) ;
   MX_ERROR_CHECK(err) ;

   return ans ;

  abort:
   return 0 ;
}

/*-------------------------------------------------
 * FUNCTION: collection::comp_key
 *
 * DESCRIPTION: Compare the keys on two nodes
 * 
 *
 */

int   collection::comp_key(int                 &err,
                   mx_attribute_value  &node_key,
                   mx_attribute_value  &new_key) 
{
   err = MX_ERROR_OK ;

   int ans = mx_attribute_cmp(err,
                  key_type,
                  node_key,
                  new_key) ;
   MX_ERROR_CHECK(err) ;

   return ans ;

  abort:
   return 0 ;
}

/*-------------------------------------------------
 * FUNCTION: collection::delete_data
 *
 * DESCRIPTION: delete the data on a node
 * 
 *
 */

void collection::delete_data(int                 &err,
                 mx_attribute_value  &node_data) 
{
   err = MX_ERROR_OK ;

   mx_attribute_free(data_type,node_data) ;
}

/*-------------------------------------------------
 * FUNCTION: collection::delete_key
 *
 * DESCRIPTION: delete the key on a node
 * 
 *
 */

void collection::delete_key(int                 &err,
                mx_attribute_value  &node_key) 
{
   err = MX_ERROR_OK ;

   mx_attribute_free(key_type,node_key) ;
}


/*-------------------------------------------------
 * FUNCTION: collection::get_data
 *
 * DESCRIPTION: get the data on a node by direct
 * copy
 *
 */

void collection::get_data(int                 &err,
              mx_attribute_value  &node_data,
              mx_attribute_value  &new_data) 
{
   err = MX_ERROR_OK ;

   new_data = node_data ;

   return ;
}



/*-------------------------------------------------
 * FUNCTION: collection::get_key
 *
 * DESCRIPTION: get the key on a node
 * 
 *
 */

void collection::get_key(int                &err,
           mx_attribute_value &node_key, 
           mx_attribute_value &new_key) 
{
   err = MX_ERROR_OK ;

   // Just do an attribute copy 
   new_key = node_key ;

   return ;
}



/*-------------------------------------------------
 * FUNCTION: collection::set_data
 *
 * DESCRIPTION: set the data on a node
 * 
 *
 */

void collection::set_data(int                &err,
              mx_attribute_value &node_data, 
              mx_attribute_value &new_data) 
{
   err = MX_ERROR_OK ;

   /* do a real copy */

   mx_attribute_copy(err,data_type,1,node_data,new_data) ;
   MX_ERROR_CHECK(err) ;

  abort:
   return ;
}



/*-------------------------------------------------
 * FUNCTION: collection::set key
 *
 * DESCRIPTION: set the key on a node
 * 
 *
 */

void collection::set_key(int                 &err,
             mx_attribute_value  &node_key,
             mx_attribute_value  &new_key) 
{
   err = MX_ERROR_OK ;
   
   /* Do a real copy */

   mx_attribute_copy(err,key_type,1,node_key,new_key);
   MX_ERROR_CHECK(err) ;

  abort:
   return ;
}






tree_collection::tree_collection(mx_attribute_type keytype,
                 bool              allowmultiple,
                 mx_attribute_type datatype)
   :t(allowmultiple,
      keytype,
      datatype,
      this), 
    collection(keytype,datatype) 
{} 

