/*
 * postscript.c
 *
 * Report everything on the style data base
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana
 * $Id: report.c,v 1.23.2.2 1997/06/26 09:17:10 demaille Exp $
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "a2ps.h"
#include "styles.h"
#include "report.h"
#include "routines.h"
#include "utilsty.h"
#include "pathwalk.h"

extern char * program_name;

/* Take priviledged access to job */
extern print_job * job;

static char *
prescript_font(FACE font)
{
  switch (font & FACE_MASK) {
  case PLAIN:
    return "\\texttt";
  case SYMBOL:
    return "\\textsy";
  case KEYWORD:
    return "\\keyword";
  case KEYWORD_STRONG:
    return "\\Keyword";
  case COMMENT:
    return "\\comment";
  case COMMENT_STRONG:
    return "\\Comment";
  case LABEL:
    return "\\Label";
  case LABEL_STRONG:
    return "\\LABEL";
  case STRING:
    return "\\String";
  default:
    error (1, 0, "Not enough fonts in prescript_font (%d)", font);
  }
  return NULL; /* For -Wall */
}

/************************************************************************/
/*			check consistence of styles 			*/
/************************************************************************/
/*
 * Check that keywords and symbols are in lower case when lang is
 * case insensitive
 */
static int
check_sensitivity (STYLE lang)
{
  int OK = TRUE;
  
  switch (languages[lang].sensitiveness) {
  case CASE_SENSITIVE:
    break;
  case CASE_INSENSITIVE: 
    {
      const keyword * keywords=languages[lang].keywords;
      const symbol * symbols=languages[lang].regulars;
      const sequence * sequences=languages[lang].sequences;
      
      if (keywords->theKeyword)
	while (keywords->theKeyword) {
	  if (!is_strlower(keywords->theKeyword)) {
	    error (0, 0, 
		   _("In the style %s, the keyword '%s' uses upper chars"),
		    languages[lang].name, keywords->theKeyword);
	    OK = FALSE;
	  }
	  keywords ++;
	}
      
      if (symbols->theKeyword)
	while (symbols->theKeyword) {
	  if (!is_strlower(symbols->theKeyword)) {
	     error (0, 0, 
		    _("In the style %s, the symbol '%s' uses upper chars"),
		    languages[lang].name, symbols->theKeyword);
	    OK = FALSE;
	  }
	  symbols ++;
	}

      if (sequences->theOpening)
	while (sequences->theOpening) {
	  if (!is_strlower(sequences->theOpening)
	      || !is_strlower(sequences->theClosing)) {
	    error (0, 0, 
		   _("In the style %s, the sequence '%s-%s' uses upper chars"),
		    languages[lang].name, 
		    sequences->theOpening, sequences->theClosing);
	    OK = FALSE;
	  }
	  sequences ++;
	}
    }
  }
  return OK;
}

/*
 * Check that there are no doublons in the style sheets
 */
static int
check_doubles (STYLE lang)
{
  int OK = TRUE;
  symbol * symbols = NULL;
  keyword * keywords = NULL;
  int num, i;

  num = sort_keywords(&keywords, lang, NULL, NULL);
  for (i = 0; i+1 < num ; i++)
    if (!ustrcmp(keywords[i].theKeyword, keywords[i+1].theKeyword)) {
      OK = FALSE;
      error (0, 0, _("in style %s, keyword `%s' is defined twice"),
	       languages[lang].name, keywords[i].theKeyword);
    }
  
  num = sort_regulars(&symbols, lang, NULL, NULL);
  for (i = 0; i+1 < num ; i++)
    if (!ustrcmp(symbols[i].theKeyword, symbols[i+1].theKeyword)) {
      OK = FALSE;
      error (0, 0, _("in style %s, regular symbol `%s' is defined twice"),
	     languages[lang].name, symbols[i].theKeyword);
    }
  
  num = sort_specials(&symbols, lang, NULL, NULL);
  for (i = 0; i+1 < num ; i++)
    if (!ustrcmp(symbols[i].theKeyword, symbols[i+1].theKeyword)) {
      OK = FALSE;
      error (0, 0, _("in style %s, special symbol `%s' is defined twice"),
	     languages[lang].name, symbols[i].theKeyword);
    }
  
  if (keywords)
    free (keywords);
  if (symbols)
    free(symbols);

  return OK;
}

/*
 * Checks everything, return TRUE iff everything's alright
 */
int
check(void)
{
  int OK = TRUE;
  STYLE lang;
 
  /* Check that case sensitivity and keywords definition are consistent */
  for (lang=0 ; lang < nbr_languages ; lang++)
    OK = OK && check_sensitivity(lang);

  /* Check that no keywrds are defined twice */
  for (lang=0 ; lang < nbr_languages ; lang++)
    OK = OK && check_doubles(lang);

  return OK;
}

/************************************************************************/
/* 				report styles				*/
/************************************************************************/
/*
 *      This is a correspondance between LaTeX chars and font symbol
 *		(extracted by ./getwx)
 */
uchar * Symbol_to_LaTeX [256] = {
/* .notdef (/0) */      	UNULL,
/* .notdef (/1) */      	UNULL,
/* .notdef (/2) */      	UNULL,
/* .notdef (/3) */      	UNULL,
/* .notdef (/4) */      	UNULL,
/* .notdef (/5) */      	UNULL,
/* .notdef (/6) */      	UNULL,
/* .notdef (/7) */      	UNULL,
/* .notdef (/8) */      	UNULL,
/* .notdef (/9) */      	UNULL,
/* .notdef (/10) */     	UNULL,
/* .notdef (/11) */     	UNULL,
/* .notdef (/12) */     	UNULL,
/* .notdef (/13) */     	UNULL,
/* .notdef (/14) */     	UNULL,
/* .notdef (/15) */     	UNULL,
/* .notdef (/16) */     	UNULL,
/* .notdef (/17) */     	UNULL,
/* .notdef (/18) */     	UNULL,
/* .notdef (/19) */     	UNULL,
/* .notdef (/20) */     	UNULL,
/* .notdef (/21) */     	UNULL,
/* .notdef (/22) */     	UNULL,
/* .notdef (/23) */     	UNULL,
/* .notdef (/24) */     	UNULL,
/* .notdef (/25) */     	UNULL,
/* .notdef (/26) */     	UNULL,
/* .notdef (/27) */     	UNULL,
/* .notdef (/28) */     	UNULL,
/* .notdef (/29) */     	UNULL,
/* .notdef (/30) */     	UNULL,
/* .notdef (/31) */     	UNULL,
/* space (/32) */       	(ustring) " ",
/* exclam (/33) */      	(ustring) "!",
/* universal (/34) */   	(ustring) "$\\forall$",
/* numbersign (/35) */  	(ustring) "\\#",
/* existential (/36) */ 	(ustring) "$\\exists$",
/* percent (/37) */     	(ustring) "\\%",
/* ampersand (/38) */   	(ustring) "\\&",
/* suchthat (/39) */    	(ustring) "$\\suchthat$",
/* parenleft (/40) */   	(ustring) "(",
/* parenright (/41) */  	(ustring) ")",
/* asteriskmath (/42) */	(ustring) "*",
/* plus (/43) */        	(ustring) "+",
/* comma (/44) */       	(ustring) ",",
/* minus (/45) */       	(ustring) "-",
/* period (/46) */      	(ustring) ".",
/* slash (/47) */       	(ustring) "/",
/* zero (/48) */        	(ustring) "0",
/* one (/49) */ 		(ustring) "1",
/* two (/50) */ 		(ustring) "2",
/* three (/51) */       	(ustring) "3",
/* four (/52) */        	(ustring) "4",
/* five (/53) */        	(ustring) "5",
/* six (/54) */ 		(ustring) "6",
/* seven (/55) */       	(ustring) "7",
/* eight (/56) */       	(ustring) "8",
/* nine (/57) */        	(ustring) "9",
/* colon (/58) */       	(ustring) ":",
/* semicolon (/59) */   	(ustring) ";",
/* less (/60) */        	(ustring) "<",
/* equal (/61) */       	(ustring) "=",
/* greater (/62) */     	(ustring) ">",
/* question (/63) */    	(ustring) "?",
/* congruent (/64) */   	(ustring) "$\\cong$",
/* Alpha (/65) */       	(ustring) "$\\Alpha$",
/* Beta (/66) */        	(ustring) "$\\Beta$",
/* Chi (/67) */ 		(ustring) "$\\Chi$",
/* Delta (/68) */       	(ustring) "$\\Delta$",
/* Epsilon (/69) */     	(ustring) "$\\Epsilon$",
/* Phi (/70) */ 		(ustring) "$\\Phi$",
/* Gamma (/71) */       	(ustring) "$\\Gamma$",
/* Eta (/72) */ 		(ustring) "$\\Eta$",
/* Iota (/73) */        	(ustring) "$\\Iota$",
/* theta1 (/74) */      	(ustring) "$\\vartheta$",
/* Kappa (/75) */       	(ustring) "$\\Kappa$",
/* Lambda (/76) */      	(ustring) "$\\Lambda$",
/* Mu (/77) */  		(ustring) "$\\Mu$",
/* Nu (/78) */  		(ustring) "$\\Nu$",
/* Omicron (/79) */     	(ustring) "$\\Omicron$",
/* Pi (/80) */  		(ustring) "$\\Pi$",
/* Theta (/81) */       	(ustring) "$\\Theta$",
/* Rho (/82) */ 		(ustring) "$\\Rho$",
/* Sigma (/83) */       	(ustring) "$\\Sigma$",
/* Tau (/84) */ 		(ustring) "$\\Tau$",
/* Upsilon (/85) */     	(ustring) "$\\Upsilon$",
/* sigma1 (/86) */      	(ustring) "$\\varsigma$",
/* Omega (/87) */       	(ustring) "$\\Omega$",
/* Xi (/88) */ 		 	(ustring) "$\\Xi$",
/* Psi (/89) */ 		(ustring) "$\\Psi$",
/* Zeta (/90) */        	(ustring) "$\\Zeta$",
/* bracketleft (/91) */ 	(ustring) "[",
/* therefore (/92) */   	(ustring) "$\\therefore$",
/* bracketright (/93) */       	(ustring) "]",
/* perpendicular (/94) */      	(ustring) "$\\perp$",
/* underscore (/95) */  	(ustring) "\\_",
/* radicalex (/96) */   	(ustring) "$\\radicalex$",
/* alpha (/97) */       	(ustring) "$\\alpha$",
/* beta (/98) */        	(ustring) "$\\beta$",
/* chi (/99) */ 		(ustring) "$\\chi$",
/* delta (/100) */      	(ustring) "$\\delta$",
/* epsilon (/101) */    	(ustring) "$\\epsilon$",
/* phi (/102) */        	(ustring) "$\\phi$",
/* gamma (/103) */      	(ustring) "$\\gamma$",
/* eta (/104) */        	(ustring) "$\\eta$",
/* iota (/105) */       	(ustring) "$\\iota$",
/* phi1 (/106) */       	(ustring) "$\\phi$",
/* kappa (/107) */      	(ustring) "$\\kappa$",
/* lambda (/108) */     	(ustring) "$\\lambda$",
/* mu (/109) */ 		(ustring) "$\\mu$",
/* nu (/110) */ 		(ustring) "$\\nu$",
/* omicron (/111) */    	(ustring) "$\\omicron$",
/* pi (/112) */ 		(ustring) "$\\pi$",
/* theta (/113) */      	(ustring) "$\\theta$",
/* rho (/114) */        	(ustring) "$\\rho$",
/* sigma (/115) */      	(ustring) "$\\sigma$",
/* tau (/116) */        	(ustring) "$\\tau$",
/* upsilon (/117) */    	(ustring) "$\\upsilon$",
/* omega1 (/118) */     	(ustring) "$\\varpi$",
/* omega (/119) */      	(ustring) "$\\omega$",
/* xi (/120) */ 		(ustring) "$\\xi$",
/* psi (/121) */        	(ustring) "$\\psi$",
/* zeta (/122) */       	(ustring) "$\\zeta$",
/* braceleft (/123) */  	(ustring) "\\{",
/* bar (/124) */        	(ustring) "$\\|$",
/* braceright (/125) */ 	(ustring) "\\}",
/* similar (/126) */    	(ustring) "$\\sim$",
/* .notdef (/127) */    	UNULL,
/* .notdef (/128) */    	UNULL,
/* .notdef (/129) */    	UNULL,
/* .notdef (/130) */    	UNULL,
/* .notdef (/131) */    	UNULL,
/* .notdef (/132) */    	UNULL,
/* .notdef (/133) */    	UNULL,
/* .notdef (/134) */    	UNULL,
/* .notdef (/135) */    	UNULL,
/* .notdef (/136) */    	UNULL,
/* .notdef (/137) */    	UNULL,
/* .notdef (/138) */    	UNULL,
/* .notdef (/139) */    	UNULL,
/* .notdef (/140) */    	UNULL,
/* .notdef (/141) */    	UNULL,
/* .notdef (/142) */    	UNULL,
/* .notdef (/143) */    	UNULL,
/* .notdef (/144) */    	UNULL,
/* .notdef (/145) */    	UNULL,
/* .notdef (/146) */    	UNULL,
/* .notdef (/147) */    	UNULL,
/* .notdef (/148) */    	UNULL,
/* .notdef (/149) */    	UNULL,
/* .notdef (/150) */    	UNULL,
/* .notdef (/151) */    	UNULL,
/* .notdef (/152) */    	UNULL,
/* .notdef (/153) */    	UNULL,
/* .notdef (/154) */    	UNULL,
/* .notdef (/155) */    	UNULL,
/* .notdef (/156) */    	UNULL,
/* .notdef (/157) */    	UNULL,
/* .notdef (/158) */    	UNULL,
/* .notdef (/159) */    	UNULL,
/* .notdef (/160) */    	UNULL,
/* Upsilon1 (/161) */   	(ustring) "$\\varUpsilon$",/***/
/* minute (/162) */     	(ustring) "$\\prime$",
/* lessequal (/163) */  	(ustring) "$\\leq$",
/* fraction (/164) */   	(ustring) "/",
/* infinity (/165) */   	(ustring) "$\\infty$",
/* florin (/166) */     	(ustring) "$\\florin$",/****/
/* club (/167) */       	(ustring) "$\\clubsuit$",
/* diamond (/168) */    	(ustring) "$\\diamondsuit$",
/* heart (/169) */      	(ustring) "$\\heartsuit$",
/* spade (/170) */      	(ustring) "$\\spadesuit$",
/* arrowboth (/171) */  	(ustring) "$\\leftrightarrow$",
/* arrowleft (/172) */  	(ustring) "$\\leftarrow$",
/* arrowup (/173) */    	(ustring) "$\\uparrow$",
/* arrowright (/174) */ 	(ustring) "$\\rightarrow$",
/* arrowdown (/175) */  	(ustring) "$\\downarrow$",
/* degree (/176) */     	(ustring) "$\\circ$",/***/
/* plusminus (/177) */  	(ustring) "$\\pm$",
/* second (/178) */     	(ustring) "''",
/* greaterequal (/179) */       (ustring) "$\\geq$",
/* multiply (/180) */   	(ustring) "$\\times$",
/* proportional (/181) */       (ustring) "$\\propto$",
/* partialdiff (/182) */        (ustring) "$\\partial$",
/* bullet (/183) */    		(ustring) "$\\bullet$",
/* divide (/184) */     	(ustring) "$\\div$",
/* notequal (/185) */   	(ustring) "$\\neq$",
/* equivalence (/186) */        (ustring) "$\\equiv$",
/* approxequal (/187) */        (ustring) "$\\approx$",
/* ellipsis (/188) */   	(ustring) "$\\ldots$",
/* arrowvertex (/189) */        (ustring) "$\\|$",
/* arrowhorizex (/190) */       UNULL,
/* carriagereturn (/191) */     (ustring) "\\carriagereturn",
/* aleph (/192) */      	(ustring) "$\\aleph$",
/* Ifraktur (/193) */   	(ustring) "$\\Im$",
/* Rfraktur (/194) */   	(ustring) "$\\Re$",
/* weierstrass (/195) */        (ustring) "$\\wp$",
/* circlemultiply (/196) */     (ustring) "$\\otimes$",
/* circleplus (/197) */ 	(ustring) "$\\oplus$",
/* emptyset (/198) */ 		(ustring) "$\\emptyset$",
/* intersection (/199) */       (ustring) "$\\cap$",
/* union (/200) */     		(ustring) "$\\cup$",
/* propersuperset (/201) */     (ustring) "$\\supset$",
/* reflexsuperset (/202) */     (ustring) "$\\supseteq$",
/* notsubset (/203) */  	(ustring) "$\\not\\subset$",
/* propersubset (/204) */       (ustring) "$\\subset$",
/* reflexsubset (/205) */       (ustring) "$\\subseteq$",
/* element (/206) */    	(ustring) "$\\in$",
/* notelement (/207) */ 	(ustring) "$\\not\\in$",
/* angle (/208) */      	(ustring) "$\\angle$",
/* gradient (/209) */   	(ustring) "$\\nabla$",
/* registerserif (/210) */      (ustring) "$\\register$", /****/
/* copyrightserif (/211) */     (ustring) "\\copyright",/****/
/* trademarkserif (/212) */     (ustring) "$\\trademark$",/****/
/* product (/213) */    	(ustring) "$\\prod$",
/* radical (/214) */   	 	(ustring) "$\\surd$",
/* dotmath (/215) */    	(ustring) "$\\cdot$",
/* logicalnot (/216) */ 	(ustring) "$\\not$",
/* logicaland (/217) */ 	(ustring) "$\\wedge$",
/* logicalor (/218) */  	(ustring) "$\\vee$",
/* arrowdblboth (/219) */       (ustring) "$\\Leftrightarrow$",
/* arrowdblleft (/220) */       (ustring) "$\\Leftarrow$",
/* arrowdblup (/221) */ 	(ustring) "$\\Uparrow$",
/* arrowdblright (/222) */      (ustring) "$\\Rightarrow$",
/* arrowdbldown (/223) */       (ustring) "$\\Downarrow$",
/* lozenge (/224) */    	(ustring) "$\\diamondsuit$",	/****/
/* angleleft (/225) */  	(ustring) "$\\langle$",
/* registersans (/226) */       (ustring) "$\\register$",/***/
/* copyrightsans (/227) */      (ustring) "\\copyright",/***/
/* trademarksans (/228) */      (ustring) "$\\trademark$",/****/
/* summation (/229) */  	(ustring) "$\\sum$",
/* parenlefttp (/230) */        UNULL,
/* parenleftex (/231) */        UNULL,
/* parenleftbt (/232) */        UNULL,
/* bracketlefttp (/233) */      (ustring) "$\\lceil$",
/* bracketleftex (/234) */      UNULL,
/* bracketleftbt (/235) */      (ustring) "$\\lfloor$",
/* bracelefttp (/236) */        UNULL,
/* braceleftmid (/237) */       UNULL,
/* braceleftbt (/238) */        UNULL,
/* braceex (/239) */    	UNULL,
/* .notdef (/240) */    	(ustring) "\\apple",
/* angleright (/241) */ 	(ustring) "$\\rangle$",
/* integral (/242) */   	(ustring) "$\\int$",
/* integraltp (/243) */ 	UNULL,
/* integralex (/244) */		UNULL,
/* integralbt (/245) */		UNULL,
/* parenrighttp (/246) */       UNULL,
/* parenrightex (/247) */       UNULL,
/* parenrightbt (/248) */       UNULL,
/* bracketrighttp (/249) */     (ustring) "$\\rceil$",
/* bracketrightex (/250) */     UNULL,
/* bracketrightbt (/251) */     (ustring) "$\\rfloor$",
/* bracerighttp (/252) */       UNULL,
/* bracerightmid (/253) */      UNULL,
/* bracerightbt (/254) */       UNULL,
};

/*
 * sends to stdout the string quoted for LaTeX
 * math mode is never in a \text??{}, and special chars are backslashed
 */
static void 
print_quoted_latex(FACE font, const uchar* string)
{
  int font_given=FALSE;

  switch (font) {
  case INVISIBLE:
    return;
  case SYMBOL:
    for (/*skip */; *string; string++)
      if (*Symbol_to_LaTeX[*string])
	printf("%s", Symbol_to_LaTeX[*string]);
      else
	printf("%c", *string);
    break;
  default:
    for (/* skip */; *string; string++)
      switch (*string) {
      case '#':
      case '&':
      case '_':
      case '$':
      case '%':
	if (font_given) {
	  printf("}");
	  font_given=FALSE;
	}
	printf("\\%c", *string);
	break;
      case '\\':
	if (font_given) {
	  printf("}");
	  font_given=FALSE;
	}
	printf("$backslash$");
	break;
      case '\n':
	if (font_given) {
	  printf("}");
	  font_given=FALSE;
	}
	printf("$\\backslash$n");
	break;
      default:
	if (!font_given) {
	  printf("%s{", prescript_font(font));
	  font_given=TRUE;
	}
	printf("%c", *string);
      }
    if (font_given)
      printf("}");
  }
}

/*
 * printf a string in the \verb+ + command
 */
static void 
print_verb_latex(const uchar *string)
{
  uchar* verb_delims=(uchar *)"!#+|";

  /* search for a delimiter (for \verb) which is not in the string */
  for (/* skip */; *verb_delims; verb_delims++)
    if (!ustrchr(string, *verb_delims))
      break;

  if (*verb_delims) {
    printf("\\verb%c", *verb_delims);
    for (/* skip*/ ; *string; string++)
      switch (*string) {
      case '\n':
	printf("\\n");
	break;
      default:
	printf("%c", *string);
      }
    printf("%c", *verb_delims);
  } else 
    error (1, 0, _("not enough \\verb delimiters (%s) to print %s"), 
	    verb_delims, string);
}

/*
 *
 */
static void 
print_char_verb_latex(const uchar c)
{
  switch (c) {
  case '|':
    printf("\\verb+%c+", c);
    break;
  default:
    printf("\\verb|%c|", c);
    break;
  }
}

/*
 * Give the content of an alphabet
 */
static void 
report_alphabet(const uchar *alphabet)
{
  uchar letter;

  while (*alphabet) {
    if (*alphabet == *(alphabet+1)) {
      printf("'");
      print_char_verb_latex(*alphabet);
      printf("'(%d)%s",
	     *alphabet,
	     *(alphabet+2) != '\0' ? ", " : ".\n");
    } else {
      printf("'");
      for (letter = *alphabet; letter <= *(alphabet+1); letter++)
	print_char_verb_latex(letter);
      printf("'(%d-%d)%s", *alphabet, *(alphabet+1),
	     *(alphabet+2) != '\0' ? ", " : ".\n");
    }
    alphabet += 2;
  }
}

/*
 * Give the alphabets of lang
 */
static void 
report_alphabets(STYLE lang)
{
  /* are alphabets equal ? */
  if (!ustrcmp(languages[lang].first_characters, 
	      languages[lang].following_characters)) {
    printf("\\item \\textrm{first and second alphabet are:}\n");
    report_alphabet(languages[lang].first_characters);
  } else {
    printf("\\item \\textrm{first alphabet is:}\n");
    report_alphabet(languages[lang].first_characters);
    printf("\\item \\textrm{second alphabet is:}\n");
    report_alphabet(languages[lang].following_characters);
  }
}
/*
 * Give the description note of lang
 */
static void 
report_description(STYLE lang)
{
  const uchar * desc = languages[lang].description;

  if (*desc != NUL) {
    printf("\\item %s\n", desc);
    /*print_quoted_latex((uchar *)TIMES, desc);*/
  }
}

/*
 * Give the abbreviations admited of lang
 */
static void 
report_abbreviations(STYLE lang)
{
  const char ** abbr=languages[lang].abbreviations;

  if (**abbr == NUL)
    printf("\\item \\textrm{there are no abbreviation.} \n");
  else {
    printf("\\item \\textrm{the abbreviation are:} ");
    for (abbr = languages[lang].abbreviations ; **abbr ; abbr++) {
      printf("%s%s", *abbr, **(abbr+1)==NUL ? ".\n" : ", ");
    }
  }
}

/*
 * Give the case sensitivity of lang
 */
static void 
report_sensitivity(STYLE lang)
{
  printf("\\item \\textrm{keywords, symbols and sequences recognition ");
  printf("is case %s.}\n",
	 languages[lang].sensitiveness == CASE_SENSITIVE ?
	 "sensitive" : "insensitive");
}

/*
 * Give the keywords of lang
 */
static void 
report_keywords(STYLE lang)
{
  const keyword * array=languages[lang].keywords;
  int count=0;

  if (IS_EMPTY(array->theKeyword))
    printf ("\\item \\textrm{there are no keywords.}\n");
  else {
    printf ("\\item \\textrm{the keywords are:}\n");
    while (!IS_EMPTY(array->theKeyword)) {
      print_quoted_latex (array->theFont, array->theKeyword);
      printf ("%s", IS_EMPTY((array + 1)->theKeyword) 
	     ? ".\n"  : (count % 4) ? ", " : ", %\n");
      array ++;
      count ++;
    }
  }
}

/*
 * Give the symbols of lang
 */
static void 
report_symbols(STYLE lang)
{
  const symbol * array=languages[lang].regulars;
  int count=0;

  if (array->theKeyword)
    printf("\\item \\textrm{there are no regular symbols.}\n");
  else {
    printf("\\item \\textrm{the regular symbols are:}\n");
    while (array->theKeyword) {
      /* the original word */
      print_verb_latex(array->theKeyword);
      printf(" $\\to$ ");
      /* the written string & font */
      print_quoted_latex(array->theFont, array->theSymbol);
      printf("%s", 
	     IS_EMPTY((array+1)->theKeyword) ? ".\n" : 
	     (count % 2) ? ", " : ", %\n");
      array ++;
      count ++;
    }
  }
}

/*
 * Give the specials of lang
 */
static void 
report_specials(STYLE lang)
{
  const symbol * array=languages[lang].specials;
  int count=0;

  if (IS_EMPTY(array->theKeyword))
    printf("\\item \\textrm{there are no special symbols.}\n");
  else {
    printf("\\item \\textrm{the special symbols are:}\n");
    while (!IS_EMPTY(array->theKeyword)) {
      /* the original word */
      print_verb_latex(array->theKeyword);
      printf(" $\\to$ ");
      /* the written string & font */
      print_quoted_latex(array->theFont, array->theSymbol);
      printf("%s", IS_EMPTY((array+1)->theKeyword) ? ".\n" : (count % 2) ? ", " : ", %\n");
      array ++;
      count ++;
    }
  }
}

/*
 * Give the sequences of lang
 */
static void 
report_sequences(STYLE lang)
{
  const sequence * array=languages[lang].sequences;
  uchar string[256];

  if (IS_EMPTY(array->theOpening))
    printf("\\item \\textrm{there are no sequences.}\n");
  else {
    printf("\\item \\textrm{the sequences are:}\n");
    while (!IS_EMPTY(array->theOpening)) {
      sprintf((char *) string, "%sinside%s", 
	      array->theOpening, array->theClosing);
      print_verb_latex(string);
      printf(" $\\to$ "); 
      /* written opening string */
      print_quoted_latex(array->theOpeningFont, array->theOpening);
      /* the inside font */
      print_quoted_latex(array->theInsideFont, (ustring) "inside");
      /* written closing string */
      print_quoted_latex(array->theClosingFont, array->theClosing);
      printf("%s", IS_EMPTY((array+1)->theOpening) ? ".\n" : ", %\n");
      array ++;
    }
  }
}

/*
 * Give the verbatims of lang
 */
static void 
report_verbatims(STYLE lang)
{
  const uchar **array=languages[lang].verbatims;

  if (IS_EMPTY(*array))
    printf("\\item \\textrm{there are no verbatims.}\n");
  else {
    printf("\\item \\textrm{the verbatims are:}\n");
    while (!IS_EMPTY(*array)) {
      printf("\"");
      print_verb_latex(*array);
      printf("\"%s",
	     *(array+1) ? ", " : ".\n");
      array ++;
    }
  }
}

/*
 * Give the escapes of lang
 */
static void 
report_escapes(STYLE lang)
{
  const uchar **array=languages[lang].escapes;

  if (IS_EMPTY(*array))
    printf ("\\item \\textrm{there are no escapes.}\n");
  else {
    printf ("\\item \\textrm{the escapes are:}\n");
    while (!IS_EMPTY(*array)) {
      printf ("\"");
      print_verb_latex (*array);
      printf ("\"%s", *(++array) ? ", " : ".\n");
    }
  }
}

/*
 * Report about a language 
 */
static void 
report_language (STYLE lang)
{
  if (lang == plain_style || lang == unknown_style)
    return;

  printf("\\subsection{%s style}\n", languages[lang].name);
  printf("\\begin{itemize}\n");
  printf("\\header{%s style}\n", languages[lang].name);
  report_description(lang);
  report_abbreviations(lang);
  report_alphabets(lang);
  report_sensitivity(lang);
  report_keywords(lang);
  report_symbols(lang);
  report_specials(lang);
  report_sequences(lang);
  report_verbatims(lang);
  report_escapes(lang);
  printf("\\end{itemize}\n");
}

/*
 * Print the prescript banner
 */
static void 
report_styles(void)
{
  if (!pw_paste_file (job->lib_path, "styledef", ".pre"))
    error (1, errno, _("couldn't find prolog \"%s\""),
	   "styledef.pre");
}

/*
 * Print the prescript banner
 */
static void 
report_prologue(void)
{
  printf("%% This has been generated by %s version %s\n", program_name, VERSION);
  printf("%% It should be processed by either LaTeX or\n");
  printf("%% %s with options -Epre to be printable.\n", PACKAGE);
  printf("%%\n\n");
  
  if (!pw_paste_file (job->lib_path, "report", ".pre"))
    error (1, errno, _("couldn't find prolog \"%s\""), "report.pre");
}

/************************************************************************/
/*				global functions 			*/
/************************************************************************/
/*
 * Report just about one language
 */
void 
report_one_language (STYLE lang)
{
  /* try to be as close as possible of LaTeX */
  report_prologue();
  printf("\\begin{document}\n");

  printf("\\textbi{This is the automatic report about %s-style of ",
	 languages[lang].name);
  printf("%s, version %s}\n\n", program_name, VERSION);
  printf("\n\n\n");
  
  /* Language by language */
  report_language(lang);

  printf("\\end{document}\n");
  exit(EXIT_SUCCESS);
}

/*
 * The whole report about styles
 */
void 
report(void)
{
  STYLE lang;

  /* try to be as close as possible of LaTeX */
  report_prologue();
  printf("\\begin{document}\n");

  printf("\\textbi{This is the automatic report about styles of ");
  printf("%s, version %s}\n\n", program_name, VERSION);
  printf("\\footer{Style sheets of %s version %s}\n", program_name, VERSION);
  printf("\\header{%s report}\n", program_name);
  printf("\n\n\n");

  /* Explanation about the style definition */
  report_styles();

  /* Check the consistancy of styles */
  printf("\\section{Consistancy}\n");
  if (check())
    printf("\\textrm{There are no found problems in the styles.}\n");
  else
    printf("\\textrm{Problems were found in the file styles.c}\n");
  printf("\n\n\n");

  /* list of the known languages */
  printf("\\section{Known languages}\n");
  printf("\\begin{enumerate}\n");
  for (lang=0 ; lang < nbr_languages ; lang++)
    printf("\\magictt{%2d}\\item %s\n",
	   lang + 1, languages[lang].name);
  printf("\\end{enumerate}\n");
  printf("\n\n\n");

  /* Description of the language styles */
  printf("\\section{Description of the language styles}\n");
  /* Language by language */
  for (lang=0 ; lang < nbr_languages ; lang++) {
    printf("\\magicbf{%d} ", lang + 1);
    report_language(lang);
    if (lang < nbr_languages - 1)
      printf("\n\n\n");
  }
  printf("\\end{document}\n");

  exit(EXIT_SUCCESS);
}
