/*
 * main.c
 *
 * main loop, and interface with user
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/************************************************************************/
/*									*/
/*			I n c l u d e   f i l e s			*/
/*                                                                      */
/************************************************************************/

#include "a2ps.h"
#include "routines.h"
#include "styles.h"
#include "utilsty.h"
#include "jobs.h"
#include "confg.h"
#include "psgen.h"
#include "buffer.h"
#include "report.h"
#include "pathwalk.h"
#include "darray.h"


/************************************************************************/
/*									*/
/*		   G l o b a l   d e f i n i t i o n s			*/
/*									*/
/************************************************************************/
char *program_name;		/* Name of this program 		*/
print_job * job = NULL;		/* The whole information		*/
int message_verbosity;

/************************************************************************/
/*				Interface				*/
/************************************************************************/
/*
 * Print the version, and information depending on the installation.
 * Then exit.
 */
static void
version (void)
{
  printf ("%s %s -- Akim Demaille, Miguel Santana\n", PACKAGE, VERSION);
  printf (_("Send bugs reports and comments to demaille@inf.enst.fr\n"));
}

/*
 * Print information depending on the installation.
 */
static void
list_options(void)
{
  uchar buf[256], buf2[256];

  printf (_("\
General information on %s:\n\
  version             = %s\n\
  library path        = %s\n\
  verbosity level     = %d\n"),
	  program_name, VERSION, 
	  UNNULL (job->lib_path),
	  message_verbosity);

  printf (_("\
Sheets:\n\
  default paper         = %s\n\
  layout per page       = %d x %d (%s, %s)\n\
  borders               = %s\n\
  compact mode          = %s\n\
  inside margin         = %d\n"), 
	  medias[job->paper].name,
	  job->columns, job->rows, 
	  job->orientation ? _("portrait") : _("landscape"),
	  job->Major == major_rows ? _("rows first") : _("columns first"),
	  job->border ? _("yes") : _("no"),
	  job->compact_mode ? _("yes") : _("no"),
	  job->margin);

  if (job->columns_requested > 0)
    sprintf ((char *) buf, _("%d columns per line"), 
	     job->columns_requested);
  else if (job->lines_requested > 0)
    sprintf ((char *) buf, _("%d lines per page"), 
	     job->lines_requested);
  else
    /* Here "free", means that the number of chars per line is
     * not fixed, nor the number of lines per page */       
    sprintf ((char *) buf, _("free"));

  switch (job->numbering) {
  case 0:
    ustrcpy (buf2, _("no"));
    break;
  case 1:
    /* number line: each line */
    ustrcpy (buf2, _("each line"));
    break;
  default:
    /* number line: each %d line */
    sprintf ((char *) buf2, _("each %d lines"), job->numbering);
  }

 printf (_("\
Virtual pages:\n\
  number lines          = %s\n\
  format                = %s\n\
  tabulation size       = %d\n\
  non printable format  = %s\n"),
	 buf2,
	 buf,
	 job->tabsize,
	 job->only_printable ? _("space") : _("caret"));

  printf (_("\
Headers:\n\
  header              = %s\n\
  left footer         = %s\n\
  footer              = %s\n\
  right footer        = %s\n\
  left title          = %s\n\
  center title        = %s\n\
  right title         = %s\n\
  under lay           = %s\n"),
	  UNNULL (job->header),
	  UNNULL (job->left_footer),
	  UNNULL (job->footer),
	  UNNULL (job->right_footer),
	  UNNULL (job->left_title),
	  UNNULL (job->center_title),
	  UNNULL (job->right_title),
	  UNNULL (job->water));

  printf (_("\
Input:\n\
  truncate lines        = %s\n\
  interpret             = %s\n\
  default encoding      = %s\n\
  default prologue      = %s\n\
  print anyway          = %s\n"),
	  job->folding ? _("no") : _("yes"),
	  job->interpret ? _("yes") : _("no"),
	  encodings[job->requested_encoding].names[1],
	  job->prolog,
	  job->print_binaries ? _("yes") : _("no"));
  
  if (job->lpr_print) {
    if (!IS_EMPTY(job->printer))
      sprintf ((char *) buf, _("to the printer %s"), job->printer);
    else
      sprintf ((char *) buf, _("to the default printer"));
  } else {
    if (!IS_EMPTY (job->output_filename))
      sprintf ((char *) buf, _("to the file %s"), job->output_filename);
    else
      sprintf ((char *) buf, _("to the standard output"));
  }
  
  printf (_("\
Output:\n\
  destination             = %s\n\
  number of copies        = %d\n\
  page device definitions = "),
	  buf,
	  job->copies);
  hash_map (job->pagedevice, dict_entry_print);
  printf (_("\n\
  statusdict definitions  = "));
  hash_map (job->statusdict, dict_entry_print);
  printf (_("\n\
  page prefeed            = %s\n"),
	  job->page_prefeed ? _("yes") : _("no"));

  printf (_("\
Pretty-printing:\n\
  style sheet           = %s\n\
  automatic style       = %s\n\
  graphic symbols       = %s\n\
  strip level           = %d\n"),
	  job->a2ps_stat->style,
	  job->a2ps_stat->automatic_style ? _("yes") : _("no"),
	  job->a2ps_stat->translate_symbols_request ? _("yes") : _("no"),
	  job->a2ps_stat->strip);
}

/*
 * Print information depending on the installation.
 */
static void
list_features(void)
{
  ENCODING e;
  int m;
  STYLE i;
  char buf[512];

  /* Known languages */
  printf (_("Known languages (%d)"), nbr_languages);
  for (i=0 ; i < nbr_languages ; i++) {
    if (!(i % 5))
      fputs ("\n  ", stdout);
    printf ("%-14s ", languages[i].name);
  }
  printf ("\n");  

  /* Known char sets */
  printf (_("Known encodings (%d)"), nbr_encodings);
  for (e=0 ; e < nbr_encodings ; e++) {
    sprintf (buf, "%s (%s)", encodings[e].names[1], encodings[e].names[0]);
    if (!(e % 3))
      fputs ("\n  ", stdout);
    printf ("%-25s", buf);
  }
  printf ("\n");

  /* Known media */
  printf (_("Known media (%d)\n"), nbr_medias);
  printf ("  %-10s\t%11s (%4s, %4s, %4s, %4s)\n", 
	  _("Name"),
	  _("dimensions"), 
	  "llx", "lly",
	  "urx", "ury");
  for (m=0 ; m < nbr_medias ; m++)
    printf ("  %-10s\t%4d x %4d (%4d, %4d, %4d, %4d)\n", 
	    medias[m].name,
	    medias[m].w, medias[m].h, 
	    medias[m].llx, medias[m].lly,
	    medias[m].urx, medias[m].ury);

  /* Known prologues */
  printf (_("Known prologues\n"));
  pw_list_suffix (job->lib_path, _("  In %s:\n\t"), "%s ", "\n", "pro");

  printf (_("Known output destination (printers, etc.)\n"));
  /* Known printer */
  list_printers (job);

  printf (_("Known user options\n"));
  list_user_options (job);
}

/*
 * Print a usage message.
 */
static void
usage(void)
{
  printf (_("\
Usage: %s [OPTIONS]... [FILES]...\n\
Mandatory arguments to long options are mandatory for short options too.\n\
Long options marked with * require a yes/no argument, corresponding\n\
short options stand for `yes'.\n"),
	  program_name);

  fputs (_("\
Global behavior:\n\
  -q, --quiet, --silent      be really quiet\n\
  -v, --verbose              tell what we are doing\n\
  -V, --version              print version and exit\n\
  -=, --user-option=OPTION   use the user defined shortcut OPTION\n\
  --report[=LANG]            generate PreScript report on all/lang style\n\
  --guess                    write guessed languages (don't print)\n\
  --help                     print this help and exit\n\
  --list-options             list all options and their values\n\
  --list-features            list media, encodings, languages, prologues,\n\
                             printers, and user option shortcuts\n"),
	 stdout);

  fputs (_("\
Sheets:\n\
  -M, --media=NAME           use output media NAME\n\
  -1, -2, ..., -9            predefined layouts for 1, ..., 9 virtual pages\n\
  --columns=NUM              number of columns per sheet (NUM > 0)\n\
  --rows=NUM                 number of rows per sheet (NUM > 0)\n\
  --major=DIRECTION          first fill (DIRECTION=)rows, or columns\n\
  -r, --landscape            print in landscape mode\n\
  -R, --portrait             print in portrait mode\n\
  -j, --borders*             print borders around columns\n\
  -A, --compact*             allow two files on the same sheet\n\
      --margin[=NUM]         define an interior margin of size NUM\n"),
	 stdout);

  fputs (_("\
Virtual pages:\n\
  --line-numbers=NUM         precede each NUM lines with its line number\n\
  -C                         alias for --line-numbers=5\n\
  -f, --fontsize=SIZE        use font SIZE (float) for the body text\n\
  -L, --lines-per-page=NUM   specify how many lines are printed on each page\n\
  -l, --columns-per-page=NUM specify how many columns are printed per page\n\
  -m  --catman               process the file as a man (same as -L66)\n\
  -T, --tabsize=NUM          set tabulator size to NUM\n\
  --non-printable-format=FMT specify how non-printable chars are printed\n"),
	 stdout);

  /* xgettext:no-c-format */
  fputs (_("\
Headers:\n\
  -B, --no-header            no page headers at all\n\
  -b, --header[=TEXT]        set page header\n\
  -u, --underlay[=TEXT]      print TEXT under every page\n\
      --center-title[=TEXT]  set page title to TITLE\n\
  --left-title[=TEXT]        set left and right page title to TEXT\n\
  --right-title[=TEXT]\n\
  --left-footer[=TEXT]       set sheet footers to TEXT\n\
  --footer[=TEXT]            TEXT may use special $ and % escapes\n\
  --right-footer[=TEXT]\n"), stdout);

  fputs (_("\
Input:\n\
  -c, --truncate-lines*      cut long lines\n\
  -i  --interpret*           interpret tab, bs and ff chars\n\
  -X, --encoding=NAME        use input encoding NAME\n\
  -t, --title=NAME           set the name of the input file stdin\n\
  --prologue=FILE            include FILE.pro as PostScript prologue\n\
  --print-anyway*            force binary printing\n"),
	 stdout);

  fputs (_("\
Output:\n\
  -o, --output=FILE          leave output to file FILE.  If FILE is `-',\n\
                             leave output to stdout.\n\
  -n, --copies=NUM           print NUM copies of each page\n\
  -D, --setpagedevice=K[:V]  pass a page device definition to output\n\
  -S, --statusdict=K[:[:]V]  pass a statusdict definition to the output\n\
  -k, --page-prefeed         enable page prefeed\n\
  -K, --no-page-prefeed      disable page prefeed\n"),
	 stdout);
  fputs (_("\
  -P, --printer=NAME         send output to printer NAME\n\
  -d                         send output to the  default printer\n\
  -s  --sides=NUM            number of sheet sides (recto/recto-verso)\n"),
	 stdout);

  fputs (_("\
Pretty-printing:\n\
  -E, --pretty-print=LANG    use the LANG pretty print style sheet\n\
  -e, --automatic-style*     use automatic style sheets\n\
  -g, --graphic-symbols*     set symbols graphical convertion\n\
      --strip-level=NUM      level of comments stripping\n"),
	 stdout);
}

/************************************************************************/
/*			Main routine for this program.			*/
/************************************************************************/
int
main(int argc, char *argv[])
{
  char * command_path = NULL;	/* Where the executable is stored */
  STYLE lang = plain_style;	/* style used to print current file */
  char buf[256];
  int optind;

  /* name under which  and path from which this program has been called */
  program_name = strrchr (argv[0], DIR_SEP);
  if (program_name == NULL)
    program_name = argv[0];
  else {
    *program_name = NUL;
    command_path = argv[0];
    program_name++;
  }
  argv[0] = program_name;
  message_verbosity = 0;

  /* Initialize internationalization stuff */
  setlocale (LC_TIME, "");
  setlocale (LC_MESSAGES, "");
  
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
  job = new_print_job ();

  /* System's config.    */
  read_sys_config (job);
  
  /* Personal config.: only when installed, because there
   * may be incompatibilities between config files versions */
  if (! (getenv ("NO_SYSCONF"))) {
    sprintf (buf, ".%s/%src", PACKAGE, PACKAGE);
    (void) read_config (job, job->home, buf);
  }
  
  /* Local config. */
  sprintf (buf, ".%src", PACKAGE);
  (void) read_config (job, ".", buf);
  
  /* Process global options */
  optind = handle_option (job, argc, argv);
  
  switch (job->a2ps_stat->behavior) {
    /* do we have not to print, but return the type of the files */
  case b_guess:
    for (; optind < argc; optind++) {
      lang = guess_language (job, argv[optind]);
      printf("[%s(%s)]\n", argv[optind], 
	     lang == plain_style ? _("unknown") : languages[lang].name);
    }
    break;
  
  case b_report:
    if (!IS_EMPTY (job->output_filename))
      if (freopen (job->output_filename, "w", stdout) == NULL)
	error (1, errno, _("unable to create output file `%s'"), 
	       job->output_filename);
    if (job->a2ps_stat->style)
      report ();
    else
      report_one_language (select_style (job->a2ps_stat->style));
    break;

  case b_version:
    version ();
    break;

  case b_help:
    usage ();
    break;

  case b_list_options:
    list_options ();
    break;

  case b_list_features:
    list_features ();
    break;

    /*
     * ASCII to PostScript generator
     */
  case b_ps:
    ps_init (job);
    if (optind == argc) {		/* Print stdin */
      setStyle (select_style (job->a2ps_stat->style));
      setSymbolTranslation (job->a2ps_stat->translate_symbols_request);
      new_job (job, NULL);
      print_postscript (job, select_style (job->a2ps_stat->style));
    } else				/* Print following files */
      for ( ; optind < argc ; optind++) {
	/* Select the style corresponding to the language */
	if (strcmp (job->a2ps_stat->style, "plain"))
	  lang = select_style (job->a2ps_stat->style);
	else if (job->a2ps_stat->automatic_style)
	  lang = guess_language (job, argv[optind]);
	else
	  lang = plain_style;
	
	setStyle (lang);
	setSymbolTranslation (job->a2ps_stat->translate_symbols_request);
	new_job (job, argv[optind]);
	print_postscript (job, lang);
      }
    ps_close (job);
    break;
  }
  
  return(EXIT_SUCCESS);
}
