/*
 * psgen.c
 *
 * routines for the postscript generation
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana
 * $Id: psgen.c,v 1.4.2.2 1997/06/17 09:55:54 demaille Exp $
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "a2ps.h"		/* most global variables 		*/
#include "psgen.h"
#include "encoding.h"
#include "routines.h"		/* general interest routines		*/
#include "output.h"		/* Diverted outputs			*/
#include "jobs.h"
#include "medias.h"
#include "faces.h"
#include "fonts.h"

#define PORTRAIT_HEADER		20
#define LANDSCAPE_HEADER	15
#define PAGE_MARGIN 12		/* space between virtual pages		*/
#define HEADERS_H   12		/* Space for header/footer		*/

/* Space between lowest line in the text, and bottom of the frame */
#define SIDE_MARGIN_RATIO	0.7
#define BOTTOM_MARGIN_RATIO	0.7

/*
 * Some static functions
 */
inline static void clean_up PARAMS ((print_job * job));


#define PRINT_HEADER	\
	(!IS_EMPTY(job->header))
#define PRINT_FOOTER	\
	(!IS_EMPTY(job->footer)				\
	 || !IS_EMPTY(job->left_footer)			\
	 || !IS_EMPTY(job->right_footer))
#define PRINT_TITLE	\
	(!IS_EMPTY(job->center_title)				\
	 || !IS_EMPTY(job->left_title)			\
	 || !IS_EMPTY(job->right_title))

/* Width in characters for line prefix	*/
#define prefix_size	(job->numbering ? 5 : 0)

/* Is this the first page for the current file?		*/
#define is_first_page	((job->total_pages - job->jobs->first_page) > 1)

#define jdiv	job->divertion

/* return the max of two >-comparable stuff */
#define MAX(X,Y)        (((X) > (Y)) ? (X) : (Y))


/****************************************************************/
/*		Formating help routines				*/
/****************************************************************/
/*
 * Print a char in a form accepted by postscript printers.
 * Returns number of columns used (on the output) to print the char.
 */
static int 
format_char(print_job * job, uchar c, ustring res)
{
  /* The number of columns used must be calculated here because of the
   * \ before non-ascii chars, and (, ), and \ itself */

  int len=0;

  /* Printable and 7bit clean caracters */
  if (' ' <= c && c < 0177) {
    if (c == '(' || c == ')' || c == '\\')
      USTRCCAT(res, '\\');
    USTRCCAT(res, c);
    return 1;
  }

  /* Printable, but not 7bit clean caracters */
  if (encodings[job->encoding].WX[job->status->face][c] && ((0177 < c) || (c < 040))) {
    sprintf ((char *)res, "%s\\%o", res, c);
    return 1;
  }

  /* We are dealing with a non-printing character */
  job->status->nonprinting_chars++;

  switch (job->only_printable) {
  case TRUE:
    USTRCCAT(res, ' ');
    return 1;
    break;
    
  case FALSE:
    if (0177 < c) {
      ustrcat(res, "M-");
      len += 2; 
      c &= 0177;
    }
    
    if (c < ' ') {
      USTRCCAT(res, '^');
      len += 2; 
      if ((c = c + '@') == '(' || c == ')' || c == '\\')
	USTRCCAT(res, '\\');
      USTRCCAT(res, c);
    } else if (c == 0177) {
      ustrcat(res, "^?");
      len += 2;
    } else {
      if (c == '(' || c == ')' || c == '\\')
	USTRCCAT(res, '\\');
      USTRCCAT(res, c);
      len++;
    }
    return len;
    break;
  }
  return (0);
}

/*
 * Print a string in a form accepted by postscript printers.
 */
static int 
format_string(print_job * job, ustring string, ustring res)
{
  int i;
  int delta_column=0;

  for (i = 0 ; i < ustrlen(string) ; i++) 
    delta_column += format_char (job, string[i], res);
  
  return delta_column;
}

/*
 * Output the formated marker.
 */
static void
output_marker (print_job * job, char * kind, ustring marker)
{
  uchar buf[256], buf2[256];
  int i;

  *buf = NUL;
  *buf2 = NUL;
  
  message (3, "Diverting marker (%s)\n", marker);
  format_user_string (job, kind, marker, buf);
  
  for (i = 0 ; buf[i] ; i++) {
    switch (buf[i]) {
    case FILE_NB_PAGES:
      output_delayed_int (jdiv, &(job->jobs->pages));
      break;
    case FILE_NB_SHEETS:
      output_delayed_int (jdiv, &(job->jobs->sheets));
      break;
    case FILE_NB_LINES:
      output_delayed_int (jdiv, &(job->jobs->linenum));
      break;
    case JOB_NB_PAGES:
      output_delayed_int (jdiv, &job->total_pages);
      break;      
    case JOB_NB_SHEETS:
      output_delayed_int (jdiv, &job->total_sheets);
      break;
    case JOB_NB_FILES:
      output_delayed_int (jdiv, &job->total_files);
      break;
    default:
      *buf2 = NUL;
      format_char (job, buf[i], buf2);
      output (jdiv, (char *) buf2);
      break;
    }
  }
}
/*
 * Change the current encoding
 */
void
ps_set_encoding (print_job * job, ENCODING enc)
{
  if (!job->used_encodings[enc])
    {
      FACE f; 
      /* If the encoding was not used yet, include its PS definition
       * in the prologue */
      output_file (job->ps_encodings, job, encodings[enc].names[0], ".ps");
      /* Moreover, link its WX definitions */
      for (f = 0; f < NB_FACES ; f++)
	encodings[enc].WX[f] = 
	  encodings[enc].font_WX[job->face_to_font[f]];
    }
  job->encoding = enc;
  job->used_encodings[enc] = TRUE;
  job->status->current_wx = encodings[enc].WX;
  output (job->divertion, "%% Encoding is %s\n", encodings [enc].names[1]);
  output (job->divertion, "%sdict begin\n", encodings [enc].names[0]);
}

void
ps_end_encoding (print_job * job)
{
  output (job->divertion, "end %% of %sdict\n",
	  encodings [job->encoding].names[0]);
}

void
ps_internal_switch_encoding (print_job * job, ENCODING enc)
{
  if (job->encoding != enc) {
    ps_end_encoding (job);
    ps_set_encoding (job, enc);
    /* Make sure to re-declare the font */
    job->status->face_declared = FALSE;
  }
}

void
ps_switch_encoding (print_job * job, ENCODING enc)
{
  if (job->encoding != enc) 
    {
      if (!job->status->start_line)
	if (!job->status->face_declared)
	  output (job->divertion, ") %s\n", face_to_ps(job->status->face));
	else
	  output (job->divertion, ") S\n");
      ps_end_encoding (job);
      ps_set_encoding (job, enc);
      /* Make sure to re-declare the font */
      job->status->face_declared = FALSE;
      if (!job->status->start_line)
	output_char (job->divertion, '(');
    }
}


void
ps_push_encoding (print_job * job, ENCODING enc)
{
  job->saved_encoding = job->encoding;
  ps_internal_switch_encoding (job, enc);
}

void
ps_pop_encoding (print_job * job)
{
  ps_internal_switch_encoding (job, job->saved_encoding);
}

/*
 * We want to optimize the number of fonts defined in the
 * postscript: we may run into limitcheck on poor printers
 * The optimization is not very strong: we consider that
 * if say fce is used in a face, then it will be produced
 * in _every_ encoding.  A more parsimonous scheme may be
 * used, but I'm afraid to slow down the whole stuff because
 * of the test required.
 */
void
dump_encodings (print_job * job)
{
  /* If the encoding is the main encoding (that of the
   * headers etc.), dump what is required for the headers */

  underivation (job->ps_encodings);
}

/****************************************************************/
/*		Printing a document				*/
/****************************************************************/
/*
 * Print the PostScript prolog.
 */
static void 
output_prolog (print_job * job)
{
  /* Comments */
  output (jdiv, "%s\n", job->magic_number);
  output (jdiv, "%%%%BoundingBox: %d %d %d %d\n", 
	  medias[job->paper].llx, medias[job->paper].lly,
          medias[job->paper].urx, medias[job->paper].ury);
  output (jdiv, "%%%%Title: %s %s\n", PACKAGE, "output");
  output (jdiv, "%%%%For: %s\n", job->pw_gecos);
  output (jdiv, "%%%%Creator: %s version %s\n", PACKAGE, VERSION);
  output (jdiv, "%%%%CreationDate: %s\n", job->datestring);
  output (jdiv, "%%%%DocumentData: Clean7Bit\n");
  output (jdiv, "%%%%Orientation: %s\n", 
	  (job->orientation == landscape) ? "Landscape" : "Portrait");
  output (jdiv, "%%%%Pages: ");
  output_delayed_int (jdiv, &(job->total_sheets));
  output_char (jdiv, '\n');
  output (jdiv, "%%%%PageOrder: Ascend\n");
  output (jdiv, "%%%%DocumentMedia: %s %d %d 0 () ()\n",
	  medias[job->paper].name, 
	  medias[job->paper].w, medias[job->paper].h);
  output_delayed_routine (jdiv, dump_needed_resources, job);
  output_requirements (job);
  output_delayed_routine (jdiv, dump_supplied_resources, job);
  output (jdiv, "%%%%EndComments\n");
  
  /* If we are in debug mode, download a PS error handler */
  if (job->area)
    output_file (jdiv, job, "ehandler", ".ps");

  /* a2ps dict, (needed for good eps files) */
  output (jdiv, "/a2psdict 200 dict def\n");
  output (jdiv, "a2psdict begin\n");

  /* Prolog */
  output (jdiv, "%%%%BeginProlog\n");

  output_file (jdiv, job, job->prolog, ".pro");
  /* At this point, everything should be known about the faces */
  if (check_face_to_font (job->face_to_font))
    error (1, 0, _("incomplete knowledge of faces"));
  
  /* Encodings used */
  output_delayed_routine (jdiv, dump_encodings, job);
  output (jdiv, "%%%%EndProlog\n");
}

/*
 * Print the PostScript document setup part.
 */
static void 
output_document_setup (print_job * job)
{
  int i, j;

  /* Set up */
  output (jdiv, "%%%%BeginSetup\n");
  /* Complete the prolog with static variables */
  output (jdiv, "%% Initialize page description variables.\n");
  switch (job->orientation) {
  case portrait:
    /*FIXME: prendre en compte les marges */
    output (jdiv, "/sh %d def\n", medias[job->paper].h); 
    output (jdiv, "/sw %d def\n", medias[job->paper].w);
    output (jdiv, "/llx %d def\n", medias[job->paper].llx);
    output (jdiv, "/urx %d def\n", medias[job->paper].urx - job->margin);
    output (jdiv, "/ury %d def\n", medias[job->paper].ury);
    output (jdiv, "/lly %d def\n", medias[job->paper].lly);
    break;
  case landscape:
    output (jdiv, "/sh %d def\n", medias[job->paper].w);
    output (jdiv, "/sw %d def\n", medias[job->paper].h);
    output (jdiv, "/llx %d def\n", medias[job->paper].lly);
    output (jdiv, "/urx %d def\n", medias[job->paper].ury);
    output (jdiv, "/ury %d def\n", 
	    medias[job->paper].w - medias[job->paper].llx);
    output (jdiv, "/lly %d def\n", 
	    medias[job->paper].w - medias[job->paper].urx + job->margin);
    break;
  }

  /* Misceleanous PostScript variables */
  output (jdiv, "/#copies %d def\n", job->copies);

  /* Page prefeed */
  if (job->page_prefeed)
    output (jdiv, "true page_prefeed\n");

  /* statusdict definitions */
  output_statusdict (job);

  /* Page device definitions */
  output_pagedevice (job);
  
  /* Header size */
  output (jdiv, "/th %f def\n", job->status->title_bar_height);

  /* General format */
  /* Font sizes */
  output (jdiv, "/fnfs %d def\n", job->status->title_font_size);
  output (jdiv, "/dfs %f def\n", job->status->title_font_size * 0.8);
  output (jdiv, "/bfs %f def\n", job->fontsize);
  output (jdiv, "/cw %f def\n", job->fontsize * 0.6);  /* char width */
  output (jdiv, "\n");

  output_delayed_routine (jdiv, dump_setup, job);

  add_needed_resource (job, "font", "Symbol");
  output (jdiv, "/fs bfs /Symbol		deffont\n");
  output (jdiv, "/hm fnfs 0.25 mul def\n");
  
  /* Page attributes */
  output (jdiv, "/pw\n");
  output (jdiv, "   cw %f mul\n", 
	  (float) job->status->columnsperline + 2 * SIDE_MARGIN_RATIO);
  output (jdiv, "def\n");
  output (jdiv, "/ph\n");
  output (jdiv, "   %f th add\n", 
	  (job->status->linesperpage + BOTTOM_MARGIN_RATIO) * job->fontsize);
  output (jdiv, "def\n");
  if (job->columns > 1)
    output (jdiv, "/pmw urx llx sub pw %d mul sub %d div def\n",
	    job->columns, job->columns - 1);
  else
    output (jdiv, "/pmw 0 def\n");
  if (job->rows > 1)
    output (jdiv, "/pmh ury lly sub ph %d mul sub %d sub %d div def\n",
	    job->rows, 
	    (PRINT_HEADER + PRINT_FOOTER) * HEADERS_H, 
	    job->rows - 1);
  else
    output (jdiv, "/pmh 0 def\n");
  output (jdiv, "/v 0 def\n");
  
  switch (job->Major) {
  case major_rows:
    output (jdiv, "/x [\n");
    for (j = 1 ; j <= job->rows ; j++) {
      output (jdiv, "  0\n");
      for (i = 2 ; i <= job->columns ; i++)
	output (jdiv, "  dup pmw add pw add\n");
    }
    output (jdiv, "] def\n");

    output (jdiv, "/y [\n");
    for (j = job->rows ; 1 <= j ; j--) {
      output (jdiv, "  pmh ph add %d mul ph add\n", j - 1);
      for (i = 2 ; i <= job->columns ; i++)
	output (jdiv, "  dup\n");
    }
    output (jdiv, "] def\n");
    break;

  case major_columns:
    output (jdiv, "/x [\n");
    for (i = 1 ; i <= job->columns ; i++) {
      output (jdiv, "  pmw pw add %d mul\n", i - 1);
      for (j = 2 ; j <= job->rows ; j++) {
	output (jdiv, "  dup\n");
      }
    }
    output (jdiv, "] def\n");

    output (jdiv, "/y [\n");
    for (i = 1 ; i <= job->columns ; i++)
      for (j = job->rows ; j >= 1 ; j--) {
	output (jdiv, "  pmh ph add %d mul ph add\n", j - 1);
      }
    output (jdiv, "] def\n");
    break;
  default:
    error (1, 0, "internal error in output_document_setup");
    break;
  }
    
  output (jdiv, "/scx sw 2 div def\n");
  output (jdiv, "/scy sh 2 div def\n");
  
  output (jdiv, "/snx urx def\n");
  output (jdiv, "/sny lly 2 add def\n");
  output (jdiv, "/dx llx def\n");
  output (jdiv, "/dy sny def\n");
  output (jdiv, "/fnx scx def\n");
  output (jdiv, "/fny dy def\n");
  output (jdiv, "/lx snx def\n");
  output (jdiv, "/ly ury dfs sub def\n");
  output (jdiv, "/sx %d def\n", prefix_size);
  output (jdiv, "/tab %d def\n", job->tabsize);
  output (jdiv, "/x0 0 def\n");
  output (jdiv, "/y0 0 def\n");

  /* Close prolog */
  output (jdiv, "%%%%EndSetup\n\n");

}

static void
end_document(print_job * job)
{
  clean_up (job);
  output (jdiv, "\n%%%%Trailer\n");
  if (job->page_prefeed)
    output (jdiv, "false page_prefeed\n");
  /* Close a2ps dict */
  output (jdiv, "end\n");
  output (jdiv, "%%%%EOF\n");
}

/****************************************************************/
/*		Printing a physical page			*/
/****************************************************************/
/*
 * Print the prolog necessary for printing each physical page.
 * Adobe convention for page independence is enforced through this routine.
 * Note that is may called for a second time on a single sheet if two
 * different files are printed on the same sheet.
 */
static void 
begin_sheet(print_job * job)
{
  job->virtual = 0;
  job->total_sheets++;

  output (jdiv, 
	  "%%%%Page: (%d) %d\n", 
	  job->total_sheets - job->jobs->first_sheet, job->total_sheets);
  
  /* Reinitialize state variables for each new sheet */
  output (jdiv, "%%%%BeginPageSetup\n");
  output (jdiv, "/pagesave save def\n");

  /* Shift front side sheets */
  if (job->margin
      && (!job->rectoverso || (job->total_sheets % 2)))
    output (jdiv, "%d 0 translate\n", job->margin);
  output (jdiv, "%%%%EndPageSetup\n");

  if (job->orientation == landscape) {
    output (jdiv, "sh 0 translate\n");
    output (jdiv, "90 rotate\n");
  }
  
  if (job->area)
    output (jdiv, "\
%% Display the bounding box\n\
  gsave\n\
    llx lly moveto\n\
    2 setlinewidth\n\
    0.9 setgray\n\
    urx lly lineto\n\
    urx ury lineto\n\
    llx ury lineto\n\
    closepath stroke\n\
  grestore\n\n");

  /* Set the encoding */
  ps_set_encoding (job, job->requested_encoding);

  /* water marks (only once per sheet) */
  if (!IS_EMPTY(job->water)) {
    output_char (jdiv, '(');
    output_marker (job, _("water mark"), job->water);
    output (jdiv, 
	    ") %f water\n",
	    ((float) atan2 ((double) medias[job->paper].w - job->margin, 
			    (double) medias[job->paper].h) 
	     / 3.14159265 * 180));
  }
  
  /* Move to the lower left point of the drawable area */
  output (jdiv, "gsave\n");
  output (jdiv, "llx lly %d add translate\n", 
	  PRINT_FOOTER * HEADERS_H);
  /* Set the encoding */
  ps_switch_encoding (job, job->encoding);
}

/* The job on the page is over: puts the headers and footers, 
 * then print the physical page.
 */
static void 
end_sheet (print_job * job)
{
  uchar buf[256];

  *buf = NUL;

  output (jdiv, "%% End of virtual page\n");
  output (jdiv, "grestore\n");

  /* All the headers should be written using the requested encoding */
  ps_push_encoding (job, job->requested_encoding);

  /* Print the right header */
  if (PRINT_HEADER) { 
    output_char (jdiv, '(');
    output_marker (job, _("right header"), job->header);
    output (jdiv, ") rhead\n");
  }

  /* Print the center footer.
   * Use dynamic markers in priority
   */
  if (PRINT_FOOTER) {
    if (!IS_EMPTY(job->footer)) {
      output_char (jdiv, '(');
      output_marker (job, _("center footer"), job->footer);
      output (jdiv, ") ");
    } else
      output (jdiv, "() ");
    
    /* Print the right footer */
    if (!IS_EMPTY(job->right_footer)) { 
      output_char (jdiv, '(');
      output_marker (job, _("right footer"), job->right_footer);
      output (jdiv, ") ");
    } else
      output (jdiv, "() ");
    
    /* Print the left footer */
    if (!IS_EMPTY(job->left_footer)) { 
      output_char (jdiv, '(');
      output_marker (job, _("left footer"), job->left_footer);
      output (jdiv, ") ");
    }
    else
      output (jdiv, "() ");
    output (jdiv, "footer\n");
  }
  
  /* Close the current encoding */
  ps_end_encoding (job);
  
  output (jdiv, "pagesave restore\n");
  output (jdiv, "showpage\n");

  job->virtual = job->virtuals;
}

/****************************************************************/
/*		Printing a virtual page				*/
/****************************************************************/
/*
 * Prints page header and page border and
 * initializes printing of the file lines.
 */
static void 
begin_page(print_job * job)
{
  job->virtual++;
  job->total_pages++;
  job->jobs->toplinenum = job->jobs->linenum;
  
  output (jdiv, "/v %d store\n", job->virtual - 1);
  output (jdiv, "/x0 x v get %f add sx cw mul add store\n", 
	  SIDE_MARGIN_RATIO * job->fontsize * 0.6);
  output (jdiv, "/y0 y v get bfs %s sub store\n",
	  PRINT_TITLE ? "th add" : "");
  output (jdiv, "x0 y0 moveto\n");
}

/*
 * Adds a sheet number to the page (footnote) and prints the formatted
 * page (physical impression). Activated at the end of each source page.
*/
void 
end_page (print_job * job)
{
  uchar buf[256];

  *buf = NUL;

  /*
   *	Print the titles with the option-given encoding
   */

  /* Draw the header and its content */
  if (PRINT_TITLE) {
    ps_push_encoding (job, job->requested_encoding);
    if (!IS_EMPTY(job->center_title)) { 
      output_char (jdiv, '(');
      output_marker (job, _("center title"), job->center_title);
      output (jdiv, ") ");
    } else
      output (jdiv, "() ");
    
    if (!IS_EMPTY(job->right_title)) { 
      output_char (jdiv, '(');
      output_marker (job, _("right title"), job->right_title);
      output (jdiv, ") ");
    } else
      output (jdiv, "() ");
    
    if (!IS_EMPTY(job->left_title)) { 
      output_char (jdiv, '(');
      output_marker (job, _("left title"), job->left_title);
      output (jdiv, ") ");
    } else
      output (jdiv, "() ");
    
    output (jdiv, "title\n");
    ps_pop_encoding (job);
  }
  
  if (job->border)
    output (jdiv, "border\n");
  
  if (job->virtual == job->virtuals)
    end_sheet (job);
}

/*
 * Flush the page, ready for new sheet
 */
inline static void 
clean_up (print_job * job)
{
  /* If the sheet has not been printed, flush it */
  if (job->virtual < job->virtuals)
    end_sheet (job);

  if (!job->compact_mode 
      && job->rectoverso 
      && (job->total_sheets & 0x1) != 0) {
    job->total_sheets++;
    output (jdiv, "%%%%Page: (%d) %d\n", 
	    job->total_sheets - job->jobs->first_sheet,
	    job->total_sheets);
    /* Reinitialize state variables for each new sheet */
    output (jdiv, "%%%%BeginPageSetup\n");
    output (jdiv, "/pagesave save def\n");
    output (jdiv, "%%%%EndPageSetup\n");
    
    output (jdiv, "%% Blank 2nd side sheet\n");
    output (jdiv, "pagesave restore\n");
    output (jdiv, "showpage\n");
  }
}

/*
 * Begins a new logical page.
 */
void
ps_skip_page(print_job * job)
{
  if (job->virtual + 1 > job->virtuals)
    begin_sheet(job);
  
  begin_page(job);
}

/****************************************************************/
/*		Service routines				*/
/****************************************************************/
/*
 * Test if we have a binary file.
 *
 * Printing binary files is not very useful. We stop printing
 * if we detect one of these files. Our heuristic to detect them:
 * if 40% characters are non-printing characters,
 * the file is a binary file. (40% is taken from the case of a2ps istself).
 */
static void
check_binary_file (print_job * job)
{
  if (job->status->chars > 120) 
    {
      if (!job->print_binaries 
	  && (job->status->nonprinting_chars*100 / job->status->chars) >= 40)
	error (1, 0, _("`%s' is a binary file, printing aborted"),
	       job->jobs->name);
    }
}

#define EOL() \
      do {							\
	if (!job->status->face_declared) {			\
	  output (jdiv, ") %s n\n", face_to_ps(job->status->face));\
	  job->status->face_declared = TRUE;			\
	} else							\
	  output (jdiv, ") N\n");				\
        job->status->line++;					\
      } while (0)

#define page_full	\
	(job->status->line >= job->status->linesperpage)

#define line_full (job->status->wx > job->status->wxperline)

/*
 * Fold a line too long. Return false if was binary file
 */
#define FOLD_LINE()						\
 do {								\
   EOL ();							\
   if (page_full) {						\
     end_page (job);						\
     ps_skip_page (job);					\
     job->status->line = 0;					\
     job->status->face_declared = FALSE ;			\
   }								\
   if (job->numbering) {					\
     if (job->jobs->linenum % job->numbering == 0)		\
       output (jdiv, "(+) # (");				\
     else							\
       output (jdiv, "0 T (");					\
   } else {							\
     /* This is a new line, hence we can consider that there is	\
      * no need to close the current font: just consider it is	\
      * the new font, but not declared */			\
     output_char (jdiv, '(');					\
     job->status->face_declared &= (job->status->face 		\
				    == new_face);		\
     job->status->face = new_face;				\
   }								\
 } while (0)


void
ps_print_char (print_job * job, int c, FACE new_face)
{
  /*
   * Preprocessing (before printing):
   * - Catch dynamic header and footer
   * - TABs expansion (see interpret option)
   * - FF and BS interpretation
   * - replace non printable characters by a space or a char sequence
   *   like:
   *     ^X for ascii codes < 0x20 (X = [@, A, B, ...])
   *     ^? for del char
   *     M-c for ascii codes > 0x3f
   * - prefix parents and backslash ['(', ')', '\'] by backslash
   *   (escape character in postcript)
   */
  if (job->status->is_in_cut 
      && (c != '\f' )
      && (c != encodings[job->encoding].new_line))
    return;
  job->status->is_in_cut = FALSE;

  /* Start a new line ? */
  if (job->status->start_line) {
    if (job->status->start_page) 
      {
	/* only if there is something to print! */
	ps_skip_page(job);
	job->status->start_page = FALSE ;
	/* This is the first line of a new page, hence we need (page
	 * independance) to repeat the current font */
	job->status->face = new_face;
	job->status->face_declared = FALSE;

	if (job->numbering) {
	  if (job->jobs->linenum % job->numbering == 0) {
	    if (job->jobs->linenum == job->status->last_line_num) { 
	      /* The line was broken by a \f */
	      output (jdiv, "(+) # (");
	      job->status->line_continued = FALSE;
	    } else
	      output (jdiv, "(%d) # (", job->jobs->linenum);
	    job->status->last_line_num = job->jobs->linenum;
	  } else
	    output (jdiv, "0 T (");
	} else
	  output_char (jdiv, '(');
      } 
    else
      {
	/* This is a new line, but not the first in the page */
	if (job->numbering) {
	  if (job->jobs->linenum % job->numbering == 0)
	    output (jdiv, "(%d) # (", job->jobs->linenum);
	  else
	    output (jdiv, "0 T (");
	  job->status->last_line_num = job->jobs->linenum;
	} else {
	  /* This is a new line, hence we can consider that there is
	   * no need to close the current font: just consider it is
	   * the new font, but not declared */
	  output_char (jdiv, '(');
	  /* Why the hell did I do this? */
	  /* FIXME: This is suppresed because of the changes of encoding */
	  job->status->face_declared = (job->status->face_declared
					&& (job->status->face == new_face));
	  job->status->face = new_face;
	}
      }
    job->status->start_line = FALSE;
  }
  
  /* Is a new font ? */
  if (job->status->face != new_face) {
    if (!job->status->face_declared) {
      output (jdiv, ") %s\n(", face_to_ps(job->status->face));
      job->status->face_declared = TRUE;
    } else
      output (jdiv, ") S\n(");
    job->status->face = new_face;
    job->status->face_declared = FALSE;
  }
  
  /* Interpret each character */
  switch (c) {
  case '\f':  		/* Form feed */
    if (!job->interpret)
      goto print;

    /* Close current line */
    if (!job->status->start_line) {
      EOL ();
      job->status->start_line = TRUE;
    }
    /* start a new page ? */
    if (job->status->start_page)
      ps_skip_page(job);
    /* Close current page and begin another */
    end_page (job);
    job->status->start_page = TRUE;
    /* Verification for binary files */
    job->status->line = 0;
    job->status->column = 0;
    job->status->wx = 0;
    break;

  case '\n':
  case '\r':		/* One of these is just a plain character */
    if (c != encodings[job->encoding].new_line)
      goto print;
    job->status->column = 0;
    job->status->wx = 0;
    job->status->start_line = TRUE;
    EOL ();

    if (job->status->line >= job->status->linesperpage) {
      end_page (job);
      job->status->start_page = TRUE ;
      job->status->line = 0;
    }
    break;

  case '\t':
    if (!job->interpret)
      goto print;
      
    /* Tabs are interpreted  but we want to go to the same
     * column as if the font were Courier
     */
    job->status->column = 
      (MAX(job->status->wx / COURIER_WX, job->status->column)
       / job->tabsize + 1) * job->tabsize;
    job->status->wx = job->status->column * COURIER_WX;
    if (line_full) {
      if (job->folding) {
	FOLD_LINE ();
	job->status->column = 0;
	job->status->wx = 0;
      } else {
	job->status->is_in_cut = TRUE;
	return;
      }
    }
    /* Make sure that the font is given */
    if (!job->status->face_declared) {
      output (jdiv, ") %s", face_to_ps(job->status->face));
      job->status->face_declared = TRUE;
    } else
      output (jdiv, ") S");
    output (jdiv, " %d T (", job->status->column);
    break;
  print:
  default:
    {
      uchar buf[256];
      int nchars;
      *buf = '\0';
      
      nchars = format_char(job, c, buf);
      job->status->wx += 
	((nchars == 1) ? char_WX(job, c) : string_WX(job, buf));
      job->status->column += nchars;
      if (line_full) {
	if (job->folding) {
	  FOLD_LINE ();
	  job->status->column = nchars;
	  job->status->wx = 
	    ((nchars == 1) ? char_WX(job, c) : string_WX(job, buf));
	} else {
	  job->status->is_in_cut = TRUE;
	  return;
	}
      }
      output (jdiv, "%s", buf);
      job->status->chars++;
    }
    break;
  }
}

void
ps_print_string (print_job * job, ustring string, FACE new_face)
{
  while (*string)
    ps_print_char (job, *(string++), new_face);
}

void
ps_end_file (print_job * job)
{
  if (!job->status->start_line)
    output (job->divertion, ") S\n");
  if (!job->status->start_page)
    end_page (job);
  
  /* If we are in compact mode and the sheet is not full,
   * end the current sheet */
  if (job->virtual > 0 && !job->compact_mode)
    clean_up (job);
  
  /* Set the number of pages/sheets printed */
  job->jobs->pages = job->total_pages - job->jobs->first_page;
  job->jobs->sheets = job->total_sheets - job->jobs->first_sheet;

  /* If we don't want to print binaries, complain and quit */
  check_binary_file (job);
}

/************************************************************************
 * 			Routines called by the main loop		*
 ************************************************************************/
/*
 * Called by the main loop to initialize postscript printing
 */
void
ps_init (print_job * job)
{
  float area_w, area_h;		/* Dimension of drawable area	*/
  float printing_h, printing_w;  	/* Body size of virtual pages	*/

  /* Postcript prolog printing */
  /* This will allow us to read information from the prologs,
   * such as the size they'd like to have to print the headers,
   * before we actually calculate the parameters */
  output_prolog (job);

  job->virtuals = job->columns * job->rows;
  job->virtual = job->virtuals; /* not very nice, huh? ;) */

  switch (job->orientation) {
  case portrait:
    area_h = (medias[job->paper].ury - medias[job->paper].lly
	      /* Room for header and footer */
	      - (PRINT_HEADER + PRINT_FOOTER) * HEADERS_H);
    area_w = (medias[job->paper].urx - medias[job->paper].llx
	      - job->margin);
    break;
  case landscape:
    area_w = (medias[job->paper].ury - medias[job->paper].lly);
    area_h = (medias[job->paper].urx - medias[job->paper].llx
	      /* Room for header and footer */
	      - (PRINT_HEADER + PRINT_FOOTER) * HEADERS_H
	      - job->margin);
    break;
  }

  /* Initialize variables related to the header */
  if (!PRINT_TITLE) {
    job->status->title_font_size = 0;
    job->status->title_bar_height = 0.0;
  } else if (job->virtuals > 1) {
    job->status->title_font_size = 11;
    job->status->title_bar_height = LANDSCAPE_HEADER;
  } else {
    job->status->title_font_size = 15;
    job->status->title_bar_height = PORTRAIT_HEADER;
  }
  
  /* Area inside the frame of a virtual page */
  printing_h = ((area_h
		 /* room for title */
		 - job->rows * job->status->title_bar_height
		 /* Space between the virtual pages */
		 - ((job->rows > 1) ? PAGE_MARGIN : 0))
		/ job->rows);
  printing_w = ((area_w 
		 /* Space between the virtual pages */
		 - ((job->columns > 1) ? PAGE_MARGIN : 0))
		/ job->columns);
  
  /* 
   * Determine the font size according to (decreasing priority):
   * 1. --font=@<size>
   * 2. --columns-per-page, 
   * 3. --lines-per-page,
   * 4. default values
   */
  /* width = 0.6 font size */
  if (job->columns_requested != 0) {
    job->fontsize = ((printing_w / 
		      (job->columns_requested + prefix_size + 
		       2 * SIDE_MARGIN_RATIO))
		     / 0.6);
  } else if (job->lines_requested != 0) {
    job->fontsize = (printing_h / 
		     (job->lines_requested + BOTTOM_MARGIN_RATIO));
  } else if (job->fontsize == 0.0)
    job->fontsize = ((job->orientation == landscape) ? 6.8 
		     : (job->virtuals > 1) ? 6.4 : 9.0);

  /* fontsize is OK.  Calculate the other variables */
  job->status->linesperpage = 
    (int) ((printing_h / job->fontsize) - BOTTOM_MARGIN_RATIO);
  job->status->columnsperline = 
    (int) ((printing_w / (job->fontsize * 0.6)) - 2 * SIDE_MARGIN_RATIO);

  if (job->columns_requested > 0)
    job->status->columnsperline = job->columns_requested + prefix_size;
  else if (job->lines_requested > 0)
    job->status->linesperpage = job->lines_requested;
  
  if (job->status->linesperpage <= 0 || job->status->columnsperline <= 0)
    error (1, 0, _("font %f too big"), job->fontsize);

  job->status->wxperline = 
    (job->status->columnsperline - prefix_size) * COURIER_WX;

  output_document_setup (job);
}

/*
 * Finish the poscript printing
 */
void
ps_close (print_job * job)
{
  int total;		/* Number of sheets printed */
  
  /* Close the postscript file */
  end_document (job);

  /* Undivert */
  undivert (job);

  /* Report the total number of pages printed */
  if (job->jobs->num != 1) {
    message (0, job->total_pages == 1 
	     ? _("[Total: %d page on ") : _("[Total: %d pages on "), 
	     job->total_pages);
    total = job->total_sheets;
    if (job->rectoverso)
      total = (total+1) / 2;
    message (0, "%d %s]\n", 
	     total, total == 1 ? _("sheet") : _("sheets"));
  }
}
