/*
 * pathwalk.c
 *
 * functions for ooking for files, reading files etc. 
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana
 * $Id: pathwalk.c,v 1.1 1997/03/16 18:19:55 demaille Exp $
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "pathwalk.h"
#include "message.h"
#include "error.h"
#include "pathwalk.h"

#include <stdio.h>

#if STDC_HEADERS

#  include <stdlib.h>

#else /* no STDC_HEADERS */

#  if HAVE_STDLIB_H
#    include <stdlib.h>
#  endif

#  ifndef HAVE_STRCHR
#    define strchr index
#  endif
char *strchr ();
#endif /* no STDC_HEADERS */

#ifdef HAVE_STRING_H
# if !STDC_HEADERS && HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#else
# include <strings.h>
#endif

#define strequ(s1, s2)          (strcmp ((s1), (s2)) == 0)
#define strnequ(s1, s2, n)      (strncmp ((s1), (s2), (n)) == 0)

#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif

#include <errno.h>

#ifndef _
/* This is for other GNU distributions with internationalized messages.
   When compiling libc, the _ macro is predefined.  */
#if defined (HAVE_LIBINTL_H) || defined (ENABLE_NLS)
# include <libintl.h>
# define _(msgid)       gettext (msgid)
#else
# define _(msgid)       (msgid)
#endif
#endif

/*
 * From savedir.c
 */
char * savedir PARAMS ((char *dir, unsigned name_size));

/*
 * Path walk functions
 */
typedef struct file_lookup_ctx_st {
  char name[256];
  char suffix[256];
  char fullname[512];
} FileLookupCtx;

typedef struct file_suffix_ctx_st {
  char * dirfmt;
  char * filefmt;
  char * enddir;
  char suffix[256];
} FileSuffixCtx;

typedef int (*PathWalkProc) PARAMS ((char *path, void *context));


/*
 * Walk through a path
 * This routine is taken from GNU enscript written by
 * Markku Rossi <mtr@iki.fr>
 */
static int
path_walk (char *path, PathWalkProc proc, void *context)
{
  char buf[512];
  char *cp;
  char *cp2;
  int len, i;
  
  message (3, _("path_walk() on path %s\n"), path);

  for (cp = path; cp; cp = strchr (cp, PATH_SEP))
    {
      if (cp != path)
	cp++;

      cp2 = strchr (cp, PATH_SEP);
      if (cp2)
	len = cp2 - cp;
      else
	len = strlen (cp);

      if (len == 0) {
	cp++;
	continue;
      }
      
      memcpy (buf, cp, len);
      buf[len] = '\0';

      i = (*proc) (buf, context);
      if (i != 0) {
	message (3, _("success in %s\n"), buf);
	return i;
      }
    }
  
  message (3, _("failure\n"));
  return 0;
}


/*
 * Return TRUE iff the file NAMESUFFIX is in the library path
 */
static int
file_lookup (char *path, void *context)
{
  int len;
  FileLookupCtx *ctx = context;
  struct stat stat_st;
  int i;

  message (2, _("file_lookup(): %s%c%s%s\t"), path, DIR_SEP, 
	   ctx->name, ctx->suffix);

  len = strlen (path);
  if (len && path[len - 1] == DIR_SEP)
    len--;

  sprintf (ctx->fullname, "%.*s%c%s%s", 
	   len, path, DIR_SEP, ctx->name, ctx->suffix);
  
  i = stat (ctx->fullname, &stat_st) == 0;

  message (2, "#%c\n", i ? 't' : 'f');
  return i;
}

int
pw_file_exists (char * path, const char *name, const char *suffix)
{
  FileLookupCtx ctx;

  strcpy (ctx.name, name);
  strcpy (ctx.suffix, suffix ? suffix : "");

  return path_walk (path, file_lookup, &ctx);
}

char *
pw_find_file (char * path, const char *name, const char *suffix)
{
  FileLookupCtx ctx;
  int res;

  strcpy (ctx.name, name);
  strcpy (ctx.suffix, suffix ? suffix : "");

  res = path_walk (path, file_lookup, &ctx);
  if (res == 0)
    return NULL;
  else
    return strdup (ctx.fullname);
}

/*
 * Print the files in the path that have SUFFIX as suffix.
 */
static int
suffix_lookup (char *path, void *context)
{
  FileSuffixCtx *ctx = context;
  char *entries, *cp, *cp2;

  message (2, _("suffix_lookup(): %s%c*.%s\n"), path, DIR_SEP, 
	   ctx->suffix);
  
  printf (ctx->dirfmt, path);
  
  cp = entries = savedir (path, 1024);
  if (cp) {
    while (*cp) {
      if ((cp2 = strrchr (cp, '.')))
	if (strequ (cp2 + 1, ctx->suffix))
	  printf (ctx->filefmt, cp);
      cp += strlen (cp) + 1;
    }
    free (entries);
  }
  fputs (ctx->enddir, stdout);
  
  return 0;
}

/*
 * Print the files in the path that have SUFFIX as suffix.
 */
int
pw_list_suffix (char * path, 
		char * dirfmt, char * filefmt, char * enddir, 
		const char *suffix)
{
  FileSuffixCtx ctx;
  
  strcpy (ctx.suffix, suffix);
  ctx.dirfmt = dirfmt;
  ctx.filefmt = filefmt;
  ctx.enddir = enddir;

  return path_walk (path, suffix_lookup, &ctx);
}

/*
 * Dump a library file content
 */
int
pw_paste_file (char * path, const char *name, const char *suffix)
{
  char buf[512];
  FILE *fp;
  FileLookupCtx ctx;
  int line = 0;

  strcpy (ctx.name, name);
  strcpy (ctx.suffix, suffix ? suffix : "");

  if (!path_walk (path, file_lookup, &ctx))
    return 0;
  fp = fopen (ctx.fullname, "r");
  if (fp == NULL)
    return 0;

  /* Find the end of the header. */
#define HDR_TAG "% -- code follows this line --"
  while ((fgets (buf, sizeof (buf), fp)))
    {
      line++;
      if (strnequ (buf, HDR_TAG, strlen (HDR_TAG)))
	break;
    }

  /* Dump rest of file. */
#define INCL_TAG "% -- include file:"
  while ((fgets (buf, sizeof (buf), fp)))
    {
      line++;
      if (strnequ (buf, INCL_TAG, strlen (INCL_TAG))) {
	char * file = buf + strlen (INCL_TAG);
	file = strtok (file, " \n\t");
	message (2, "Including file '%s' upon request given in '%s':%d\n", 
		 file, ctx.fullname, line);
	if (!pw_paste_file (path, file, NULL))
	  error_at_line (1, errno, ctx.fullname, line,
			 _("couldn't find file \"%s\""), file);
	continue;
      }
      fputs (buf, stdout);
    }

  fclose (fp);

  return 1;
}
