/*
 * output.c
 *
 * routines for ram-diverted output
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana
 * $Id: output.c,v 1.11.2.3 1997/06/16 14:56:52 demaille Exp $
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "a2ps.h"
#include "routines.h"
#include "output.h"
#include "assert.h"
#include "jobs.h"
#include "pathwalk.h"
#include <sys/wait.h>

#if HAVE_VPRINTF || HAVE_DOPRNT || _LIBC
# if __STDC__
#  include <stdarg.h>
#  define VA_START(args, lastarg) va_start(args, lastarg)
# else
#  include <varargs.h>
#  define VA_START(args, lastarg) va_start(args)
# endif
#else
# define va_alist a1, a2, a3, a4, a5, a6, a7, a8
# define va_dcl char *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8;
#endif


#define MIN_CONTENT	1024 * 50 	/* 50 Kb buffer			*/
#define CHUNK_MARGIN	1024		/* Be sure there is always that left */
#define MIN_DIVERTIONS	10

typedef enum {
  none, delayed_int, delayed_routine, delayed_chunk
} divertion_type;

typedef struct derivation {
  divertion_type type;
  void * arg;
  void (*delayed_routine) PARAMS ((struct print_job * job));
  int * ptr_int;
  struct chunk * delayed_chunk;
} derivation;

typedef struct chunk {
  ustring content;
  unsigned int size;
  unsigned int len;
  derivation * divertions;
  int nb_divertions;
  int size_divertions;
  int to_void; 	/* if true, what is sent here is forgotten	*/
} chunk;


static void
more_content (chunk *a_chunk)
{
  message (2, "Doubling chunk size (%p from %d): ", 
	   a_chunk->content, a_chunk->size);

  a_chunk->size *= 2;
  a_chunk->content = REALLOC (a_chunk->content, uchar, a_chunk->size);
  
  message (2, "(%p with %d bytes)\n", 
	   a_chunk->content, a_chunk->size);
}

static void
more_divertions (chunk * a_chunk)
{
  int first_new = a_chunk->size_divertions;
  int i;

  message (2, "Doubling num of divertions (from %d)\n", 
	   a_chunk->size_divertions);
  a_chunk->size_divertions *= 2;
  a_chunk->divertions = REALLOC (a_chunk->divertions, 
				 derivation, a_chunk->size_divertions);
  for (i = first_new ; i < a_chunk->size_divertions ; i++)
    a_chunk->divertions[i].type = none;

}

void
#if defined(VA_START) && __STDC__
output (chunk * div, const char *format, ...)
#else
output (div, format, va_alist)
  chunk * div;
  char * format;
  va_dcl
#endif
{
#ifdef VA_START
  va_list args;

  VA_START (args, format);
#endif

  if (div->to_void)
    return;

  if (div->size < div->len + CHUNK_MARGIN)
    more_content(div);

#ifdef VA_START
  vsprintf ((char *)div->content + div->len, format, args);
  va_end (args);
#else
  vsprintf ((char *)div->content + div->len, format, a1, a2, a3, a4, a5, a6, a7, a8);
#endif

  div->len += ustrlen (div->content + div->len);
} 

void
output_char (chunk * div, uchar c)
{
  if (div->to_void)
    return;

  if (div->len >= div->size)
    more_content (div);
  div->content[(div->len)++] = c;
}

void
output_delayed_int (chunk * div, int * ptr)
{
  if (div->to_void)
    return;

  if (div->nb_divertions == div->size_divertions - 1)
    more_divertions (div);
  output_char (div, '\0');
  div->divertions [div->nb_divertions].type = delayed_int;
  div->divertions [div->nb_divertions++].ptr_int = ptr;
}

/*
 * Dump a library file content
 */
void
output_file (chunk * div, print_job * job, 
	     const char *name, const char *suffix)
{
  char buf[512];
  FILE *fp;
  char * file;
  char * filename = MALLOC (strlen (name) + strlen (UNNULL (suffix)) + 1);
  int line = 0;
  int dont_output = FALSE;
  chunk * dest = div;

  if (div->to_void)
    return;

  sprintf (filename, "%s%s", name, UNNULL (suffix));
  message (3, "Outputing file %s\n", filename);

  file = pw_find_file (job->lib_path, name, suffix);
  if (!file)
    error (1, errno, _("couldn't find file \"%s\""), filename);

  fp = fopen (file, "r");
  if (fp == NULL)
    error (1, errno, _("couldn't open file \"%s\""), file);
  
  /* Find the end of the header. */
#define HDR_TAG "% -- code follows this line --"
  while ((fgets (buf, sizeof (buf), fp))) {
    line++;
    if (strprefix (HDR_TAG, buf))
      break;
  }
  
  /* Dump rest of file. */
  while ((fgets (buf, sizeof (buf), fp))) {
    line++;

#define END_FONTUSED_TAG	"%%EndFontUsed"
    if (strprefix (END_FONTUSED_TAG, buf))
      {
	dont_output = FALSE;
	continue;
      }

    else if (dont_output)
      continue;

#define FONTUSED_TAG		"%%IfFontUsed:"
    /* After this tag, forget unless the font is used.
     * This is for small memory printers */
    else if (strprefix (FONTUSED_TAG, buf))
      {
	char * fontname;
	FONT font;
	
	fontname =  buf + strlen (FONTUSED_TAG);
	/* What is the face declared? */
	fontname = strtok (fontname, " \t\n");
	font = string_to_font (fontname);
	
	if (font == unknown_font)
	  error_at_line (1, 0, filename, line,
			 _("unrecognized font: `%s'"), fontname);
	if (!(job->status->font_used [font]))
	  dont_output = TRUE;
	continue;
      }
    
#define INCL_RES_TAG 	"%%IncludeResource:"
    if (strprefix (INCL_RES_TAG, buf)) 
      {
	char * value, * res, * buf2;
	buf2 = xstrdup (buf);
	res = buf + strlen (INCL_RES_TAG);
	res = strtok (res, " \n\t");
	if (strequ (res, "file")) {
	  /* We want to include a file only once */
	  value = strtok (res + strlen (res) + 1, " \n\t");
	  if (!exist_resource (job, res, value)) {
	    add_needed_resource (job, res, value);
	    message (2, "Including file '%s' upon request given in '%s':%d\n", 
		     value, filename, line);
	    output_file (dest, job, value, NULL);
	  }
	} else {
	  output (dest, "%s", buf2);
	  while ((value = strtok (NULL, " \n\t"))) 
	    add_needed_resource (job, res, value);
	}
	XFREE (buf2);
	continue;
      }
#define COLOR_TAG 	"%%DocumentProcessColors:"
    else if (strprefix (COLOR_TAG, buf)) 
      {
	char * color;
	color = buf + strlen (COLOR_TAG);
	color = strtok (color, " \n\t");
	add_process_color (job, color);
	while ((color = strtok (NULL, " \n\t")))
	  add_process_color (job, color);
	continue;
      }
#define SUPP_RES_TAG 	"%%BeginResource:"
    else if (strprefix (SUPP_RES_TAG, buf))
      {
	char * res, * value;
	output (dest, "%s", buf);
	res = buf + strlen (SUPP_RES_TAG);
	res = strtok (res, " \n\t");
	value = strtok (NULL, "\n");
	add_supplied_resource (job, res, value);
	continue;
      }

#define END_SETUP_TAG	"%%EndSetup"
    else if (strprefix (END_SETUP_TAG, buf))
      {
	if (dest == div)
	  error (1, 0, "`setup' incoherence in output_file");
	dest = div;
	continue;
      }
    
#define SETUP_TAG 	"%%BeginSetup"
    else if (strprefix (SETUP_TAG, buf))
      {
	dest = job->setup;
	continue;
      }

#define FONT_TAG "%Face:"
    else if (strprefix (FONT_TAG, buf))
      {
	char * logical, *physical;
	int log, phy;
	logical =  buf + strlen (FONT_TAG);
	/* What is the face declared? */
	logical = strtok (logical, " \t");
	log = string_to_face (logical);
	if (log == NO_FACE)
	  error_at_line (1, 0, filename, line,
			 _("unrecognized face: `%s'"), logical);
	/* What is the corresponding physical font? */
	physical = strtok (NULL, " \t\n");
	phy = string_to_font (physical);
	if (phy == unknown_font)
	  error_at_line (1, 0, filename, line,
			 _("unrecognized font: `%s'"), physical);
	job->face_to_font [log] = phy;
	job->status->font_used [phy] = TRUE;
	continue;
      }

    output (dest, "%s", buf);
  }

  if (dest != div)
    /* E.g. `%%BeginSetup' with no matching `%%EndSetup' */
    error_at_line (1, 0, filename, line,
		   _("`%s' with no matching `%s'"),
		   SETUP_TAG, END_SETUP_TAG);
  
  fclose (fp);
  XFREE (file);
  XFREE (filename);
}

void
output_delayed_routine (chunk * div, 
			void (*fn) (print_job * job), 
			void * arg)
{
  if (div->to_void)
    return;
  if (div->nb_divertions == div->size_divertions - 1)
    more_divertions (div);
  output_char (div, '\0');
  div->divertions [div->nb_divertions].type = delayed_routine;
  div->divertions [div->nb_divertions].delayed_routine = fn;
  div->divertions [div->nb_divertions++].arg = arg;
}

void
output_delayed_chunk (chunk * div, chunk * div2)
{
  if (div->to_void)
    return;
  if (div->nb_divertions == div->size_divertions - 1)
    more_divertions (div);
  output_char (div, '\0');
  div->divertions [div->nb_divertions].type = delayed_chunk;
  div->divertions [div->nb_divertions++].delayed_chunk = div2;
}

/*
 * Create a new divertion
 */
chunk *
new_divertion (void)
{
  int i;
  chunk * res = NULL;
  
  res = ALLOC (chunk, 1);
  res->size = MIN_CONTENT;
  res->content = ALLOC (uchar, res->size);
  res->content[0] = NUL;
  res->len = 0;
  /* Divertions */
  res->size_divertions = MIN_DIVERTIONS;
  res->divertions = ALLOC (derivation, res->size_divertions);
  res->nb_divertions = 0;
  for (i = 0 ; i < res->size_divertions ; i++)
    res->divertions[i].type = none;
  res->to_void = FALSE;
  
  return res;
}

/*
 * Should the divertion forget what it receives?
 */
void
divertion_to_void (chunk * div, int forget)
{
  div->to_void = forget;
}

void
underivation (chunk * div)
{
  int i;
  ustring piece = div->content;

  message (3, "Divertions stats:\n\
  nb_divertions: %d\n\
  chunk_size: %d\n\
  chunk_len: %d (%.2f%%)\n", 
	   div->nb_divertions, div->size, div->len, 
	   (float) div->len/div->size * 100.0);
  
   for (i = 0 ; i <= div->nb_divertions ; i++) {
    fputs ((char *) piece, stdout);
    piece += ustrlen (piece);
    switch (div->divertions[i].type) {
    case delayed_routine:
      div->divertions[i].delayed_routine (div->divertions[i].arg);
      piece ++;
      break;
    case delayed_int:
      printf ("%d", *div->divertions[i].ptr_int);
      piece ++;
      break;
    case delayed_chunk:
      underivation (div->divertions[i].delayed_chunk);
      piece ++;
      break;
    case none:
      piece ++;
      break;
    default:
      error (1, 0, "undivert");
    }
   }
}

void
undivert (print_job * job)
{
  pid_t pid;
  int state;
  
  /* Open the destination */
  if (job->lpr_print) 
    stdout_into_printer (job);
  else if (!IS_EMPTY(job->output_filename))
    stdout_into_file (job->output_filename);
  
  /* Dump the PostScript and close */
  underivation (job->divertion);
  fclose (stdout);
  
  /* Wait as long as there are childs */
  if (job->lpr_print)
    while ((pid = wait (&state)) != -1)
      /* empty */;
}
