/*
 * jobs.h
 *
 * Recording information about the print jobs
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana
 * $Id: jobs.h,v 1.14.2.2 1997/05/26 08:25:18 demaille Exp $
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef _JOBS_H_
#define _JOBS_H_

#include "encoding.h"
#include "output.h"
#include "fonts.h"
#include "confg.h"
#include "medias.h"

/* Definition of the hash'd structure used for page device definitions */
typedef struct dict_entry {
  char *key;
  char *value;
  int def;
} dict_entry;

typedef struct multivalued_entry {
  char * key;
  int size;
  int len;
  char ** content;
} multivalued_entry;


typedef struct hash_table hash_table;
void dict_entry_print PARAMS ((void const * item));

/* First horinzontally, or vertically? */
typedef enum {
  major_rows, major_columns
} MAJOR;


/*
 * setpagedevice
 */
void output_pagedevice PARAMS ((struct print_job * job));
void output_requirements PARAMS ((struct print_job * job));
void setpagedevice PARAMS ((struct print_job * job, char * key, char * value));
void delpagedevice PARAMS ((struct print_job * job, char * key));

/*
 * statusdict
 */
void output_statusdict PARAMS ((struct print_job * job));
void setstatusdict PARAMS ((struct print_job * job, char * key, char * value, int def));
void delstatusdict PARAMS ((struct print_job * job, char * key));

/*
 * Printers
 */
void add_printer PARAMS ((struct print_job * job, char * key, char * value));
void list_printers PARAMS ((struct print_job * job));
void stdout_into_pipe PARAMS ((char * command));
void stdout_into_file PARAMS ((char * file));
void stdout_into_printer PARAMS ((struct print_job * job));

/*
 * User options
 */
void add_user_option PARAMS ((struct print_job * job, char * key, char * value));
char * get_user_option PARAMS ((struct print_job * job, char * key));
void list_user_options PARAMS ((struct print_job * job));


/*
 * Pattern rules
 */
void add_pattern_rule PARAMS ((struct print_job * job, char * lang, char * pattern));
char * get_style_sheet_name PARAMS ((struct print_job * job, char * filename));

/*
 * needed ps resources (such as fonts)
 */
void add_needed_resource PARAMS ((struct print_job * job, char * key, char * value));
int exist_resource PARAMS ((struct print_job * job, char * key, char * value));
void add_process_color PARAMS ((struct print_job * job, char * value));
void dump_needed_resources PARAMS ((struct print_job * job));
void add_needed_resource PARAMS ((struct print_job * job, char * key, char * value));
void add_supplied_resource PARAMS ((struct print_job * job, char * key, char * value));
void dump_supplied_resources PARAMS ((struct print_job * job));

/*
 * Declaration of the fonts, etc.
 */
void dump_setup PARAMS ((struct print_job * job));

/* Relative to the current file */
typedef struct file_job {
  int first_sheet;		/* num of the first sheet for this file	*/
  int first_page;		/* Id. but pages			*/
  int pages;
  int sheets;
  int num;			/* No of the file in the args		*/
  int toplinenum;		/* The top most line of the current page */
  int linenum;			/* Current line number			*/
  ustring name;			/* Guess what it is :)			*/
  struct tm mod_tm;		/* Modif. info for the file 		*/
  
  struct file_job *next_job;
} file_job;

/* Information that should be kept private to the postscript
 * generating engine of a2ps */
typedef struct ps_status {
  /* Set once at the beginning */
  unsigned int columnsperline;	/* Characters per output line 		*/
  unsigned long wxperline;	/* Points per output line		*/
  int linesperpage;		/* Lines per page 			*/
  float title_bar_height;	/* Height of the bar for v. pages title */
  int title_font_size;		/* Font size for the main title		*/

  /* Changes all the time */
  int start_page;
  int start_line;
  int line_continued;
  int is_in_cut;		/* Do we have to skip the incoming text	*/
  FACE face;
  int face_declared;		/* Does the ps knows the current font?	*/
  int nonprinting_chars;
  int chars;			/* Number of nonprinting and total chars */
  int line;			/* Line number (in current page) 	*/
  unsigned int column;		/* Column number in chars 		*/
  unsigned long wx;		/* Column number in pts (in current line) */
  int last_line_num;
  unsigned int ** current_wx;	/* What are the wx of the current
				 * encoding?				*/
  int font_used[NB_FONTS];	/* Since we do not want to put too
				 * much in the ps, remind what has been
				 * used					*/
} ps_status;

/* Information that is used only by the a2ps program (not the library)
 * but since now the library reads the config files and hence the options,
 * we need a place where to store them which does not depend on whether
 * a2ps (prog) is used or not */
typedef struct a2ps_status {
  char * style;			/* -k, What is the current style?	*/
  int automatic_style;		/* Should the language be guessed? 	*/
  int translate_symbols_request;/* -g symbol convertion asked?		*/
  BEHAVIOR behavior;		/* What should be done at exec time?	*/
  hash_table * pattern_rules;	/* To guess style sheet to use		*/
  int strip;			/* Don't print comments?		*/
} a2ps_status;

/* All the information on the current print job */
typedef struct print_job {
  /* Private information for the a2ps program	*/
  a2ps_status * a2ps_stat;

  /* Path to the files used by a2ps */
  char lib_path [1024];

  /* User information */
  char * home;
  char * pw_gecos;
  char * pw_name;
  hash_table * user_options; 	/* Short cuts defined by the user	*/

  /* Relative to the whole process */
  struct tm run_tm; 		/* Time when this program is run	*/
  char datestring[256];		/* Full today's date			*/
  int total_sheets;		/* number of sheets before the curr file*/
  int total_pages;		/* id. but pages			*/
  int total_files;		/* id but files				*/
  ORIENTATION orientation;	/* landscape mode ?			*/
  int rectoverso;		/* Recto verso printing ? 		*/
  int columns;			/* number of columns/page		*/
  int rows;			/* number of rows/page			*/
  MAJOR Major;			/* firt rows, or columns?
				 * (not major, because of a SunOS macro)*/
  int virtuals;			/* number of virtual pages/page		*/
  int virtual;			/* number of the virtual page on that sheet */
  int copies;			/* number of copies			*/
  int margin;			/* margin to leave for binding		*/

  char * magic_number; 		/* The very first line of a PS file	*/
  hash_table * needed_resources;/* such as fonts etc.			*/
  hash_table * supplied_resources;/* such as fonts etc.			*/
  hash_table * colors;		/* such as fonts etc.			*/
  hash_table * pagedevice;	/* for PS setpagedevice			*/
  hash_table * statusdict;	/* For PS statusdict definitions	*/
  struct chunk * setup;		/* Files read may have sth to put in the
				 * set up part of the ps file		*/
  int * used_encodings;		/* What are the used encodings?		*/
  struct chunk * ps_encodings; 	/* PS definition of the encodings used	*/
  int page_prefeed;		/* Page prefeed mecanism		*/
  /* How to get the physical font (eg Courier), from the logical
   * (eg, Keyword) */
  FONT face_to_font[NB_FACES];

  /*
   * Flags related to options.
   */
  ustring stdin_title;		/* Filename given to stdin		*/
  hash_table * printers;	/* The user defined available printers	*/
  int lpr_print; 		/* Print? 	 			*/
  char *printer;		/* Options to lpr 			*/
  char *output_filename;	/* NULL -> stdout unless lpr_print	*/
  int folding;			/* Line folding option 			*/
  int numbering;		/* Line numbering option 		*/
  int only_printable;		/*Replace non printable char by space 	*/
  int interpret;		/* Interpret TAB, FF and BS chars option */
  int print_binaries;		/* Force printing for binary files 	*/
  int compact_mode;		/* Allow 2 files per sheet 		*/
  int border;			/* print the surrounding border ?	*/
  int area;			/* Draw the BoundingBox ? 		*/
  char * prolog;		/* postscript header file 		*/
  int paper;			/* Media to use 			*/
  int tabsize;			/* length of tabulations		*/
  int lines_requested;		/* Number of line per page 		*/
  int columns_requested;	/* Number of columns per page		*/
  float fontsize;		/* Size of a char for body font 	*/
  ENCODING encoding; 		/* What is the current char set ?	*/
  ENCODING requested_encoding; 	/* Given by the options			*/
  ENCODING saved_encoding; 	/* Given by the options			*/

  /* Headers and footers */
  ustring header;		/* Allow different header text	 	*/
  ustring center_title;		/* Allow different header text	 	*/
  ustring left_title;		/* Allow different header text	 	*/
  ustring right_title;		/* Allow different header text	 	*/
  ustring left_footer;		/* Allow different header text	 	*/
  ustring footer;		/* Allow different header text	 	*/
  ustring right_footer;		/* Allow different footer text	 	*/
  ustring water;		/* Water marks text			*/
  /* Used to grab headers etc. from the file */
  uchar tag1[256], tag2[256], tag3[256], tag4[256];

  /* Private information on the PS generation engine */
  ps_status * status;

  /* Where the output is diverted */
  struct chunk * divertion;

  /* List of the jobs */
  struct file_job * jobs;
} print_job;

print_job * new_print_job PARAMS ((void));
file_job * new_job PARAMS ((print_job * job, char * name));

#endif
