/*
 * jobs.c
 *
 * Recording information about the print jobs
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "a2ps.h"
#include "jobs.h"
#include "routines.h"
#include "xstrrpl.h"
#include "xfnmatch.h"
#include "hashtab.h"

/*
 * Used with the page device(-D), and status dict (-S) definitions 
 */
unsigned long
key_hash_1 (void const *key)
{
  return_STRING_HASH_1 (((dict_entry *)key)->key);
}

unsigned long
key_hash_2 (void const *key)
{
  return_STRING_HASH_2 (((dict_entry *)key)->key);
}

int
key_hash_cmp (void const *x, void const *y)
{
  return_STRING_COMPARE (((dict_entry *)x)->key, ((dict_entry *)y)->key);
}

int
key_qsort_cmp (void const *x, void const *y)
{
  return_STRING_COMPARE (((dict_entry *)x)->key, ((dict_entry *)y)->key);
}

/*
 * Used with the page device definitions (-D)
 */
void
dict_entry_print (void const * item)
{
  dict_entry * tok = (dict_entry *) item;
  if (tok->def) 
    printf ("%s::%s ", tok->key, tok->value);
  else
    printf ("%s:%s ", tok->key, tok->value);
}

void
hash_dump_pagedevice (void const * item)
{
  dict_entry * tok = (dict_entry *) item;
  printf ("%%%%BeginFeature: *%s %c%s\n", 
	  tok->key, toupper (tok->value[0]), tok->value + 1);
  printf ("  << /%s %s >> setpagedevice\n",
	   tok->key, tok->value);
  printf ("%%%%EndFeature\n");
}

/* Page device definitions */
void
output_pagedevice (print_job * job)
{
  dict_entry ** entries = NULL;
  dict_entry ** entry;
  entries = (dict_entry **) hash_dump (job->pagedevice, NULL, NULL);

  if (*entries || job->rectoverso) {
    output (job->divertion, "%% Pagedevice definitions:\n");
    output (job->divertion, "gs_languagelevel 1 gt {\n");
    for ( entry = entries ; *entry ; entry++) {
      output (job->divertion, "%%%%BeginFeature: *%s %c%s\n", 
	      (*entry)->key, toupper ((*entry)->value[0]), 
	      (*entry)->value + 1);
      output (job->divertion, "  << /%s %s >> setpagedevice\n",
	      (*entry)->key, (*entry)->value);
      output (job->divertion, "%%%%EndFeature\n");
    }
    if (job->rectoverso) {
      output (job->divertion, "%%%%BeginFeature: *Duplex True\n");
      output (job->divertion, "  << /Duplex true >> setpagedevice\n");
      output (job->divertion, "%%%%EndFeature\n");
    }
    output (job->divertion, "} if\n");
  }
  
  XFREE (entries);
}

void
output_requirements (print_job * job)
{
  dict_entry ** entries = NULL;
  dict_entry ** entry;
  entries = (dict_entry **) hash_dump (job->pagedevice, NULL, NULL);
  /* Dump only if there is something to say */
  if (*entries || job->rectoverso)
    {
      output (job->divertion, "%%%%Requirements: ");
      for ( entry = entries ; *entry ; entry++)
	output (job->divertion, "%s ", (*entry)->key);
      if (job->rectoverso)
	output (job->divertion, "duplex ");
      output_char (job->divertion, '\n');
    }
  /* We don't want this one which breaks some collating systems
     output (job->divertion, "numcopies(%d)", job->copies);
     */
  XFREE (entries);
}

void
setpagedevice (print_job * job, char * key, char * value)
{
  dict_entry * item = ALLOC (dict_entry, 1);
  item->key = key;
  item->value = value;
  item->def = FALSE;
  hash_insert (job->pagedevice, item);
}

void
delpagedevice (print_job * job, char * key)
{
  dict_entry * item = ALLOC (dict_entry, 1);
  item->key = key;
  hash_delete (job->pagedevice, item);
  XFREE (item);
}

/*
 * Used with the status dict definitions (-S)
 */
void
setstatusdict (print_job * job, char * key, char * value, int def)
{
  dict_entry * item = ALLOC (dict_entry, 1);
  item->key = key;
  item->value = value;
  item->def = def;
  hash_insert (job->statusdict, item);
}

void
delstatusdict (print_job * job, char * key)
{
  dict_entry * item = ALLOC (dict_entry, 1);
  item->key = key;
  hash_delete (job->statusdict, item);
  XFREE (item);
}

void
output_statusdict (print_job * job)
{
  dict_entry ** entries = NULL;
  dict_entry ** entry;

  entries = (dict_entry **) hash_dump (job->statusdict, NULL, NULL);
  if (*entries) {
    output (job->divertion, "%% Statustdict definitions:\n");
    output (job->divertion, "statusdict begin\n");
    for ( entry = entries ; *entry ; entry++)
      if ((*entry)->def) 
	output (job->divertion, "  /%s %s def\n",
		(*entry)->key, (*entry)->value);
      else
	output (job->divertion, "  %s %s\n", 
		(*entry)->value, (*entry)->key);
    output (job->divertion, "end\n");
  }
  XFREE (entries);
}
/*
 *  Printers management
 */
void
add_printer (print_job * job, char * key, char * value)
{
  dict_entry * item = ALLOC (dict_entry, 1);
  item->key = key;
  item->value = value;
  hash_insert (job->printers, item);
}

/*
 * Retrieve command associated with a printer, (or default printer
 * if no printer given) and if SUBSTITUTE then perform
 * all the needed substitutions (%s for -P<printer>)
 */
char *
get_printer_command (print_job * job, char * printer_name, int substitute)
{
  char * res = NULL;
  dict_entry * item;
  dict_entry token;
  hash_table * printers = job->printers;

  switch (IS_EMPTY (printer_name)) {
  case FALSE:	/* A printer name is given: this is -P<printer> */
    token.key = printer_name;
    item = (dict_entry *) hash_find_item (printers, &token);
    
    if (item == NULL) {
      /* The name of a printer is given, but it is not known */
      token.key = ":";
      item = (dict_entry *) hash_find_item (printers, &token);
      if (item == NULL)
	error (1, 0, _("no default command for unknown printer `%s'"), 
	       printer_name);
    }

    if (substitute)
      /* Perform a substitution of %s in res */
      res = xvstrrpl (item->value, "%s", printer_name, "%%", "%", NULL);
    else
      res = xstrdup (item->value);
    break;
    
  case TRUE:	/* No name of printer is given, this is -d */
    token.key = "::";
    item = (dict_entry *) hash_find_item (printers, &token);
    
    if (item == NULL)
      error (1, 0, _("no default command for option `-d'"));
    
    res = xstrdup (item->value);
    
    break;
  }
  return res;
}

void
dump_printer_command (char * printer, char * command)
{
  switch (*command) {
  case '|':
    printf (_("  %-20s = pipe in %s\n"), printer, command + 1);
    break;
  case '>':
    printf (_("  %-20s = save in %s\n"), printer, command + 1);
    break;
  default:
    error (1, 0, "printers");
  }
}

/*
 * Report only the regular printers (not void nor unknwon)
 */
void
dump_printer (void const * item)
{
  dict_entry * tok = (dict_entry *) item;
  
  if (strequ (tok->key, ":") || strequ (tok->key, "::"))
    return;
  
  dump_printer_command (tok->key, tok->value);
}

void
list_printers (print_job * job)
{
  char * command;

  command = get_printer_command (job, "::", FALSE);
  dump_printer_command (_("Default printer"), command);
  XFREE (command);

  command = get_printer_command (job, ":", FALSE);
  dump_printer_command (_("Unknown printer"), command);
  XFREE (command);

  hash_map (job->printers, dump_printer);
}

/*
 * Open a pipe on the given command, bind its entry to our stdout
 */
void
stdout_into_pipe (char * command)
{
  int fd[2];
  message (2, "Opening a pipe on `%s'\n", command);
  
  /* Start the process */
  if (pipe (fd) == -1)
    error (1, errno, _("unable to open a pipe"));
  if (fork () == 0) {
    dup2 (fd[0], 0);
    close (fd[0]); 
    close (fd[1]);
    execl ("/bin/sh", "sh", "-c", command, NULL);
    error (1, errno, _("error starting `/bin/sh -c %s'"), command);
  }
  dup2 (fd[1], 1);
  close (fd[0]);
  close (fd[1]);
}

/*
 * Redirect stdout into a file
 */
void
stdout_into_file (char * file)
{
  if (freopen(file, "w", stdout) == NULL)
    error (1, errno, _("unable to create output file `%s'"), 
	   file);
}

/*
 * Redirect stdout into the destination associated to job->printer
 */
void
stdout_into_printer (print_job * job)
{
  char * lp_command;

  lp_command = get_printer_command (job, job->printer, TRUE);

  switch (*lp_command) {
  case '|':	/* Pipe */
    stdout_into_pipe (lp_command + 1);
    break;
  case '>':	/* File */
    stdout_into_file (lp_command + strspn (lp_command, "\t >"));
    break;
  }
  XFREE (lp_command);
}

/*
 *  User options management
 */
/*
 * Add a custom option
 */
void
add_user_option (print_job * job, char * key, char * value)
{
  dict_entry * item = ALLOC (dict_entry, 1);
  item->key = key;
  item->value = value;
  hash_insert (job->user_options, item);
}

/*
 * Retrieve a user option
 */
char *
get_user_option (print_job * job, char * shortcut)
{
  dict_entry * item;
  dict_entry token;

  token.key = shortcut;
  item = (dict_entry *) hash_find_item (job->user_options, &token);
  
  if (item == NULL)
    error (1, 0, _("user option `%s' not defined"), shortcut);
    
  return item->value;
}

/*
 * Dump one user option
 */
void
dump_user_option (void const * item)
{
  dict_entry * tok = (dict_entry *) item;
  
  printf ("  %-10s = %s\n", tok->key, tok->value);
}

/*
 * List the values defined
 */
void
list_user_options (print_job * job)
{
  hash_map (job->user_options, dump_user_option);
}

/*
 * Information which relates the filename to a style sheet.
 * It's a pity this goes into the library (since liba2ps
 * does no use of the style sheets), but since this kind
 * of info is stored in the very same config files that
 * the one used in a2ps-the-program, the lib must handle
 * this
 */
void
add_pattern_rule (print_job * job, char * lang, char * pattern)
{
  dict_entry * item = ALLOC (dict_entry, 1);
  item->key = pattern;
  item->value = lang;
  hash_insert (job->a2ps_stat->pattern_rules, item);
}
/*
 * Get style sheet name from file name, using pattern rules
 */
char *
get_style_sheet_name (print_job * job, char * filename)
{
  dict_entry ** entries = NULL;
  dict_entry ** entry;

  entries = (dict_entry **) hash_dump (job->a2ps_stat->pattern_rules,
				       NULL, NULL);

  for (entry = entries ; *entry ; entry++)
    if (!fnmatch ((*entry)->key, filename, 0))
      return (*entry)->value;

  XFREE (entries);
  return "plain";
}

/************************************************************************
 * Multivalued hash tables						*
 ************************************************************************/
/*
 * Used with the page device(-D), and status dict (-S) definitions 
 */
unsigned long
mv_key_hash_1 (void const *key)
{
  return_STRING_HASH_1 (((multivalued_entry *)key)->key);
}

unsigned long
mv_key_hash_2 (void const *key)
{
  return_STRING_HASH_2 (((multivalued_entry *)key)->key);
}

int
mv_key_hash_cmp (void const *x, void const *y)
{
  return_STRING_COMPARE (((multivalued_entry *)x)->key,
			 ((multivalued_entry *)y)->key);
}

/*
 * Is this resource already recorded?
 */
int
exist_resource (print_job * job, char * key, char * value)
{
  int i;
  multivalued_entry * item;
  multivalued_entry token;
  hash_table * resources = job->needed_resources;

  token.key = key;
  item = (multivalued_entry *) hash_find_item (resources, &token);

  if (item != NULL)
    for (i = 0 ; i < item->len ; i++)
      if (strequ (item->content [i], value))
	return TRUE;

  return FALSE;
}
/*
 * Used to record the requirements needed
 */
static void
add_resource (hash_table * resources, char * key, char * value)
{
  int i;
  multivalued_entry * item;
  multivalued_entry token;

  token.key = key;
  item = (multivalued_entry *) hash_find_item (resources, &token);
  if (item == NULL) {
    item = ALLOC (multivalued_entry, 1);
    item->key = xstrdup (key);
    item->size = 10;
    item->len = 0;
    item->content = ALLOC (char *, item->size);
    hash_insert (resources, item);
  }
  /* Requirements need to be recorded only once */
  for (i = 0 ; i < item->len ; i++)
    if (strequ (item->content [i], value))
      return;

  if (item->len + 1 >= item->size) {
    item->size *= 2;
    item->content = REALLOC (item->content, char *, item->size);
  }
  item->content [item->len] = xstrdup (value);
  item->len++;
}

/*
 * Used to record the requirements needed
 */
void
add_supplied_resource (print_job * job, char * key, char * value)
{
  add_resource (job->supplied_resources, key, value);
}

void
hash_dump_supplied_resources (void const * the_item)
{
  int i;
  multivalued_entry * item = (multivalued_entry *) the_item;
  printf ("%%%%DocumentSuppliedResources: %s %s\n",
	   item->key, item->content [0]);
  for (i = 1; i < item->len ; i++)
    printf ("%%%%+ %s %s\n", item->key, item->content [i]);
}

void
dump_supplied_resources (print_job * job)
{
  hash_map (job->supplied_resources, hash_dump_supplied_resources);
}

/*
 * Used to record the requirements needed
 */
void
add_needed_resource (print_job * job, char * key, char * value)
{
  add_resource (job->needed_resources, key, value);
}

void
hash_dump_needed_resources (void const * the_item)
{
  int i;
  multivalued_entry * item = (multivalued_entry *) the_item;
  
  if (strequ (item->key, "color")) {
    printf ("%%%%DocumentProcessColors: ");
    for (i = 0; i < item->len ; i++)
      printf ("%s ", item->content [i]);
    printf ("\n");
  } else if (strequ (item->key, "file")) {
    /* Do not report the files (they are yet included */
  } else {
    printf ("%%%%DocumentNeededResources: %s %s\n",
	    item->key, item->content [0]);
    for (i = 1; i < item->len ; i++)
      printf ("%%%%+ %s %s\n", item->key, item->content [i]);
  }
}

void
dump_needed_resources (print_job * job)
{
  hash_map (job->needed_resources, hash_dump_needed_resources);
}

/*
 * Colors used by the document
 */

void
add_process_color (print_job * job, char * value)
{
  add_resource (job->needed_resources, "color", value);
}

/*
 * Dump the setup code read in the various prologue (.pro and .ps)
 * files.  The hard part is that we don't want to dump too
 * many definitions of fonts, to avoid running out of memory on
 * too old PS level 1 printers.
 * Nevertheless, I still wait for somebody to tell me if this is
 * really needed (useful is sure, needed is not)
 */
void 
dump_setup (print_job * job)
{
  underivation (job->setup);
}

/* 
 * Private information for the PS generation engine 
 */
ps_status *
new_ps_status (void)
{
  int f;
  ps_status * res = ALLOC (ps_status, 1);
  for (f = 0 ; f < NB_FONTS ; f++)
    res->font_used [f] = FALSE;
  return res;
}

void
initialize_ps_status (ps_status * status)
{
  /* Reinitialize the ps status */
  status->start_page = TRUE;
  status->start_line = TRUE;
  status->line_continued = FALSE;
  status->is_in_cut = FALSE;
  status->face = PLAIN;
  status->face_declared = FALSE;
  status->nonprinting_chars = 0;
  status->chars = 0;
  status->line = 0;
  status->column = 0;
  status->wx = 0;
  status->last_line_num = 0;
}

/*
 * Relative to a2ps the prog, not liba2ps
 */
a2ps_status *
new_a2ps_status (void)
{
  a2ps_status * res = ALLOC (a2ps_status, 1);

  res->style = "plain";		/* Style is plain			*/
  res->automatic_style = TRUE;	/* Guess the language		 	*/
  res->translate_symbols_request = FALSE; /* no symbol convertion	*/
  res->behavior = b_ps;		/* Produce PS				*/
  res->pattern_rules = ALLOC (hash_table, 1);
  hash_init (res->pattern_rules, 8,
	     mv_key_hash_1, mv_key_hash_2, mv_key_hash_cmp);
  res->strip = 0;		/* Print everything 			*/

  return (res);
}

/*
 * Build the file struct for a new file
 */
file_job *
new_job (print_job * job, char * name)
{
  file_job * res;
  struct tm *tm;
  struct stat statbuf;		/* to get file modification time */
  
  res = ALLOC (file_job, 1);

  /* Initialize the file dependant counters */
  res->first_sheet = job->total_sheets;
  res->first_page = job->total_pages;
  res->pages = 0;
  res->sheets = 0;
  res->toplinenum = 0;
  res->linenum = 0;
  res->num = ++(job->total_files);
  * job->tag1 = NUL;
  * job->tag2 = NUL;
  * job->tag3 = NUL;
  * job->tag4 = NUL;

  /* Retrieve file modification date and hour */
  if (!IS_EMPTY(name)) {
    res->name = (ustring) name;
    if (freopen(name, "r", stdin) == NULL)
      error (1, errno, _("error opening `%s'"), name);
    if (stat(name, &statbuf) == -1)
      error (1, errno, _("error getting file \"%s\" modification time"), 
	     name);
    else { 
      time_t tim = statbuf.st_mtime;
      tm = localtime (&tim);
      memcpy (&(res->mod_tm), tm, sizeof (*tm));
    }
  } else {
    res->name = job->stdin_title;
    /* stdin's modification time is the current time. */
    memcpy (&res->mod_tm, &job->run_tm, sizeof (job->run_tm));
  }
  
  /* Reinitialize the ps status */
  initialize_ps_status (job->status);

  /* Link it */
  res->next_job = job->jobs;
  job->jobs = res;
  
  return res;
}

print_job *
new_print_job (void)
{
  time_t tim;
  struct tm *tm;
  print_job * res;
  struct passwd *passwd = NULL;
  char * cp;
  int e;

  res = ALLOC (print_job, 1);

  /* Information relative to a2ps, not liba2ps	*/
  res->a2ps_stat = new_a2ps_status ();

  /* Information related to the user */
  passwd = getpwuid (getuid ());

  /* Home dir */
  if (!IS_EMPTY (cp = getenv ("HOME")))
    res->home = cp;
  else if (passwd)
    res->home = passwd->pw_dir;
  res->home = UNNULL (res->home);

  /* Login */
  if (!IS_EMPTY (cp = getenv ("LOGNAME")))
    res->pw_name = cp;
  else if (!IS_EMPTY (cp = getenv ("USERNAME")))
    res->pw_name = cp;
  else if (passwd)
    res->pw_name = passwd->pw_name;
  else
    res->pw_name = _("user");
  res->pw_name = UNNULL (res->pw_name);
  

  /* User name (always malloc'd, so that free is easy) */
  if (passwd)
    res->pw_gecos = xstrdup (passwd->pw_gecos);
  else if (!IS_EMPTY (res->pw_name)) {
    res->pw_gecos = xstrdup (res->pw_name);
    res->pw_gecos[0] = toupper (res->pw_gecos[0]);
  } else    
    res->pw_gecos = xstrdup (_("Unknown User"));
  res->pw_gecos = UNNULL (res->pw_gecos);

  /* As the special feature of finger(1), whenever `&'
   * appears, substitute name with capitalized first letter */
  cp = xstrdup (res->pw_name);
  cp [0] = toupper (cp [0]);
  vstrrpl (&(res->pw_gecos), "&", cp, NULL);
  XFREE (cp);
  
  /* Short cuts defined by the user */
  res->user_options = ALLOC (hash_table, 1);
  hash_init (res->user_options, 8, 
	     key_hash_1, key_hash_2, key_hash_cmp);

  /* Check A2PS_LIBRARY for custom library location. */
  cp = getenv ("A2PS_LIBRARY");
  if (cp) { /* If $LIBRARY is given, add regular path at the end */
    sprintf (res->lib_path, "%s%c%s", cp, PATH_SEP, LIBRARY);
  } else { /* Otherwise, just put user's $HOME/.PACKAGE directory */
    sprintf (res->lib_path, "%s%c.%s%c%s",
	     res->home, DIR_SEP, PACKAGE, PATH_SEP, LIBRARY);
  }

  
  /* Get current time information */
  tim = time (NULL);
  tm = localtime (&tim);
  memcpy (&(res->run_tm), tm, sizeof (*tm));

  strcpy (res->datestring, asctime(&res->run_tm));
  /* trailling \n */
  res->datestring [strlen(res->datestring) - 1] = NUL;

  res->total_sheets = 0;
  res->total_pages = 0;
  res->total_files = 0;
  res->orientation = portrait;
  res->rectoverso = FALSE;
  res->columns = 1;
  res->rows = 1;
  res->Major = major_rows;	/* by default, write horizontally	*/
  res->virtuals = 1;
  res->virtual = 1;
  res->copies = 1;
  res->margin = 0;

  /* The very first line of a PS file */
  res->magic_number = "%!PS-Adobe-3.0";
  /* for fonts etc. */
  res->needed_resources = ALLOC (hash_table, 1);
  hash_init (res->needed_resources, 8,
	     mv_key_hash_1, mv_key_hash_2, mv_key_hash_cmp);

  /* Things to put in the preamble */
  res->supplied_resources = ALLOC (hash_table, 1);
  hash_init (res->supplied_resources, 8,
	     mv_key_hash_1, mv_key_hash_2, mv_key_hash_cmp);

  /* for setpagedevice */
  res->pagedevice = ALLOC (hash_table, 1);
  hash_init (res->pagedevice, 8,
	     key_hash_1, key_hash_2, key_hash_cmp);

  /* PS statusdict definitions */
  res->statusdict = ALLOC (hash_table, 1);
  hash_init (res->statusdict, 8, 
	     key_hash_1, key_hash_2, key_hash_cmp);

  /* The setups read in the files */
  res->setup = new_divertion ();

  /* Lists of the encodings used, and chunk in which their PS def
   * is stored */
  res->used_encodings = ALLOC (int, nbr_encodings);
  for (e = 0 ; e < nbr_encodings ; e++)
    res->used_encodings[e] = FALSE;
  res->ps_encodings = new_divertion ();

  res->page_prefeed = FALSE;	/* No page prefeed			*/

  /* Make sure not to be happy to use a not initialized array */
  init_face_to_font (res->face_to_font);

  /* Available virtual printers, i.e., outputs	*/
  res->printers = ALLOC (hash_table, 1);
  hash_init (res->printers, 8, 
	     key_hash_1, key_hash_2, key_hash_cmp);

  /* virtual file name given to stdin */
  res->stdin_title = (ustring) "stdin";

  res->lpr_print = TRUE; 	/* Print by default ? 			*/
  res->printer = NULL;		/* Options to lpr 			*/
  res->output_filename = NULL;	/* NULL -> stdout unless lpr_print	*/
  res->folding = TRUE;		/* Line folding option 			*/
  res->numbering = FALSE;	/* Line numbering option 		*/
  res->only_printable = FALSE;	/* Replace non printable char by space 	*/
  res->interpret = TRUE;	/* Interpret TAB, FF and BS chars option */
  res->print_binaries = FALSE;	/* Force printing for binary files 	*/
  res->compact_mode = FALSE;	/* Allow 2 files per sheet 		*/
  res->border = TRUE;		/* print the surrounding border ?	*/
  res->area = FALSE;		/* Don't draw the BoundingBox 		*/
  res->prolog = "bw";		/* default ps header file		*/
  res->paper = 0;		/* default paper is A4			*/
  res->tabsize = 8;		/* length of tabulations		*/
  res->lines_requested = 0;	/* lines per page			*/
  res->columns_requested = 0;	/* columns per page			*/
  res->fontsize = 0.0;		/* Size of a char for body font 	*/
  res->encoding = ENC_ASCII; 	/* What is the current char set ?	*/
  res->requested_encoding = ENC_ASCII;/* Encoding requested by -X	*/
  /* Headers and footers */
  res->header = UNULL;
  res->center_title = UNULL;
  res->left_title = UNULL;
  res->right_title = UNULL;
  res->left_footer = UNULL;
  res->footer = UNULL;
  res->right_footer = UNULL;
  res->water = UNULL;
  * res->tag1 = NUL;
  * res->tag2 = NUL;
  * res->tag3 = NUL;
  * res->tag4 = NUL;
  /* Private info for PS generation */
  res->status = new_ps_status();

  /* Where the diverted output is stored */
  res->divertion = new_divertion ();

  /* List of the jobs */
  res->jobs = NULL;
  return res;
}
