/* xstrrpl.c -- replacement of substrings
   Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
   Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Author: Akim Demaille <demaille@inf.enst.fr> */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>

#if __STDC__
# include <stdarg.h>
#else
# include <varargs.h>
#endif

#if STDC_HEADERS || _LIBC
# include <stdlib.h>
# include <string.h>
#endif

#if ENABLE_NLS
# include <libintl.h>
# define _(Text) gettext (Text)
#else
# define textdomain(Domain)
# define _(Text) Text
#endif

#if DEBUG
#include <assert.h>
#endif

#include "xmalloc.h"
#include "error.h"
#include "darray.h"

/*
 * Create a dynamic array
 */
struct darray *
da_create (char * name, size_t size, size_t increment, 
	   void (* self_print) (const void *),
	   int (* cmp) (const void *, const void *))
{
  struct darray * res;

  if (size < 0)
    error (1, 0, _("bad size for dynamic array \"%s\": %d"),
	   name, size);
	   
  if (increment <= 0)
    error (1, 0, _("bad increment for dynamic array \"%s\": %d"),
	   name, increment);

  res = ALLOC (struct darray, 1);
  res->name = name;
  res->size = size;
  res->content = CALLOC (void *, res->size);
  res->increment = increment;
  res->len = 0;
  
  /* Routines */
  res->self_print = self_print;
  res->cmp = cmp;

  return res;
}

void
da_erase (struct darray * arr)
{
  XFREE (arr->content);
  XFREE (arr);
}

/*
 * Make a dyn. array bigger
 */
void
da_grow (struct darray * arr)
{
  arr->size += arr->increment;
  arr->content = REALLOC (arr->content, void *, arr->size);
}

/*
 * Is it full?
 */
int
da_is_full (struct darray * arr)
{
  return (arr->len + 1 >= arr->size);
}

/*
 * Append an element
 */
void
da_append (struct darray * arr, void * elem)
{
  if (da_is_full (arr))
    da_grow (arr);

  arr->content [arr->len++] = elem;
}

/*
 * Concat the second in the first
 */
void
da_concat (struct darray * arr, struct darray * arr2)
{
  int i;
  size_t len = arr->len + arr2->len;

  if (len > arr->size) {
    arr->size = len + 1;
    arr->content = REALLOC (arr->content, void *, arr->size);
  }

  for (i = 0 ; i < arr2->len ; i++)
    arr->content [arr->len++] = arr2->content[i];
}

/*
 * Sort the element according to CMP function
 */
void
da_sort (struct darray * arr)
{
  qsort (arr->content, arr->len, sizeof (void *), arr->cmp);
}

/*
 * Report the status of the array
 */
void
da_report (struct darray * arr)
{
  fprintf (stderr, _("Dynamic array \"%s\":\n"), arr->name);
  fprintf (stderr, _("load: %d/%d (%2.1f%%)\n"), 
	   arr->len, arr->size, 100.0 * arr->len / arr->size);
}

/*
 * Dump on stderr the content
 */
void
da_dump_content (struct darray * arr)
{
  int i;

  fprintf (stderr, _("Dynamic array \"%s\":\n"), arr->name);
  for (i = 0 ; i < arr->len ; i++) {
    fprintf (stderr, "[%2d] = ", i);
    arr->self_print (arr->content [i]);
  }
}

void
da_map (struct darray * arr, void (* fn) (void *))
{
  int i;

  for (i = 0 ; i < arr->len ; i++) {
    fn (arr->content [i]);
  }
}
