/*
 * confg.c
 *
 * Read and understanding everything about the options 
 * & (dynamic) configuration of a2ps.
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97 Akim Demaille, Miguel Santana
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/************************************************************************/
/*									*/
/*			I n c l u d e   f i l e s			*/
/*                                                                      */
/************************************************************************/
#include "a2ps.h"
#include "routines.h"
#include "medias.h"
#include "getopt.h"
#include "jobs.h"
#include "getshline.h"
#include "pathwalk.h"
#include "confg.h"
#include "jobs.h"

#define MAN_LINES               66	/* no lines for a man */
extern char * program_name;

/************************************************************************/
/*				arguments				*/
/************************************************************************/
/*
 * Exit because of a bad argument
 */
static void
bad_arg(char *opt, char *arg)
{
  error (0, 0, _("wrong value for option %s: \"%s\""), opt, arg);
  fprintf (stderr, _("Try `%s --help' for more information.\n"), program_name);
  exit (EXIT_BADARG);
}

static struct option long_options[] =
{
  {"columns",		required_argument,	0, 132},
  {"rows",		required_argument,	0, 133},
  {"compact",		required_argument,      0, 150}, /* -A */
  {"header",		optional_argument,	0, 'b'},
  {"no-header", 	no_argument, 		0, 'B'},
  {"truncate-lines",	required_argument,	0, 151}, /* -c */
  {"line-numbers",	required_argument,      0, 152}, /* -C */
  {"setpagedevice",	optional_argument,      0, 'D'},
  {"automatic-style",	required_argument, 	0, 153}, /* -e */
  {"pretty-print",	required_argument,	0, 'E'},
  {"font-size",		required_argument, 	0, 'f'},
  {"graphic-symbols", 	required_argument, 	0, 154}, /* -g */
  {"help",		no_argument,		0, 'h'},
  {"interpret", 	required_argument,	0, 155}, /* -i */
  {"borders", 		required_argument, 	0, 156}, /* -j */
  {"page-prefeed",	no_argument,		0, 'k'},
  {"no-page-prefeed",	no_argument,		0, 'K'},  
  {"lines-per-page",	required_argument,	0, 'L'},
  {"columns-per-page",	required_argument,	0, 'l'},
  {"catman",		no_argument,		0, 'm'},
  {"media", 		required_argument,	0, 'M'},
  {"copies", 		required_argument,	0, 'n'},
  {"output", 		required_argument,	0, 'o'},
  {"printer", 		optional_argument,	0, 'P'},
  {"quiet", 		no_argument,		0, 'q'},
  {"silent",		no_argument,		0, 'q'},
  {"landscape", 	no_argument,		0, 'r'},
  {"portrait", 		no_argument,		0, 'R'},
  {"sides", 		required_argument,	0, 's'},
  {"statusdict",	optional_argument,	0, 'S'},
  {"title", 		optional_argument,	0, 't'},
  {"tabsize", 		required_argument,	0, 'T'},
  {"underlay", 		required_argument,	0, 'u'},
  {"verbose", 		optional_argument,	0, 'v'},
  {"version", 		no_argument,		0, 'V'},
  {"encoding", 		required_argument,	0, 'X'},

  {"guess",		no_argument,		0, 138},
  {"prologue",		required_argument,	0, 134},
  {"include",		required_argument,	0, 134},
  {"list-options", 	no_argument,		0, 139},
  {"list-features", 	no_argument,		0, 145},
  {"non-printable-format",required_argument,	0, 135},
  {"print-anyway",	no_argument,		0, 136},
  {"report",		optional_argument,	0, 137},

  {"center-title",	optional_argument,	0, 149},
  {"left-title",	optional_argument,	0, 140},
  {"right-title",	optional_argument,	0, 141},
  {"left-footer",	optional_argument,	0, 142},
  {"footer",		optional_argument,	0, 143},
  {"right-footer",	optional_argument,	0, 144},

  {"margin",		optional_argument,	0, 147},
  {"strip-level",	required_argument,	0, 148},
  {"user-option",	required_argument,	0, '='},
  {"major",		required_argument,	0, 157},

  {"debug",	 	no_argument,		0, 146},

  /* For compatibility, but to remove some day */
  {"language",		required_argument,	0, 'E'},
  {"minipages",		required_argument,	0, 132},
  {"virtual-pages",	required_argument,	0, 132},

  {NULL, 0, 0, 0}
};


#define OPT_STRING \
 "123456789=:Ab::BcCdD::eE:f:gGhijkKl:L:mM:n:o:P:qrRs:S::t::T:u::v::VX:"

#define BOOLOPTARG(_opt_, x)				 	\
 do {								\
  if (strequ(optarg, "yes") || strequ(optarg, "1"))		\
    (x) = TRUE;							\
  else if (strequ(optarg, "no") || strequ(optarg, "0"))		\
    (x) = FALSE;						\
  else {							\
    error (0, 0, _("wrong boolean given to option %s: `%s'"),		\
	   _opt_, UNNULL (optarg));				\
    fprintf (stderr, _("Try `%s --help' for more information.\n"),\
	     program_name);						\
    exit(EXIT_UKNCODE);						\
  }								\
 } while (0)

/*
 * Handle the options.
 */
int
handle_option (print_job * job, int argc, char *argv[])
{
  int c;

  /* Reset optind. */
  optind = 0;
  
  while (1)
    {
      int option_index = 0;

      c = getopt_long (argc, argv, OPT_STRING, long_options, &option_index);

      if (c == EOF)
        break;
      
      switch (c) {
      case '1':				/* 1 logical page per sheet */
	job->columns = 1;
	job->rows = 1;
	job->orientation = portrait;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '2':				/* twin pages */
	job->columns = 2;
	job->rows = 1;
	job->orientation = landscape;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '3':				/* 3 virtual pages */
	job->columns = 3;
	job->rows = 1;
	job->orientation = landscape;
	job->Major = major_rows;
	break;

      case '4':				/* 4 virtual pages */
	job->columns = 2;
	job->rows = 2;
	job->orientation = portrait;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '5':				/* 5 virtual pages */
	job->columns = 5;
	job->rows = 1;
	job->orientation = landscape;
	job->Major = major_rows;
	break;

      case '6':				/* 6 virtual pages */
	job->columns = 3;
	job->rows = 2;
	job->orientation = landscape;
	job->Major = major_rows;
	break;

      case '7':				/* 7 virtual pages */
	job->columns = 7;
	job->rows = 1;
	job->orientation = landscape;
	job->Major = major_rows;
	break;

      case '8':				/* 8 virtual pages */
	job->columns = 4;
	job->rows = 2;
	job->orientation = landscape;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '9':				/* 9 virtual pages */
	job->columns = 3;
	job->rows = 3;
	job->orientation = portrait;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '=':				/* A user option */
	{
	  /* Caution with recursive calls to getopt */
	  int saved_optind = optind; 
	  handle_string_options (job, get_user_option (job, optarg));
	  optind = saved_optind;
	}
	break;

      case 'A':				/* allow two files per sheet */
	job->compact_mode = TRUE;
	break;

      case 150:				/* allow two files per sheet */
	BOOLOPTARG ("--compact", job->compact_mode);
	break;

      case 'b':				/* sheet header value */
	job->header = (ustring) optarg;
	break;

      case 'B':				/* No headers at all */
	job->header = UNULL;
	job->left_footer = UNULL;
	job->footer = UNULL;
	job->right_footer = UNULL;
	job->left_title = UNULL;
	job->center_title = UNULL;
	job->right_title = UNULL;
	job->water = UNULL;
	break;

      case 'c':				/* cut lines too large */
	job->folding = FALSE;
	break;

      case 151:				/* cut lines too large */
	BOOLOPTARG ("--truncate-lines", job->folding);
	break;

      case 'C':				/* line numbering */
	job->numbering = 5; /* Default is 5 by 5 */
	break;

      case 152:				/* line numbering */
	if (sscanf(optarg, "%d", &(job->numbering)) != 1 || job->numbering < 0)
	  bad_arg ("--line-numbers", optarg);
	break;

      case 'd':					/* fork a process to print */ 
	job->lpr_print = TRUE;
	job->printer = NULL;
	break;

      case 'D': 			/* --setpagedevice		*/
	{
	  char *value;
	  value = strchr (optarg, ':');
	  if (IS_EMPTY (value)) {
	    delpagedevice (job, optarg);
	  } else {
	    *value = NUL;
	    value ++;
	    setpagedevice (job, optarg, value);
	  }
	}
      break;

      case 'e':				/* automatic style		*/
	job->a2ps_stat->automatic_style = TRUE;
	break;

      case 153:				/* automatic style		*/
	BOOLOPTARG ("--automatic-style", job->a2ps_stat->automatic_style);
	break;

      case 'E':           		/* select language */
	job->a2ps_stat->style = optarg;
/*
	  style = select_style (optarg);
	if (style == plain_style) {
	  error (0, 0, _("language %s unknown"), optarg);
	  fprintf (stderr, _("Try `%s --help' for more information.\n"), 
		   program_name);
	  exit(EXIT_UKNLANG);
	}
*/
	break;

      case 'f': 
	{
	  char * cp;
	  /* This is for compatibility with the previous scheme */
	  cp = strchr (optarg, '@');
	  if (cp)
	    cp ++;
	  else
	    cp = optarg;
	  /* A font size is given */
	  if (cp == NULL
	      || sscanf(cp, "%f", &(job->fontsize)) != 1 
	      || job->fontsize <= 0.0)
	    bad_arg ("--fontsize", cp);
	  job->columns_requested = 0;
	  job->lines_requested = 0;	  
	}
      break;

      case 'g':				/* Symbol translation */
	job->a2ps_stat->translate_symbols_request = TRUE;;
	break;

      case 154:				/* Symbol translation */
	BOOLOPTARG ("--graphic-symbols", 
		    job->a2ps_stat->translate_symbols_request);
	break;

      case 'h':				/* --help */
	job->a2ps_stat->behavior = b_help;
	break;

      case 'i':				/* interpret control chars */
	job->interpret = TRUE;
	break;

      case 155:				/* interpret control chars */
	BOOLOPTARG ("--interpret", job->interpret);
	break;

      case 'j':				/* surrounding border */
	job->border = TRUE;
	break;

      case 156:				/* surrounding border */
	BOOLOPTARG ("--border", job->border);
	break;

      case 'k':				/* page prefeed */
	job->page_prefeed = TRUE;
	break;

      case 'K':				/* no page prefeed */
	job->page_prefeed = FALSE;
	break;

      case 'l':
	/* set columns per line, useful for most cases */
	if (sscanf(optarg, "%d", &(job->columns_requested)) != 1
	    || job->columns_requested < 0)
	  bad_arg ("--columns-per-page", optarg);
	job->lines_requested = 0;
	break;

      case 'L':
	/* set lines per page.  Useful with preformatted files. Scaling is
	 * automatically done when necessary.  */
	if (sscanf(optarg, "%d", &(job->lines_requested)) != 1
	    || job->lines_requested < 0)
	  bad_arg ("--lines-per-page", optarg);
	/* Unset value given to columns-per-page, so that this one
	 * is not hidden */
	job->columns_requested = 0;
	break;

      case 'm':				/* Process file as a man */
	job->lines_requested = MAN_LINES;
	job->columns_requested = 0;
	break;

      case 'M':                 		/* select a media */
	job->paper = select_media(optarg);
	if (job->paper == -1)
	  error (1, 0, _("unknown media \"%s\""), optarg);
	break;

      case 'n':				/* n copies */
	if (sscanf(optarg, "%d", &(job->copies)) != 1 || job->copies <= 0)
	  bad_arg ("--copies", optarg);
	break;

      case 'o':			/* output goes into a file */
	if (!strequ(optarg, "-"))
	  job->output_filename = optarg;
	else
	  job->output_filename = NULL;
	job->lpr_print = FALSE;
	break;

      case 'P':					/* fork a process to print */ 
	job->lpr_print = TRUE;
	if (optarg) /* Name of the printer */
	  job->printer = optarg;
	break;

      case 'q':			       /* don't say anything but errors */
	message_verbosity = -1;
	break;

      case 'r':
	job->orientation = landscape;  		/* landscape format */
	break;

      case 'R':
	job->orientation = portrait;  		/* portrait format */
	break;

      case 's':
	if (strequ(optarg, "1")) {		/* one-sided printing */
	  job->rectoverso = FALSE;
	  break;
	} else if (strequ(optarg, "2")) {	/* two-sided printing */
	  job->rectoverso = TRUE;
	  break;
	}
	bad_arg ("--sides", optarg);
	break;

      case 'S':				/* statusdict definitions */
	{
	  char *value;
	  value = strchr (optarg, ':');
	  if (IS_EMPTY (value)) {
	    delstatusdict (job, optarg);
	  } else {
	    *value = NUL;
	    value ++;
	    if (*value == ':')
	       setstatusdict (job, optarg, value + 1, TRUE);
	     else
	       setstatusdict (job, optarg, value, FALSE);
	  }
	}
	break;

      case 't':				/* Page title */
	job->stdin_title = (ustring) optarg;
	break;

      case 'T':
	if ((job->tabsize = atoi(optarg)) <= 0)    /* set tab size */
	  bad_arg ("--tabsize", optarg);
	break;

      case 'u':				/* water mark (under lay) */
	job->water = (ustring) optarg;
	break;

      case 'v':					    /* verbosity */
	if (!IS_EMPTY(optarg)) {
	  if ((message_verbosity = atoi(optarg)) < 0)
	    bad_arg ("--verbose", optarg);
	} else
	  message_verbosity = 0;
	break;

      case 'V':				/* version and configuration info */
	job->a2ps_stat->behavior = b_version;
	break;

      case 'X': 			/* change the encoding scheme */
	if ((job->requested_encoding = select_encoding(optarg)) == NOENCODING)
	  {
	    error (0, 0, _("encoding %s unknown"), optarg);
	    fprintf (stderr, _("Try `%s --help' for more information.\n"), 
		     program_name);
	    exit(EXIT_UKNCODE);
	  }
	/* encoding is the dynamic guy, requested_ stores the one given
	 * by -X */
	job->encoding = job->requested_encoding;
	break;

      case 132:				/* Number of columns */
	if (sscanf(optarg, "%d", &(job->columns)) != 1 || job->columns <= 0)
	  bad_arg ("--columns", optarg);
	break;

      case 133:				/* Number of rows */
	if (sscanf(optarg, "%d", &(job->rows)) != 1 || job->rows <= 0)
	  bad_arg ("--rows", optarg);
	break;

      case 134:				/* --include ps prologue */
	job->prolog = optarg;
	break;

      case 135:				/* --non-printable-format */
	if (strequ(optarg, "caret"))
	  job->only_printable = FALSE;
	else if (strequ(optarg, "space"))
	  job->only_printable = TRUE;
	else
	  bad_arg ("--non-printable", optarg);
	break;

      case 136:				/* --print-anyway */
	job->print_binaries = TRUE;
	break;

      case 137:				/* report styles */
	if (optarg) 
	  job->a2ps_stat->style = optarg;
	/* There are no tests of the validity, because
	 * we are in the library, which doesn't anything about
	 * style sheets! */
	job->a2ps_stat->behavior = b_report;
	break;

      case 138:
	job->a2ps_stat->behavior = b_guess;
	break;

      case 139:
	job->a2ps_stat->behavior = b_list_options;
	break;

      case 140:
	job->left_title = (ustring) optarg;
	break;

      case 141:
	job->right_title = (ustring) optarg;
	break;

      case 149:
	job->center_title = (ustring) optarg;
	break;

      case 142:
	job->left_footer = (ustring) optarg;
	break;

      case 143:
	job->footer = (ustring) optarg;
	break;

      case 144:
	job->right_footer = (ustring) optarg;
	break;

      case 145:
	job->a2ps_stat->behavior = b_list_features;
	break;

      case 146:				/* --debug */
	job->area = TRUE;
	break;

      case 147:				/* --margin */
	if (optarg) {
	  if (sscanf(optarg, "%d", &(job->margin)) != 1 || job->margin <= 0)
	    bad_arg ("--margin", optarg);
	} else
	  job->margin = 12;
	break;

      case 148:				/* --strip-level */
	if (sscanf(optarg, "%d", &(job->a2ps_stat->strip)) != 1 
	    || job->a2ps_stat->strip <= 0)
	  bad_arg ("--strip-level", optarg);
	break;

      case 157: 			/* --major= */
	if (strcaseequ (optarg, "rows")
	    /* Major = row */
	    && strcaseequ (optarg, _("rows")))
	  job->Major = major_rows;
	else if (strcaseequ (optarg, "columns")
		 /* Major = column */
		 && strcaseequ (optarg, _("columns")))
	  job->Major = major_columns;
	else
	  bad_arg ("--major", optarg);
	break;

      case '?':				/* Unknown option */
	/* Error message is done by getopt */
	exit (EXIT_BADARG);
	break;

      default:
	error (1, 0, _("unexpected option: `%c' (%d)"), c, c);
      }
    }
  return optind;
}

void
handle_string_options (print_job * job, char *string)
{
  int argc;
  char **argv;
  char *str;
  int i;

  if (string == NULL)
    return;

  message (1, "handle_string_options(): \"%s\"\n", string);

  /* Copy string so we can modify it in place. */
  str = xstrdup (string);

  /*
   * We can count this, each option takes at least 1 character and one 
   * space.  We also need one for program's name and one for the 
   * trailing NULL. 
   */
  argc = (strlen (str) + 1) / 2 + 2;
  argv = (char **) CALLOC (char *, argc);
  
  /* Set program name. */
  argc = 0;
  argv[argc++] = program_name;

  /* Split string and set arguments to argv array. */
  i = 0;
  while (str[i])
    {
      /* Skip leading whitespace. */
      for (; str[i] && isspace (str[i]); i++)
	;
      if (!str[i])
	break;

      /* Check for quoted arguments. */
      if (str[i] == '"' || str[i] == '\'')
	{
	  int endch = str[i++];

	  argv[argc++] = str + i;

	  /* Skip until we found the end of the quotation. */
	  for (; str[i] && str[i] != endch; i++)
	    ;
	  if (!str[i])
	    error (1, 0, _("syntax error in option string \"%s\":\n\
missing end of quotation: %c"), string, endch);

	  str[i++] = '\0';
	}
      else
	{
	  argv[argc++] = str + i;

	  /* Skip until whitespace if found. */
	  for (; str[i] && !isspace (str[i]); i++)
	    ;
	  if (str[i])
	    str[i++] = '\0';
	}
    }
  
  /* argv[argc] must be NULL. */
  argv[argc] = NULL;

  message (2, _("found following options (argc=%d):\n"), argc);
  for (i = 0; i < argc; i++)
    message (2, "%3d = \"%s\"\n", i, argv[i]);

  /* Process options. */
  (void) handle_option (job, argc, argv);

  /* Cleanup. */
  XFREE (argv);

  /*
   * <str> must not be freed, since some global variables can point to
   * its elements
   */
}

#define GET_TOKEN(from) (strtok ((from), " \t\n"))
#define GET_LINE_TOKEN(from) (strtok ((from), "\n"))
 
#define CHECK_TOKEN() 							\
  if (token2 == NULL) 							\
    error_at_line (1, 0, fname, firstline, _("missing argument: %s"), token);

/*
 * Read the configuration file
 */
int
read_config (print_job * job, char *path, char *file)
{
  FILE *fp;
  char fname[512];
  char *buf = NULL;
  size_t bufsiz = 0;
  char *token, *token2;
  int firstline = 0, lastline = 0;

  if (path)
    sprintf (fname, "%s%c%s", path, DIR_SEP, file);
  else
    sprintf (fname, "%s", file);

  fp = fopen (fname, "r");
  if (fp == NULL)
    return 0;

  while (getshline_numbered (&firstline, &lastline, &buf, &bufsiz, fp) != -1)
    {
      token = GET_TOKEN (buf);
      if (token == NULL)
	/* Empty line. */
	continue;

      if (strequ (token, "Options:"))
	{
	  handle_string_options (job, GET_LINE_TOKEN(NULL));
	}
      else if (strequ (token, "Pattern:"))
	{
	  char * pattern;
	  char * lang;

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  pattern = xstrdup (token2);
	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  lang = xstrdup (token2);
	  add_pattern_rule (job, lang, pattern);
	}
      else if (strequ (token, "Printer:"))
	{
	  char * name, * command;

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  name = xstrdup (token2);
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  command = xstrdup (token2);
	  add_printer (job, name, command);
	}
      else if (strequ (token, "UnknownPrinter:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  add_printer (job, ":", xstrdup (token2));
	}
      else if (strequ (token, "DefaultPrinter:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  add_printer (job, "::", xstrdup (token2));
	}
      else if (strequ (token, "UserOption:"))
	{
	  char * name, * command;

	  token2 = GET_TOKEN (NULL);
	  CHECK_TOKEN ();
	  name = xstrdup (token2);
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  command = xstrdup (token2);
	  add_user_option (job, name, command);
	}
      else if (strequ (token, "OutputFirstLine:"))
	{
	  token2 = GET_LINE_TOKEN (NULL);
	  CHECK_TOKEN ();
	  job->magic_number = xstrdup (token2);
	}
      else
	error_at_line (1, 0, fname, firstline,
		       _("illegal option: `%s'"), token);
    }
  return 1;
}

 /* Global config. 
   *    This is really not an easy thing because, people may want
   *    to check the package before the installation.  The worst
   *    case is when an older a2ps is yet installed.  So we _must_
   *    have a special way to deal with the tests.  This is why
   *    I introduced yet another env-var: NO_SYSCONF, which
   *    should _not_ be documented, but used which `make check'
   *    Note that it also improves the robustness of `make distcheck'
   */
void
read_sys_config (print_job * job)
{
  if (! (getenv ("NO_SYSCONF")))
    {
      /* Well, we are not being tested, so the config file should
       * be in SYSCONFDIR.  I see not reason to end a2ps here
       * if the file is not found: other files follow.  Just say it.
       */

      if (!read_config (job, SYSCONFDIR, "a2ps.cfg"))
	{
	  /* Well, this may be because we are on a micro port,
	   * then try to read the config in the a2ps path */
	  int saved_errno = errno;
	  
	  char * configpath = pw_find_file (job->lib_path, "a2ps.cfg", NULL);
	  if (configpath)
	    if (!read_config (job, NULL, configpath))
	      error (0, saved_errno,
		     _("couldn't open config file `%s%c%s'"),
		     SYSCONFDIR, DIR_SEP, "a2ps.cfg");
	  XFREE (configpath);
	}
    }
}
