(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA                        ENS-CNRS                *)
(*              Rocquencourt                        Lyon                    *)
(*                                                                          *)
(*                                Coq V5.10                                 *)
(*                              Nov 25th 1994                               *)
(*                                                                          *)
(****************************************************************************)
(*                               equality.ml                                *)
(****************************************************************************)
#infix "o";;
#open "std";;
#open "initial";;
#open "names";;
#open "vectops";;
#open "generic";;
#open "term";;
#open "reduction";;
#open "typing";;
#open "termenv";;
#open "mach";;
#open "more_util";;
#open "pp";;
#open "stdpp";;
#open "tacmach";;
#open "proof";;
#open "proof_trees";;
#open "sopattern";;
#open "tactics0";;
#open "tactics1";;
#open "tactics2";;
#open "tactics3";;

#infix "ORELSE";;
#infix "NEXT";;
#infix "THEN";;
#infix "THENF";;
#infix "THENL";;
#infix "THENS";;

(* Tactics for equality reasoning with the "eq"  or "eqT"
   relation This code  will work with any equivalence relation which 
   is substitutive *)

let find_constructor sigma c =
    match whd_betadeltaiota_stack sigma c [] with
     DOPN(MutConstruct _,_) as hd,stack -> (hd,stack)
  | _ -> error "find_constructor"
;;

(* T is (x1:A1)..(xn:An)u build [x1:A1]..[xn:An]endpt *)
let lambda_from_prod_end endpt = drec where rec drec T = 
  match strip_outer_cast T with 
    (DOP2(Prod,C,DLAM(n,T'))) -> (DOP2(Lambda,C,DLAM(n,drec T')))
   | x                        -> endpt;;

(* similar to prod_and_pop, but gives [na:T]B intead of (na:T)B *)

let lam_and_pop_named = fun
    [] body l acc_ids -> error "lam_and_pop"
  | ((na,T)::tlenv) body l acc_ids -> 
     let (Name id)=if na=Anonymous 
                    then Name(next_ident_away (id_of_string "a") acc_ids)
                    else na
     in (tlenv,DOP2(Lambda,T,DLAM((Name id),body)),
         map (fun (0,x) -> (0,lift (-1) x)
              | (n,x) -> (n-1,x)) l,
         (id::acc_ids))
         
;;

(* similar to prod_and_popl but gives [nan:Tan]...[na1:Ta1]B instead of
 * (nan:Tan)...(na1:Ta1)B  it generates names whenever nai=Anonymous *)

let lam_and_popl_named  n env T l = poprec n (env,T,l,[])
    where rec poprec = fun
    0 (env,B,l,_) -> (env,B,l)
  | n ([],_,_,_) -> error "lam_and_popl"
  | n (env,B,l,acc_ids) -> poprec (n-1) (lam_and_pop_named env B l acc_ids)
;;

(* [lambda_ize n T endpt]
 * will pop off the first n products in T, then stick in endpt,
 * properly lifted, and then push back the products, but as lambda-
 * abstractions
 *)
let lambda_ize n T endpt =
let env = [] and carry = [insert_lifted endpt] in
let env,_,[endpt] = push_and_liftl n env T carry in
let T = extract_lifted endpt in
let _,T,[] = lam_and_popl n env T []
in T
;;


let make_case P c lf = DOPN(MutCase,vect_of_list (P::c::lf));;

let mmk = make_module_marker ["#Prelude.obj"];;
let eq_pattern = put_pat mmk "(eq ? ? ?)";;
let True_term = put_pat mmk "True";;
let False_term = put_pat mmk "False";;
let I_term = put_pat mmk "I";;
let not_pattern = put_pat mmk "(not ?)";;

let eq_term = put_pat mmk "eq";;
let eq_ind_term = put_pat mmk "eq_ind";;
let eq_rec_term = put_pat mmk "eq_rec";;
let f_equal_term = put_pat mmk "f_equal";;
let sym_equal_term = put_pat mmk "sym_equal";;

let mmk = make_module_marker ["#Logic_Type.obj"];;
let eqT_pattern = put_pat mmk "(eqT ? ? ?)";;
let eqT_term = put_pat mmk "eqT";;
let eqT_ind_term = put_pat mmk "eqT_ind";;
let eqT_rec_term = put_pat mmk "eqT_rec";;
let congr_eqT_term = put_pat mmk "congr_eqT";;
let sym_eqT_term = put_pat mmk "sym_eqT";;

let rec hd_of_prod prod =
  match strip_outer_cast prod with
    (DOP2(Prod,C,DLAM(n,T'))) -> hd_of_prod T'
  |  T -> T
;;


let is_set c = (whd_castapp c)=DOP0 (Sort (Prop Pos));;

let is_type c = match (whd_castapp c) with 
                 (DOP0 (Sort (Type(Null,_)))) -> true 
                | _ -> false 
;;

let find_eq_pattern arity = 
let ty=hd_of_prod arity in
 if is_set ty then  eq_term
  else  if is_type ty then eqT_term
           else  errorlabstrm "make_inv_predicate" [< 'S "no primitive equality on proofs" >]
;;



(* [find_positions t1 t2]

   will find the positions in the two terms which are suitable for
   discrimination, or for injection.  Obviously, if there is a
   position which is suitable for discrimination, then we want to
   exploit it, and not bother with injection.  So when we find a
   position which is suitable for discrimination, we will just raise
   an exception with that position.

   So the algorithm goes like this:

   if [t1] and [t2] start with the same constructor, then we can
   continue to try to find positions in the arguments of [t1] and
   [t2].

   if [t1] and [t2] do not start with the same constructor, then we
   have found a discrimination position

   if one [t1] or [t2] do not start with a constructor and the two
   terms are not already convertible, then we have found an injection
   position.

   A discriminating position consists of a constructor-path and a pair
   of operators.  The constructor-path tells us how to get down to the
   place where the two operators, which must differ, can be found.

   An injecting position has two terms instead of the two operators,
   since these terms are different, but not manifestly so.

   A constructor-path is a list of pairs of (operator * int), where
   the int (based at 0) tells us which argument of the operator we
   descended into.

 *)

exception DiscrFound of (sorts oper * int) list * sorts oper * sorts oper;;

let find_positions sigma sign t1 t2 =
    (try inr(findrec [] t1 t2)
     with DiscrFound posn -> inl posn)

    where rec findrec posn t1 t2 =
    match (whd_betadeltaiota_stack sigma t1 [],
           whd_betadeltaiota_stack sigma t2 []) with
  
    ((DOPN(MutConstruct _ as oper1,_) as hd1,args1),
     (DOPN(MutConstruct _ as oper2,_) as hd2,args2)) ->
    (* both sides are constructors, so either we descend, or we can
       discriminate here.
     *)
    if oper1 = oper2 then
        join(map2_i (fun i arg1 arg2 -> findrec ((oper1,i)::posn) arg1 arg2)
                    0 args1 args2)
    else raise (DiscrFound(rev posn,oper1,oper2))

  | (T1,T2) ->
    let T1 = applist T1 and
        T2 = applist T2 in
        if conv_x sigma T1 T2 then []
        else (match whd_castapp ((fexecute sigma sign T1).KIND) with
              DOP0(Sort(Prop Pos))     -> [(rev posn,T1,T2)] (* Set *)
            | DOP0(Sort(Type(Null,_))) -> [(rev posn,T1,T2)] (* Type *)
            |  DOP0(Sort(Type(Pos,_))) -> [(rev posn,T1,T2)] (* Typeset *)
            | _ -> [])
;;


let discriminable sigma sign t1 t2 =
    match find_positions sigma sign t1 t2 with
    inl _ -> true
  | _ -> false
;;

(* Once we have found a position, we need to project down to it.  If
   we are discriminating, then we need to produce False on one of the
   branches of the discriminator, and True on the other one.  So the
   result type of the case-expressions is always Prop.

   If we are injecting, then we need to discover the result-type.
   This can be difficult, since the type of the two terms at the
   injection-position can be different, and we need to find a
   dependent sigma-type which generalizes them both.

   We can get an approximation to the right type to choose by:

   (0) Before beginning, we reserve a metavariable for the default
   value of the match, to be used in all the bogus branches.

   (1) perform the case-splits, down to the site of the injection.  At
   each step, we have a term which is the "head" of the next
   case-split.  At the point when we actually reach the end of our
   path, the "head" is the term to return.  We compute its type, and
   then, backwards, make a sigma-type with every free debruijn
   reference in that type.  We can be finer, and first do a S(TRONG)NF
   on the type, so that we get the fewest number of references
   possible.

   (2) This gives us a closed type for the head, which we use for the
   types of all the case-splits.

   (3) Now, we can compute the type of one of T1, T2, and then unify
   it with the type of the last component of the result-type, and this
   will give us the bindings for the other arguments of the tuple.

 *)

(* The algorithm, then is to perform successive case-splits.  We have
   the result-type of the case-split, and also the type of that
   result-type.  We have a "direction" we want to follow, i.e. a
   constructor-number, and in all other "directions", we want to juse
   use the default-value.

   After doing the case-split, we call the afterfun, with the updated
   environment, to produce the term for the desired "direction".

   The assumption is made here that the result-type is not manifestly
   functional, so we can just use the length of the branch-type to
   know how many lambda's to stick in.

 *)

(* [descend_then sigma env head dirn]

   returns the number of products introduced, and the environment
   which is active, in the body of the case-branch given by [dirn],
   along with a continuation, which expects to be fed:

    (1) the value of the body of the branch given by [dirn]
    (2) the default-value

    (3) the type of the default-value, which must also be the type of
        the body of the [dirn] branch

   the continuation then constructs the case-split.
 *)

let descend_then sigma env head dirn =
let headj = execute_rec sigma env head in
let (construct,largs,nparams,arityind,consnamev,case_fun,type_branch_fun) = 
    let (DOPN(MutInd sp,cl) as ity,largs) =
      whd_betadeltaiota_stack sigma headj.TYP [] in 
    let mispec = mind_specif_of_mind ity in 
    let nparams = mis_nparams mispec and consnamev = mis_consnames mispec
    and arity = mis_arity mispec in 
    (DOPN(MutConstruct(sp,dirn),cl),largs,nparams,mis_arity mispec,consnamev,make_case,
     type_case_branches env sigma)

in 
let (globargs,largs) = chop_list nparams largs in

let dirn_cty = strong whd_castapp (type_of_rel sigma env (applist(construct,globargs))) in
let dirn_nlams = nb_prod dirn_cty in
let (_,dirn_env) = add_prods_rel sigma (dirn_cty,env)
in (dirn_nlams,
    dirn_env,

    (fun dirnval (dfltval,resty) ->

let nconstructors = vect_length consnamev in
let arity = hnf_prod_applist sigma "discriminate" arityind globargs in
let P = lambda_ize (nb_prod arity) arity resty in
let nb_prodP = nb_prod P in
let (bty,_) = type_branch_fun (DOP2(Cast,headj.TYP,headj.KIND)) (type_of_rel sigma env P) P head in


let build_branch i =
    let result = if i = dirn then dirnval else dfltval in
    let nlams = nb_prod bty.(i-1) in
    let typstack,_,[] = push_and_liftl (nlams-nb_prodP) [] bty.(i-1) [] in
    let _,branchval,_ = lam_and_popl_named (nlams-nb_prodP) typstack result [] in
        branchval in

    case_fun P head (map build_branch (interval 1 nconstructors))))
;;


(* Now we need to construct the discriminator, given a discriminable
   position.  This boils down to:

   (1) If the position is directly beneath us, then we need to do a
   case-split, with result-type Prop, and stick True and False into
   the branches, as is convenient.

   (2) If the position is not directly beneath us, then we need to
   call descend_then, to descend one step, and then recursively
   construct the discriminator.

 *)

(* [construct_discriminator env dirn headval]

   constructs a case-split on [headval], with the [dirn]-th branch
   giving [True], and all the rest giving False.
 *)

let construct_discriminator sigma env dirn c =
let T = type_of_rel sigma env c in
let (largs,nparams,arityind,consnamev,case_fun,type_branch_fun) = 
    let  (DOPN(MutInd sp,cl) as ity,largs) = whd_betadeltaiota_stack sigma T []    in let mispec = mind_specif_of_mind ity in 
    let nparams = mis_nparams mispec and consnamev = mis_consnames mispec
    and arity = mis_arity mispec in 
    (largs,nparams,mis_arity mispec,consnamev,make_case,type_case_branches env sigma)

in
let nconstructors = vect_length consnamev in
let (globargs,largs) = chop_list nparams largs in

let arity = hnf_prod_applist sigma "construct_discriminator"  arityind globargs and
    True = get_pat True_term and
    False = get_pat False_term and
    eq = get_pat (find_eq_pattern  arityind) in
let P = lambda_ize (nb_prod arity) arity (DOP0(Sort (Prop Null))) in

let (bty,_) = type_branch_fun (DOP2(Cast,T,type_of_rel sigma env T)) (type_of_rel sigma env P) P c in

let build_branch i =
    let nlams = nb_prod bty.(i-1) in
    let endpt = if i = dirn then True else False
    in lambda_ize nlams bty.(i-1) endpt in

let build_match () =
    case_fun P c (map build_branch (interval 1 nconstructors)) in

    build_match()
;;

let rec build_discriminator sigma env dirn c = function
    [] -> construct_discriminator sigma env dirn c
  | (MutConstruct(sp,cnum),argnum)::l ->
    let cty = type_of_rel sigma env c in
    let (ity,_) = find_mrectype sigma cty in
    let nparams = mind_nparams ity in
    let (cnum_nlams,cnum_env,kont) = descend_then sigma env c cnum in
    let newc = Rel(cnum_nlams-(argnum-nparams)) in
    let subval = build_discriminator sigma cnum_env dirn newc l in
        kont subval (get_pat False_term,DOP0(Sort(Prop Null)))
;;


let find_eq_data_decompose eqn =
if (somatches eqn eq_pattern) 
  then (eq_term,sym_equal_term,f_equal_term, eq_ind_term,(SOME eq_rec_term),dest_somatch eqn eq_pattern )
  else if (somatches eqn eqT_pattern) 
        then (eqT_term,sym_eqT_term,congr_eqT_term, eqT_ind_term,(SOME eqT_rec_term), dest_somatch eqn eqT_pattern )
         else  errorlabstrm  "find_eq_data_decompose" [< >]
;;

  
let Discr id gls =
let eqn = (pf_whd_betadeltaiota gls (clause_type (SOME id) gls)) in
let (_,_,_,eq_ind,_,[T;t1;t2])= try (find_eq_data_decompose   eqn) 
                        with UserError _ -> errorlabstrm  "Discriminate"
                [<'S (string_of_id id); 'S" Not a discriminable equation">] in

let Tj = pf_fexecute gls T in
let sigma = Project gls in
let sign = pf_hyps gls
in (match find_positions sigma sign t1 t2 with
    (inr _) ->
    errorlabstrm "Discr"
    [< 'S (string_of_id id); 'S" Not a discriminable equality" >]

  | (inl(cpath,MutConstruct(_,dirn),_)) ->
let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma Tj) sign) in
let discriminator = build_discriminator sigma e_env dirn (VAR e) cpath in

let eq_ind = get_pat eq_ind and
    I=get_pat I_term in
let pf = applist(eq_ind,[T;t1;lambda e T discriminator;I;t2])

in COMPLETE(Cut (get_pat False_term) THENS
            [OnLastHyp (Absurdity o outSOME);
             Refine (apply pf (VAR id))]) gls)
;;




let not_found_message  id =
     [<'S "During the relocation of global references, the variable(s) ["; 
        'SPC ; 'S (string_of_id id) ; 'SPC; 'S" ] were not found in the current environment" >]
;;

let insatisfied_prec_message cls =
 match cls with
    NONE -> [< 'S"goal does not satify the expected preconditions">] 
 |  SOME id -> [< 'S(string_of_id id); 'SPC; 'S"does not satify the expected preconditions" >]
;;

let DiscrClause cls gls =
   match cls with
    NONE ->
    if somatches (pf_concl gls) not_pattern then
        (HnfClause NONE THEN Intro THEN
         OnLastHyp (Discr o outSOME)) gls
    else errorlabstrm "DiscrClause" (insatisfied_prec_message cls)
  | SOME id -> try (Discr id gls)
               with  Not_found -> errorlabstrm "DiscrClause" (not_found_message  id)
;;


let DiscrConcl gls = DiscrClause NONE gls;;
let DiscrHyp id gls = DiscrClause (SOME id) gls;;


let DiscrConcl_tac = register_atomic_tactic "Discr" DiscrConcl;;
let DiscrHyp_tac = register_ident_tactic "DiscrHyp" DiscrHyp;;

(* [bind_ith na i T]

   will verify that T has no binders below [Rel i], and produce the
   term [na]T, binding [Rel i] in T.  The resulting term should be
   valid in the same environment as T, which means that we have to
   re-lift it.

*)
let bind_ith na i T = lift i (DLAM(na,lift (-(i-1)) T));;

let existS_term = put_pat mmk "existS";;
let existS_pattern = put_pat mmk "(existS ? ? ? ?)";;
let sigS_term = put_pat mmk "sigS";;
let projS1_term = put_pat mmk "projS1";;
let projS2_term = put_pat mmk "projS2";;
let sigS_rec_term = put_pat mmk "sigS_rec";;


let existT_term = put_pat mmk "existT";;
let existT_pattern = put_pat mmk "(existT ? ? ? ?)";;
let sigT_term = put_pat mmk "sigT";;
let projT1_term = put_pat mmk "projT1";;
let projT2_term = put_pat mmk "projT2";;
let sigT_rect_term = put_pat mmk "sigT_rect";;


(* returns the sigma type (sigS, sigT) with the respective
    constructor depending on the sort
*)
let find_sigma_data s =
 match strip_outer_cast s with  
    DOP0(Sort(Prop Pos))     ->                                (* Set *) 
       (projS1_term,projS2_term,sigS_rec_term,existS_term, sigS_term)     
 |  DOP0(Sort(Type(Null,_))) ->                                (* Type *)
       (projT1_term, projT2_term, sigT_rect_term, existT_term, sigT_term)  
 |   _     -> error "find_sigma_data"
;;


(* [make_tuple env na lind rterm rty]

   If [rty] depends on lind, then we will fabricate the term

          (existS A==[type_of(Rel lind)] P==(Lambda(type_of(Rel lind),
                                            [bind_ith na lind rty]))
                  [(Rel lind)] [rterm])

   [The term (Lambda(type_of(Rel lind),[bind_ith na lind rty])) is
    valid in [env] because [bind_ith] produces a term which does not
    "change" environments.]

   which should have type (sigS A P) - we can verify it by
   typechecking at the end.

 *)

let make_tuple sigma env na lind rterm rty =
    if dependent (Rel lind) rty then
      let (_,_,_,exist_term,sig_term)=find_sigma_data (type_of_rel sigma env rty) in
      let A = type_of_rel sigma env (Rel lind) in
      let P = DOP2(Lambda,A,
                     bind_ith (fst(lookup_rel lind env)) lind rty) in
            (applist(get_pat exist_term,[A;P;(Rel lind);rterm]),
             applist(get_pat sig_term,[A;P]))
    else (rterm,rty)
;;

(* check that the free-references of the type of [c] are contained in
   the free-references of the normal-form of that type.  If the normal
   form of the type contains fewer references, we want to return that
   instead.
 *)
let minimal_free_rels sigma (c,cty) =
let cty_rels = free_rels cty in
let nf_cty = strong (whd_betadeltaiota sigma) cty in
let nf_rels = free_rels nf_cty in
    if listset__subset cty_rels nf_rels then
        (cty,cty_rels)
    else (nf_cty,nf_rels)
;;


(* [sig_clausale_forme siglen ty]
    
   Will explode [siglen] [sigS,sigT ]'s on [ty] (depending on the 
   type of ty), and return:

   (1) a pattern, with meta-variables in it for various arguments,
       which, when the metavariables are replaced with appropriate
       terms, will have type [ty]

   (2) an integer, which is the last argument - the one which we just
       returned.

   (3) a pattern, for the type of that last meta

   (4) a typing for each metavariable

   WARNING: No checking is done to make sure that the 
            sigS(or sigT)'s are actually there.
          - Only homogenious pairs are built i.e. pairs where all the 
   dependencies are of the same sort
 *)


let sig_clausale_forme sort_of_ty siglen ty =
 let (_,_,_,exist_term,_)=find_sigma_data sort_of_ty 
 in sigrec_clausale_forme siglen ty 
 where rec sigrec_clausale_forme siglen ty =
    if siglen = 0 then
        let mv = newMETA()
        in (DOP0(Meta mv),(mv,ty),[(mv,ty)])
    else
    let (_,[A;P]) = whd_beta_stack ty [] in
    let mv = newMETA() in
    let rty = applist(P,[DOP0(Meta mv)]) in
    let (rpat,headinfo,mvenv) = sigrec_clausale_forme (siglen-1) rty in
        (applist(get_pat exist_term,[A;P;DOP0(Meta mv);rpat]),
         headinfo,
         (mv,A)::mvenv)
;;

(* [make_iterated_tuple sigma env DFLT c]

   Will find the free (DB) references of the S(TRONG)NF of [c]'s type,
   gather them together in left-to-right order (i.e. highest-numbered
   is farthest-left), and construct a big iterated pair out of it.
   This only works when the references are all themselves to members
   of [Set]s, because we use [sigS] to construct the tuple.

   Suppose now that our constructed tuple is of length [tuplen].

   Then, we need to construct the default value for the other
   branches.  The default value is constructed by taking the
   tuple-type, exploding the first [tuplen] [sigS]'s, and replacing at
   each step the binder in the right-hand-type by a fresh
   metavariable.

   In addition, on the way back out, we will construct the pattern for
   the tuple which uses these meta-vars.

   This gives us a pattern, which we use to match against the type of
   DFLT; if that fails, then against the S(TRONG)NF of that type.  If
   both fail, then we just cannot construct our tuple.  If one of
   those succeed, then we can construct our value easily - we just use
   the tuple-pattern.

 *)
(******
let make_iterated_tuple sigma env (DFLT,DFLTty) (c,cty) =
let (cty,rels) = minimal_free_rels sigma (c,cty) in
let sort_of_cty =type_of_rel sigma env cty in
let sorted_rels = sort (neg gt) rels in
let (tuple,tuplety) =
    it_list (fun (rterm,rty) lind ->
                 let na = fst(lookup_rel lind env)
                 in make_tuple sigma env na lind rterm rty)
    (c,cty)
    sorted_rels in

if not(closed tuplety) then failwith "make_iterated_tuple";

let (tuplepat,(headmv,headpat),mvenv) = 
    sig_clausale_forme sort_of_cty (length sorted_rels) tuplety in

let headpat = strong whd_beta headpat in

let nf_ty = strong (whd_betadeltaiota sigma) DFLTty in

let dfltval =
    try_find (fun ty ->
                  try let binding = somatch__somatch NONE headpat ty
                      in instance ((headmv,DFLT)::binding) tuplepat
                  with UserError _ -> failwith "caught")
    [DFLTty;nf_ty] in

    (tuple,tuplety,dfltval)
;;
*******)

let make_iterated_tuple sigma env (DFLT,DFLTty) (c,cty) =
let (cty,rels) = minimal_free_rels sigma (c,cty) in
let sort_of_cty =type_of_rel sigma env cty in
let sorted_rels = sort (neg gt) rels in
let (tuple,tuplety) =
    it_list (fun (rterm,rty) lind ->
                 let na = fst(lookup_rel lind env)
                 in make_tuple sigma env na lind rterm rty)
    (c,cty)
    sorted_rels in

if not(closed tuplety) then failwith "make_iterated_tuple";

let (tuplepat,(headmv,headpat),mvenv) = 
    sig_clausale_forme sort_of_cty (length sorted_rels) tuplety in

let headpat = strong whd_beta headpat in

let nf_ty = strong (whd_betadeltaiota sigma) DFLTty in

let dfltval =
    try_find (fun ty -> 
                  try let binding = if is_type headpat & is_type ty
                                     then []
                                     else  somatch__somatch NONE headpat ty
                      in instance ((headmv,DFLT)::binding) tuplepat
                  with UserError _ -> failwith "caught")
    [DFLTty;nf_ty] in

    (tuple,tuplety,dfltval)
;;



let rec build_injrec sigma env (T1,T2) c = function
    [] -> make_iterated_tuple sigma env (T1,type_of_rel sigma env T1)
                                        (c,type_of_rel sigma env c)
  | (MutConstruct(sp,cnum),argnum)::l ->
    let cty = type_of_rel sigma env c in
    let (ity,_) = find_mrectype sigma cty in
    let nparams = mind_nparams ity in
    let (cnum_nlams,cnum_env,kont) = descend_then sigma env c cnum in
    let newc = Rel(cnum_nlams-(argnum-nparams)) in
    let (subval,tuplety,dfltval) = build_injrec sigma cnum_env (T1,T2) newc l in
        (kont subval (dfltval,tuplety),
         tuplety,dfltval)
;;
(************** 
let build_injector sigma env (T1,T2) c cpath =
let (injcode,resty,_) = build_injrec sigma env (T1,T2) c cpath
in (injcode,resty)
;;


let Inj id gls =
let eqn = (pf_whd_betadeltaiota gls (clause_type (SOME id) gls)) in
let (_,_,eq_congr_term,_,_,[T;t1;t2])= (try (find_eq_data_decompose  eqn) 
                        with UserError _ -> 
       errorlabstrm "Inj"  [<'S(string_of_id id); 'S" Not a primitive  equality here " >])  in
let Tj = pf_fexecute gls T in
let sigma = Project gls in
let sign = pf_hyps gls 
in (match find_positions sigma sign t1 t2 with
    (inl _) ->
    errorlabstrm "Inj" [<'S (string_of_id id); 'S" is not a projectable equality" >]

  | (inr posns) ->
let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma Tj) sign) in
let injectors =
    map_succeed
    (fun (cpath,T1,T2) ->
         let (injbody,resty) = build_injector sigma e_env (T1,T2) (VAR e) cpath in
         let injfun = lambda e T injbody
         in try type_of sigma sign injfun ; (injfun,resty)
            with UserError _ -> failwith "caught")
         posns in

if injectors = [] then FAILTAC gls else

OnL (fun (injfun,resty) ->
         let pf = applist(get_pat eq_congr_term,
                          [T;resty;injfun;(whd_betadeltaiota sigma t1); (whd_betadeltaiota sigma t2);VAR id]) in
         let ty = pf_type_of gls pf in
             (tactics__cut_tac  ty  THENS [IDTAC;Refine pf]))
    injectors
    gls)
;;
************)



let build_injector sigma env (T1,T2) c cpath =
let (injcode,resty,_) = build_injrec sigma env (T1,T2) c cpath
in (injcode,resty)
;;

let try_delta_expand sigma t =
 let whdt= whd_betadeltaiota sigma t  in hd_rec whdt 
where rec hd_rec c  =
    match c with
       DOPN(MutConstruct _,_) -> whdt
    |  DOPN(AppL,cl)  -> hd_rec (hd_vect cl)
    |  DOP2(Cast,c,_) -> hd_rec c
    |   _  -> t
;;


(* Given t1=t2 Inj calculates the whd normal forms of t1 and t2 and it 
   expands then only when the whdnf has a constructor of an inductive type
   in hd position, otherwise delta expansion is not done
*)

let Inj id gls =
let eqn = (pf_whd_betadeltaiota gls (clause_type (SOME id) gls)) in
let (_,_,eq_congr_term,_,_,[T;t1;t2])= (try (find_eq_data_decompose  eqn) 
                        with UserError _ -> 
       errorlabstrm "Inj"  [<'S(string_of_id id); 'S" Not a primitive  equality here " >])  in
let Tj = pf_fexecute gls T in
let sigma = Project gls in
let sign = pf_hyps gls 
in (match find_positions sigma sign t1 t2 with
    (inl _) ->
    errorlabstrm "Inj" [<'S (string_of_id id); 'S" is not a projectable equality" >]

  | (inr posns) ->
let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma Tj) sign) in
let injectors =
    map_succeed
    (fun (cpath,T1,T2) ->
         let (injbody,resty) = build_injector sigma e_env (T1,T2) (VAR e) cpath in
         let injfun = lambda e T injbody
         in try type_of sigma sign injfun ; (injfun,resty)
            with UserError _ -> failwith "caught")
         posns in

if injectors = [] then FAILTAC gls else

OnL (fun (injfun,resty) ->
         let pf = applist(get_pat eq_congr_term,
                          [T;resty;injfun;(try_delta_expand sigma t1); (try_delta_expand sigma t2);VAR id]) in
         let ty = pf_type_of gls pf in
             (tactics__cut_tac  ty  THENS [IDTAC;Refine pf]))
    injectors
    gls)
;;




let InjClause cls gls =
    match cls with
    NONE ->
    if somatches (pf_concl gls) not_pattern then
        (HnfClause NONE THEN Intro THEN
         OnLastHyp (Inj o outSOME)) gls
    else errorlabstrm "InjClause" (insatisfied_prec_message  cls)
  | SOME id ->try (Inj id gls)
              with  Not_found ->errorlabstrm "InjClause" (not_found_message id)
                |   UserError("refiner__FAILTAC",_) -> 
                        errorlabstrm "InjClause"  [< 'S (string_of_id id);'S" Not a projectable equality" >]
;;

let InjConcl gls = InjClause NONE gls;;
let InjHyp id gls = InjClause (SOME id) gls;;


let InjConcl_tac = register_atomic_tactic "Inj" InjConcl;;
let InjHyp_tac = register_ident_tactic "InjHyp" InjHyp;;

(****** fait trop d'unfold, pb. de arditi:

let DecompEqThen ntac id gls =
let eqn = (pf_whd_betadeltaiota gls (clause_type (SOME id) gls)) in
let (_,_,eq_congr_term,eq_ind_t,_,[T;t1;t2])= find_eq_data_decompose  eqn in
let Tj = pf_fexecute gls T in
let sigma = Project gls in
let sign = pf_hyps gls 
in (match find_positions sigma sign t1 t2 with
    (inl(cpath,MutConstruct(_,dirn),_)) ->
(let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma Tj) sign) in
let discriminator = build_discriminator sigma e_env dirn (VAR e) cpath in

let eq_ind = get_pat eq_ind_t and
    I = get_pat I_term in
let pf = applist(eq_ind,[T;t1;lambda e T discriminator;I;t2])

in COMPLETE(Cut (get_pat False_term) THENS
            [OnLastHyp (Absurdity o outSOME);
             Refine (apply pf (VAR id))]) gls)

  | (inr posns) ->
(let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma Tj) sign) in
let injectors =
    map_succeed
    (fun (cpath,T1,T2) ->
         let (injbody,resty) = build_injector sigma e_env (T1,T2) (VAR e) cpath in
         let injfun = lambda e T injbody
         in try type_of sigma sign injfun ; (injfun,resty)
            with UserError _ -> failwith "caught")
         posns in

if injectors = [] then FAILTAC gls else

(OnL (fun (injfun,resty) ->
         let pf = applist(get_pat eq_congr_term,
                          [T;resty;injfun;(whd_betadeltaiota sigma t1);(whd_betadeltaiota sigma t2);VAR id]) in
         let ty = pf_type_of gls pf in
             (tactics__cut_tac ty THENS [IDTAC;Refine pf]))
    (rev injectors)
THEN (ntac (length injectors)))
    gls))
;;

*******)

let DecompEqThen ntac id gls =
let eqn = (pf_whd_betadeltaiota gls (clause_type (SOME id) gls)) in
let (_,_,eq_congr_term,eq_ind_t,_,[T;t1;t2])= find_eq_data_decompose  eqn in
let Tj = pf_fexecute gls T in
let sigma = Project gls in
let sign = pf_hyps gls 
in (match find_positions sigma sign t1 t2 with
    (inl(cpath,MutConstruct(_,dirn),_)) ->
(let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma Tj) sign) in
let discriminator = build_discriminator sigma e_env dirn (VAR e) cpath in

let eq_ind = get_pat eq_ind_t and
    I = get_pat I_term in
let pf = applist(eq_ind,[T;t1;lambda e T discriminator;I;t2])

in COMPLETE(Cut (get_pat False_term) THENS
            [OnLastHyp (Absurdity o outSOME);
             Refine (apply pf (VAR id))]) gls)

  | (inr posns) ->
(let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma Tj) sign) in
let injectors =
    map_succeed
    (fun (cpath,T1,T2) ->
         let (injbody,resty) = build_injector sigma e_env (T1,T2) (VAR e) cpath in
         let injfun = lambda e T injbody
         in try type_of sigma sign injfun ; (injfun,resty)
            with UserError _ -> failwith "caught")
         posns in

if injectors = [] then FAILTAC gls else

(OnL (fun (injfun,resty) ->
         let pf = applist(get_pat eq_congr_term,
                          [T;resty;injfun;t1;t2;VAR id]) in
         let ty = pf_type_of gls pf in
             (tactics__cut_tac ty THENS [IDTAC;Refine pf]))
    (rev injectors)
THEN (ntac (length injectors)))
    gls))
;;


let DecompEq = DecompEqThen (fun x -> IDTAC);;


let DEqThen ntac cls gls =
    match cls with
    NONE ->
    if somatches (pf_concl gls) not_pattern then
        (HnfClause NONE THEN Intro THEN
         OnLastHyp ((DecompEqThen ntac) o outSOME)) gls
    else errorlabstrm "DEqThen" (insatisfied_prec_message  cls)
  | SOME id -> try (DecompEqThen ntac id gls)
               with Not_found -> errorlabstrm "DEqThen" (not_found_message id)
                 |  UserError _ -> errorlabstrm "DEqThen" (insatisfied_prec_message cls)
;;

let DEq = DEqThen (fun x -> IDTAC);;

let DEqConcl gls = DEq NONE gls;;
let DEqHyp id gls = DEq (SOME id) gls;;
let DEqConcl_tac = register_atomic_tactic "DEqConcl" DEqConcl;;
let DEqHyp_tac = register_ident_tactic "DEqHyp" DEqHyp;;


let rewrite_msg = function 
   NONE ->  
     [<'S "passed term is not a primitive equality">] 
| (SOME id) ->[<'S (string_of_id id); 'S "does not satisfy preconditions ">]
;;


let swap_equands gls eqn =
let (eq_t,_,_,_,_,[T;e1;e2]) = (try find_eq_data_decompose eqn
         with _ -> errorlabstrm "swap_equamds" (rewrite_msg NONE)) 
in applist(get_pat eq_t,[T;e2;e1])
;;

let SwapEquandsInConcl gls =
 let (_,sym_eq,_,_,_,[T;e1;e2]) = (try find_eq_data_decompose (pf_concl gls)
            with _-> errorlabstrm "SwapEquandsInConcl" (rewrite_msg NONE)) in
 let sym_equal = get_pat sym_eq in
      Refine (applist(sym_equal,[T;e2;e1;DOP0(Meta(newMETA()))])) gls
;;

let SwapEquandsInHyp id gls =
    (CutReplacing id (swap_equands gls (clause_type (SOME id) gls)) THENS
     [IDTAC;
      SwapEquandsInConcl THEN Exact (VAR id)]) gls
;;

let BareRevSubstInConcl eq_ind P (T,e1,e2) gls =
    Refine (applist(eq_ind,[T;e1;P;DOP0(Meta(newMETA()));
                            e2;DOP0(Meta(newMETA()))])) gls
;;

(* [subst_tuple_term dep_pair B]

   Given that dep_pair looks like:

   (existS e1 (existS e2 ... (existS en en+1) ... ))

   and B might contain instances of the ei, we will return the term:

   ([x1:ty(e1)]...[xn:ty(en)]B
    (projS1 (Rel 1))
    (projS1 (projS2 (Rel 1)))
    ... etc ...)

   That is, we will abstract out the terms e1...en+1 as usual, but
   will then produce a term in which the abstraction is on a single
   term - the debruijn index [Rel 1], which will be of the same type
   as dep_pair.

   ALGORITHM for abstraction:

   We have a list of terms, [e1]...[en+1], which we want to abstract
   out of [B].  For each term [ei], going backwards from [n+1], we
   just do a [subst_term], and then do a lambda-abstraction to the
   type of the [ei].

 *)


let comp_carS_pattern = put_pat mmk "<<x>>(projS1 ? ? (?)@[x])";;
let comp_cdrS_pattern = put_pat mmk "<<x>>(projS2 ? ? (?)@[x])";;

let comp_carT_pattern = put_pat mmk "<<x>>(projT1 ? ? (?)@[x])";;
let comp_cdrT_pattern = put_pat mmk "<<x>>(projT2 ? ? (?)@[x])";;



let find_sigma_data_decompose ex =
try (comp_carS_pattern, comp_cdrS_pattern, dest_somatch ex existS_pattern)
   with _ ->(try (comp_carT_pattern,comp_cdrT_pattern,
                  dest_somatch ex existT_pattern)
              with _ -> errorlabstrm "find_sigma_data_decompose" [< >])
;;
                        


let decomp_tuple_term = decomprec (DLAM(Anonymous,Rel 1))
    where rec decomprec to_here_fun Ex =
    try let (comp_car_pattern,comp_cdr_pattern,[A;P;car;cdr]) = find_sigma_data_decompose  Ex in
        let car_code = soinstance comp_car_pattern [A;P;to_here_fun] and
            cdr_code = soinstance comp_cdr_pattern [A;P;to_here_fun]
        in (car,named_hd A Anonymous,car_code)::(decomprec cdr_code cdr)

    with UserError _ -> [(Ex,named_hd Ex Anonymous,to_here_fun)]
;;

let subst_tuple_term sigma sign dep_pair B =
let env=(GLOB sign) in 
let sort_of_dep_pair =type_of_rel sigma env (type_of_rel sigma env dep_pair) in
let (proj1_term,proj2_term,sig_elim_term,_,_)=find_sigma_data sort_of_dep_pair in 
let e_list = decomp_tuple_term dep_pair in
let abst_B =
    list_it (fun (e,na,_) B ->
                 let body = subst_term e B in
                 let PB = DLAM(na,body) in
                     DOP2(Lambda,type_of sigma sign e,PB))
    e_list B in

let app_B = applist(abst_B,(map (fun (_,_,c) -> (SAPP c (Rel 1))) e_list)) in

let (DOPN(Const proj1_sp,_)) = get_pat proj1_term and
    (DOPN(Const proj2_sp,_)) = get_pat proj2_term and
    (DOPN(Const sig_elim_sp,_)) = get_pat sig_elim_term in

    strong (whd_betaiota o (whd_const [proj1_sp;proj2_sp;sig_elim_sp] sigma)) app_B

;;

let find_elim P sigma sign eq_ind eq_rec =
 match hd_of_prod (type_of_rel sigma (GLOB sign) P) with
    DOP0(Sort(Prop Null))  ->  get_pat eq_ind  (* Prop *)
 |  DOP0(Sort(Prop Pos))     -> 
      match eq_rec with
         (SOME eq_rec) -> get_pat eq_rec (* Set *) 
      | NONE -> errorlabstrm "find_elim"
           [< 'S "this type of elimination is not allowed">]
 |   _   -> errorlabstrm "find_elim" [< 'S "this type of elimination is not allowed">]
;;

(* |- (P e2)
     BY RevSubstInConcl (eq T e1 e2)
     |- (P e1)
     |- (eq T e1 e2)
 *)
let RevSubstInConcl eqn gls =
let (_,_,_,eq_indt,eq_rect,[T;e1;e2]) = find_eq_data_decompose eqn in
let body = subst_tuple_term (Project gls) (pf_hyps gls) e2 (pf_concl gls) in
if not dependent (Rel 1) body then errorlabstrm  "RevSubstInConcl" [<>];
let PB = DLAM(named_hd T Anonymous,body) in
let P = DOP2(Lambda,T,PB) in
let eq_elim = try find_elim P (Project gls) (pf_hyps gls) eq_indt eq_rect 
              with UserError _ -> errorlabstrm "RevSubstIncConcl" [< 'S "this type of elimination is not allowed">]  in 
  BareRevSubstInConcl eq_elim P (T,e1,e2)  gls
;;



(* |- (P e1)
     BY SubstInConcl (eq T e1 e2)
     |- (P e2)
     |- (eq T e1 e2)
 *)
let SubstInConcl eqn gls =
    (RevSubstInConcl (swap_equands gls eqn) THENS
     [IDTAC;
      SwapEquandsInConcl]) gls
;;



let SubstInHyp eqn id gls =
let (_,_,_,eq_indt,eq_rect,[T;e1;e2]) = (find_eq_data_decompose eqn) in 
let body = subst_term e1 (clause_type (SOME id) gls) in
if not dependent (Rel 1) body then errorlabstrm  "SubstInHyp" [<>];
let PB = DLAM(named_hd T Anonymous,body) in
let P = DOP2(Lambda,T,PB) in
let eq_elim = find_elim P (Project gls) (pf_hyps gls) eq_indt eq_rect 
in  (CutReplacing id (SAPP PB e2) THENS
     [IDTAC;
      BareRevSubstInConcl eq_elim P (T,e1,e2) THENS [Exact (VAR id);IDTAC]]) gls
;;

let RevSubstInHyp eqn id gls =
       (SubstInHyp (swap_equands gls eqn) id THENS
       [IDTAC;
        SwapEquandsInConcl]) gls
;;


let try_rewrite tac gls =
 try  tac gls
 with UserError ("find_eq_data_decompose",_) -> 
         errorlabstrm "find_eq_data_decompose " [< 'S "Not a primitive equality here">]
 | UserError("find_eq_elim",s) -> 
        errorlabstrm "find_eq_elim" [<'S "This type of elimination is not allowed ">]  
 | _ -> IDTAC gls
;;



(* Rewrite c in id. Rewrite -> c in id. Rewrite <- c in id. *)

let rewrite_in LR com id gls =
  (try lookup_sign id (pf_hyps gls) ; () with Not_found -> 
     errorlabstrm "rewrite_in" [< 'S"No such hypothesis : " ; print_id id >]) ;
  let c = pf_constr_of_com gls com in
  let eqn = pf_type_of gls c in
  try (find_eq_data_decompose eqn ;
       try ((if LR then SubstInHyp else RevSubstInHyp) eqn id 
             THENS [ IDTAC ; Exact c ]) gls
       with UserError("SubstInHyp",_) -> IDTAC gls)
  with  UserError ("find_eq_data_decompose",_)->  
          errorlabstrm "rewrite_in" [< 'S"No equality here" >] 

;;

let rewriteLR_in_tac = register_tactic "RewriteLRin"
  (fun [COMMAND com; IDENTIFIER id] -> rewrite_in true com id)
  (fun sigma goal (_,[COMMAND com; IDENTIFIER id]) ->
    [< 'S"Rewrite" ; 'SPC; 'S "->"; 'SPC ; pr_com sigma goal com ; 'SPC ; 'S"in" ; 'SPC ;
       print_id id >])
;;

let rewriteRL_in_tac = register_tactic "RewriteRLin"
  (fun [COMMAND com; IDENTIFIER id] -> rewrite_in false com id)
  (fun sigma goal (_,[COMMAND com; IDENTIFIER id]) ->
    [< 'S"Rewrite" ; 'SPC ; 'S"<-" ; 'SPC ; pr_com sigma goal com ;
       'SPC ; 'S"in" ; 'SPC ; print_id id >])
;;


let Subst eqn cls gls =
    match cls with
    NONE ->    SubstInConcl eqn gls
  | SOME id -> SubstInHyp eqn id gls
;;

(* |- (P a)
   Subst_Concl a=b 
    |- (P b)
    |- a=b
*)
let SubstConcl_LR eqn gls = try_rewrite (Subst eqn NONE) gls;;
let SubstConcl_LR_tac = 
 let gentac = register_tactic "SubstConcl_LR"
  (fun [COMMAND eqn] -> 
        (fun gls ->  SubstConcl_LR (pf_constr_of_com gls eqn)  gls))
  (fun sigma  goal (_,[COMMAND eqn]) ->
     [<'S"CutRewrite ->" ; 'SPC ; pr_com sigma goal eqn >])
 in fun eqn  -> gentac [COMMAND eqn] 
;;


(* id:(P a) |- G
   SubstHyp a=b id
    id:(P b) |- G
    id:(P a) |-a=b
*)
(*******
let SubstHyp_LR  eqn id gls  =  try_rewrite  (Subst eqn (SOME id)) gls ;;
let SubstHyp_LR_tac   = 
let gentac = register_tactic "SubstHyp_LR"
 (fun [COMMAND eqn; IDENTIFIER id] -> 
       (fun gls ->  SubstHyp_LR (pf_constr_of_com gls eqn) id gls))
 (fun sigma  goal (_,[COMMAND eqn; IDENTIFIER id]) ->
    [<'S"CutRewrite ->" ; 'SPC ; pr_com sigma goal eqn; 'SPC; 'S"in"; print_id id >])
in fun eqn id -> gentac [COMMAND eqn; IDENTIFIER id] 
;;
******)

let HypSubst id cls gls =
    match cls with
    NONE -> (SubstInConcl (clause_type (SOME id) gls) THENS
             [IDTAC; Exact (VAR id)]) gls
  | SOME hypid -> (SubstInHyp (clause_type (SOME id) gls) hypid THENS
               [IDTAC;Exact (VAR id)]) gls
;;

(* id:a=b |- (P a)
   HypSubst id.
    id:a=b |- (P b)
*)

let SubstHypInConcl_LR id gls = try_rewrite (HypSubst id NONE) gls;;
let SubstHypInConcl_LR_tac =
let gentac= register_tactic "SubstHypInConcl_LR" 
   (fun [IDENTIFIER id] -> SubstHypInConcl_LR id)
   (fun _ _ (_,[IDENTIFIER id]) -> [< 'S"Dependent Rewrite <-" ; 'SPC; print_id id >])
in fun id -> gentac [IDENTIFIER id]
;;


(* id:a=b H:(P a) |- G
   SubstHypInHyp id H.
    id:a=b H:(P b) |- G
*)
(*******
let SubstHypInHyp_LR id H gls = try_rewrite (HypSubst id (SOME H)) gls;;
let SubstHypInHyp_LR_tac = 
let gentac = register_tactic "SubstHypInHyp_LR"
    (fun [IDENTIFIER id; IDENTIFIER H] -> SubstHypInHyp_LR id H)
    (fun sigma goal  (_,[IDENTIFIER id; IDENTIFIER H]) ->
        [< 'S"GRewrite"; 'S "->"; 'SPC ; print_id id ; 'SPC ; 'S"in"; print_id H >])
in fun id H -> gentac [IDENTIFIER id; IDENTIFIER H]
;;
*******)


let RevSubst eqn cls gls =
    match cls with
    NONE -> RevSubstInConcl eqn gls
  | SOME id -> RevSubstInHyp eqn id gls
;;

(* |- (P b)
   SubstConcl_RL a=b
     |- (P a)
     |- a=b
*) 
let SubstConcl_RL eqn gls = try_rewrite (RevSubst eqn NONE) gls;;
let SubstConcl_RL_tac = 
 let gentac = register_tactic "SubstConcl_RL"
  (fun [COMMAND eqn] -> 
        (fun gls ->  SubstConcl_RL (pf_constr_of_com gls eqn)  gls))
  (fun sigma  goal (_,[COMMAND eqn]) ->
     [<'S"CutRewrite <-" ; 'SPC ; pr_com sigma goal eqn >])
 in fun eqn  -> gentac [COMMAND eqn] 
;;


(* id:(P b) |-G
   SubstHyp_RL a=b id 
      id:(P a) |- G
      |- a=b  
*)
let SubstHyp_RL  eqn id gls = try_rewrite (RevSubst eqn (SOME id)) gls;;
(******
let SubstHyp_RL_tac   = 
let gentac = register_tactic "SubstHyp_RL"
 (fun [COMMAND eqn; IDENTIFIER id] -> 
       (fun gls ->  SubstHyp_RL (pf_constr_of_com gls eqn) id gls))
 (fun sigma  goal (_,[COMMAND eqn; IDENTIFIER id]) ->
    [<'S"CutRewrite <-" ; 'SPC ; pr_com sigma goal eqn; 'SPC; 'S"in"; print_id id >])
in fun eqn id -> gentac [COMMAND eqn; IDENTIFIER id] 
;;
*********)


let RevHypSubst id cls gls =
    match cls with
    NONE -> (RevSubstInConcl (clause_type (SOME id) gls) THENS
             [IDTAC; Exact (VAR id)]) gls
  | SOME hypid -> (RevSubstInHyp (clause_type (SOME id) gls) hypid THENS
               [IDTAC;Exact (VAR id)]) gls
;;

(* id:a=b |- (P b)
   HypSubst id.
    id:a=b |- (P a)
*)
let SubstHypInConcl_RL id gls = try_rewrite (RevHypSubst id NONE) gls;;
let SubstHypInConcl_RL_tac =
let gentac= register_tactic "SubstHypInConcl_RL" 
   (fun [IDENTIFIER id] -> SubstHypInConcl_RL id)
   (fun _ _ (_,[IDENTIFIER id]) -> [< 'S"Dependent Rewrite <-" ; 'SPC; print_id id >])
in fun id -> gentac [IDENTIFIER id]
;;



(* id:a=b H:(P b) |- G
   SubstHypInHyp id H.
    id:a=b H:(P a) |- G
*)
(*****
let SubstHypInHyp_RL id H gls = try_rewrite (RevHypSubst id (SOME H)) gls;;
let SubstHypInHyp_RL_tac = 
let gentac = register_tactic "SubstHypInHyp_RL"
    (fun [IDENTIFIER id; IDENTIFIER H] -> SubstHypInHyp_RL id H)
    (fun sigma goal  (_,[IDENTIFIER id; IDENTIFIER H]) ->
        [< 'S"GRewrite"; 'S "<-"; 'SPC ; print_id id ; 'SPC ; 'S"in"; print_id H >])
in fun id H -> gentac [IDENTIFIER id; IDENTIFIER H]
;;
******)
















(* $Id: equality.ml,v 1.1.2.1 1997/05/05 07:54:05 demaille Exp $ *)
