UNIT Cryptor;

(**********************************************************************

Cryptor is (c) copyright 1995 by Klaus Hartnegg, all rights reserved
Klaus Hartnegg, Kleist-Str. 7, D-79331 Teningen, Germany
e-mail: hartnegg@einstein.freinet.de

Algorithm:

 Use a simple random hash function to create an xor encryption key
 for a block of data. The block number is used to create a different
 code for each block.

 For experts: for every bit in the password and for every bit in the
 block number that changes, each bit in the code might change.
 Thus the most trivial attempts to crack this will fail.
 I use Borlands pseudo random generator as hash function but with
 some tricks to make cracking it more difficult.


Security:

 This Encryption provides only medium security but it needs some
 explanations to understand what this really means:

 This is way better than most of the crypto things floating around today.
 It avoids most of the most trivial errors almost everybody seems to do
 because they have never read a book on the subject.

 Currently this can be cracked by brute force search on a fast PC in about
 one day. I know for sure because I have tried it using a highly optimized
 algorithm. A crypto expert can probably do it in even less time by careful
 analysis of the internal gears and wheels of the pseudo random function.

 But do you really think that a crypto expert has nothing better to do?
 It can most certainly not be cracked by your roommate (except he or she
 is a crypto expert and has nothing else to do today...)
 If you think that the NSA is hunting your secrets, get a better program.
 Or better yet let a layer explain you what you might be doing wrong.


 Security of this algorithm is currently limited by two things:
 - randseed is 32 bits small. This can be cracked on a
   pentium-90 by brute force search in about one day.
 - The pseuro random function is not very good as hash function.
   A crypto expert could use this to attack the algorithm although
   I have tried to make this as difficult as possible. This means
   that it might not be necessary to do a full brute force search.

 For comparison: with a 6 key password made of lower case letters a
 brute force attack on the password is easier than trying all 32 bit
 seed numbers! So if you really want good security first start using
 real passwords before you look for better algorithms!

 Both limits of the algorithm could be improved by using a better random
 number generator.
 Could please anybody send me a free but better generator?


 Of course as always in encryption there might be something that
 I have missed. Maybe there's some small design flaw that would
 enable a crypto expert to reduce the computation time even more.
 But a crypto expert anyway likely has better computers than a
 pentium. Thus he would probably not think that this analysis is
 worth the effort. He can crack it in hours or minutes anyway using
 a highly parallel machine with thousands of processors. So why bother?

 P.S. recently in increased randseed from 32 to 64 bit.
 You are now pretty secure against brute force search
 at least as long as not supercomputers are used.
 But the other problem has not yet been attacked.

Bugs:
 both random seeds equal -> no encryption


**********************************************************************)


INTERFACE

uses
  zrandom;


const
  NumCryptorSeeds = 2;

type

  seed_array = array [1..NumCryptorSeeds] of longint;

  Crypt_Obj = object
           {---------------------------------}
            seed : seed_array;
           {---------------------------------}
            constructor init   (key:string);
            procedure   encode (var data; datasize:word; serial:word);
           {---------------------------------}
          end;

{ usage:                                      }
{ call init once, then encode for each block  }
{ supply the block number as serial number    }
{ decode by calling encode a second time      }


{ simple binary <-> ascii conversion, strips ascii codes <= 32 }
{ beware: strings get one byte longer for each code <= 32 ! }
procedure code_from_asc (var c:string);
procedure asc_from_code (var c:string);


{ this function is exported only for cracing tests: }

Procedure GetKey (seed:seed_array; serial:word; datasize:word;
                  var data);


IMPLEMENTATION


const
  Par1 = 3;
  Par2 = 5;
  Par3 = 63000;

  { par1 >= 2, par2 >= 2 or security will be low against analysis }
  { par1 + par2 <= 50   (smaller = faster but less secure) }
  { 65535/(par1+par2) <= par3 <= 65533 }


type
  bytearray1 = array [1..65535] of byte;
  wordarray1 = array [1..32767] of word;


procedure code_from_asc (var c:string);
var i : byte;
begin
  i := 1;
  while i <= length(c) do begin
     if c[i] = '@' then
        c := copy (c,1,i-1) + chr(ord(c[i+1])-32) + copy (c,i+2,255);
     inc (i);
  end;
end;


procedure asc_from_code (var c:string);
var i : byte;
begin
  i := 1;
  while i <= length(c) do begin
     if (ord(c[i]) <= 32) or (c[i] = '@') then begin
        c := copy (c,1,i-1) + '@' + chr(ord(c[i])+32) + copy (c,i+1,255);
        inc (i,2);
     end
     else inc (i);
  end;
end;



procedure mix;
var i : word;
begin
  {$R-}
  for i := 1 to par1+random(par2) do begin
     inc (randseed, random(par3));
     randseed := randseed shl 5 + randseed shr (32-5);
  end;
  {$R+}
end;


Procedure GetKey (seed: seed_array; serial:word; datasize:word;
                  var data);
var
  dat     : wordarray1 absolute data;
  oldseed : longint;
  i,n     : word;
begin
  oldseed  := randseed;

  fillchar (dat, datasize, 0);

  for n := 1 to NumCryptorSeeds do begin

     { determine initial randseed by mixing in serial }
     randseed := seed[n];
     for i := 1 to 3 do begin
        {$R-}
        inc (randseed, serial);
        {$R+}
        mix;
     end;

     { now fill array with pseudo random numbers calculated with this seed }
     for i := 1 to datasize div 2 do
        dat[i] := dat[i] xor (NextRand shr 16);

  end;

  randseed := oldseed;
end;


Constructor crypt_obj.init (key:string);
var
  oldseed : longint;
  i,j,n   : word;
begin
  oldseed := randseed;

  for n := 1 to NumCryptorSeeds do begin
     randseed := length(key);
     mix;
     for i := 1 to length(key) do begin
        inc (randseed, ord(key[i]));
        mix;
        for j := 1 to n do inc (randseed, random(1001));
        mix;
     end;
     seed[n] := randseed;
  end;

  randseed := oldseed;
end;



Procedure   crypt_obj.encode (var data; datasize:word; serial:word);
var
  dat  : bytearray1 absolute data;
  code : ^bytearray1;
  wdat : wordarray1 absolute data;
  wcode: ^wordarray1;
  i    : word;
  isodd: boolean;
  size : word;
begin
  size := datasize;
  isodd := odd(size);
  if isodd then inc (size);
  getmem (code, size);
  GetKey (seed, serial, size, code^);

{ for i := 1 to datasize do
     dat[i] := dat[i] xor code^[i];
}
  { functional equivalent but faster because it processes 2 bytes a time }
  pointer(wcode) := pointer(code);
  for i := 1 to datasize div 2 do
     wdat[i] := wdat[i] xor wcode^[i];
  if isodd then dat[datasize] := dat[datasize] xor code^[datasize];

  freemem (code, size);
end;


END.
