/****************************************************************************/
/*  video_x  version 0.21                                                   */
/*                                                                          */
/* program to access the video-blaster and compatible cards                 */
/* for the GNU djgpp/gcc compiler                                           */
/*                                                                          */
/* (C)1997 Bernhard Schwall                                                 */
/*                                                                          */
/* version 0.1                                                              */
/*    show the video under X in HiColor and control the sound               */
/*    (on Media Pro Plus)                                                   */
/* version 0.2                                                              */
/*    grab 20 frames without display and select for save                    */
/*    display under TrueColor                                               */
/*                                                                          */
/****************************************************************************/

#include "v_access.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "timer_m.h"

#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))

int uTab[256], vTab[256];
int yTab[128];

Display *dpy;
Screen *scr;
Window win;
GC gc;
XImage *ximage;
XSetWindowAttributes xswa;
int BPP, BPPpot;

/****************************************************************************/

void init(int XSize, int YSize){
char *data = NULL;
char WinName[40];

#ifndef linux
  union REGS ireg, oreg;
#endif
  int i;
  XGCValues gcv;

  V_Initialize("", 1);

  if ((dpy = XOpenDisplay(NULL)) == NULL){
    perror("cannot open display");
    exit(-1);
  }
  scr = DefaultScreenOfDisplay(dpy);
  BPP = DefaultDepthOfScreen(scr);
  if ((BPP != 15) && (BPP != 16) && (BPP != 24) && (BPP != 32)){
    perror("need HiColor or TrueColor display");
    exit(-1);
  }
  
  XSelectInput(dpy, RootWindowOfScreen(scr), SubstructureNotifyMask);
  xswa.background_pixel = BlackPixelOfScreen(scr);
  xswa.event_mask = KeyPressMask|KeyReleaseMask;
  win = XCreateWindow(dpy, RootWindowOfScreen(scr),
        0, 0, XSize, YSize, 0, 
        DefaultDepthOfScreen(scr), InputOutput,
        DefaultVisualOfScreen(scr),
        CWEventMask | CWBackPixel, &xswa);
                              
  sprintf(WinName, "XVideo %ix%ix%i", XSize, YSize, BPP);
  XChangeProperty(dpy, win, XA_WM_NAME, XA_STRING, 8,
                  PropModeReplace, WinName, strlen(WinName));
  XMapWindow(dpy, win);

  XSync(dpy, 0);
  
  gcv.plane_mask = AllPlanes;
  gcv.subwindow_mode = IncludeInferiors;
  gcv.function = GXcopy;
  gcv.foreground = WhitePixelOfScreen(scr);
  gcv.background = BlackPixelOfScreen(scr);
  gc = XCreateGC(dpy, RootWindowOfScreen(scr),
       GCFunction|GCPlaneMask|GCSubwindowMode|GCForeground|GCBackground,
       &gcv);
  switch (BPP){
    case 15:
    case 16: data = malloc(XSize*YSize*2);
             BPPpot = 2;
             break;
    /* change this if you've a real 24BPP display */
    case 24:
    case 32: data = malloc(XSize*YSize*4);
             BPPpot = 4;
             BPP = 32;
             break;
  }
  ximage = XCreateImage(dpy, DefaultVisualOfScreen(scr),
                        DefaultDepthOfScreen(scr), ZPixmap, 0, data,
                        XSize, YSize, 16, XSize*BPPpot);
  if (ximage == NULL){
    perror("XCreateImage");
    exit(-1);
  }

  for (i=-128; i<128; i++){
    uTab[i+128] = (i * 226) / 127;
    vTab[i+128] = (i * 179) / 127;
  }
  
  if (BPP == 15){
  
    for (i=0; i<128; i++){
      yTab[i] = ((i << 8) & 0x7c00) |
                ((i << 3) & 0x03e0) |
                ((i >> 2) & 0x001f);
    }
  }
  else{
    for (i=0; i<128; i++){
      yTab[i] = ((i << 9) & 0xf800) |
                ((i << 4) & 0x07e0) |
                (i >> 2);
    }
  }
}

/****************************************************************************/

void done(){
  free(ximage->data);
  ximage->data = NULL;
  XDestroyImage(ximage);
  V_Exit();
}

/****************************************************************************/

void YUVtoRGB(char *InBuffer, char *OutBuffer, int Pixel, int Depth)
/* read a line from the card and convert it to RGB 16 or 24 bit BGR */
{
  BYTE *src=InBuffer;
  register BYTE tmp;
  INT  z, r, g, b, u, v, y[4], *yz, count=1;
  signed char uBild, vBild;
  unsigned short *OutBuf = (unsigned short *)OutBuffer;

  do{
    y[0] = *src++ | 1;
    tmp = *src++;
    uBild = tmp & 0xc0;
    vBild = (tmp & 0x30) << 2;
    y[1] = *src++ | 1;
    tmp = *src++;
    uBild |= (tmp & 0xc0) >> 2;
    vBild |= (tmp & 0x30);
    y[2] = *src++ | 1;
    tmp = *src++;
    uBild |= (tmp & 0xc0) >> 4;
    vBild |= (tmp & 0x30) >> 2;
    y[3] = *src++ | 1;
    tmp = *src++;
    uBild |= tmp >> 6;
    vBild |= (tmp & 0x30) >> 4;

    u = uTab[uBild+128];
    v = vTab[vBild+128];

    yz = y;
    for (z=0; z<4; z++){
      r = max(min(*yz+v, 255), 0);
      b = max(min(*yz+u, 255), 0);
      g = ((*yz*436L) - (r*130U) - (b*50U)) >> 8;
      g = max(min(g, 255), 0);
      switch (Depth){
      case 15:
        *OutBuf++ = ((r << 7) & 0x7c00) |
                    ((g << 2) & 0x03e0) |
                    ((b >> 3) & 0x001f);
        break;
      case 16:
        *OutBuf++ = ((r << 8) & 0xf800) |
                    ((g << 3) & 0x07e0) |
                    (b >> 3);
        break;
      case 24:
        *OutBuffer++ = b;
        *OutBuffer++ = g;
        *OutBuffer++ = r;
        break;
      case 32:
        *OutBuffer++ = b;
        *OutBuffer++ = g;
        *OutBuffer++ = r;
        *OutBuffer++ = 0;
        break;
      }
      yz++;
    }
  } while ((count += 4) < Pixel);
}

/***************************************************************************/

void WriteFileTGA(char *Buffer, char *Name, int XRes, int YRes){
FILE *f;
char *LBuffer;
int  y;
char Header[18];

  if ((f = fopen(Name, "wb")) == NULL) return;
  
  LBuffer = malloc(3*(XRes+3));
      
  memset(Header, 0, 18);
  Header[2] = 2;            /* uncompressed 24 bit */
  Header[12] = XRes & 0xff;
  Header[13] = XRes >> 8;
  Header[14] = YRes & 0xff;
  Header[15] = YRes >> 8;
  Header[16] = 24;          /* 24 bit/pixel */
  Header[17] = 0x20;        /* top-down, non interlaced */
  fwrite(Header, 18, 1, f);

  for (y=0; y<YRes; y++){
    YUVtoRGB(&Buffer[y*XRes*2], LBuffer, XRes, 24);
    fwrite(LBuffer, 3, XRes, f);
  }

  fclose(f);
  free(LBuffer);
}

/***************************************************************************/

void Grab_video(char *Buffer, int SizeX, int SizeY){
  int y;
  BYTE *dest=(BYTE *)Buffer;
  BYTE *src =(BYTE *)VidMem;
    
  SizeX *= 2;
  V_FreezeVideo();
  for (y=0; y<SizeY; y++){
    memmove(dest, src, SizeX);
    src += 2048L;
    dest+= SizeX;
  }
  V_UnfreezeVideo();
}

/***************************************************************************/

void Display_video(char *Buffer, int SizeX, int SizeY, int Quality){
  int x, y;
  BYTE *dest=(BYTE *)Buffer;
  unsigned short *xpic;
    
  if (Quality == 0){
    xpic = (unsigned short *)&ximage->data[ximage->xoffset];
    dest = &ximage->data[ximage->xoffset];
    for (y=0; y<SizeY; y++){
      for (x=0; x < SizeX; x++){
        switch (BPP){
        case 15:
        case 16: *xpic++ = yTab[((BYTE)*Buffer) >> 1];
                 break;
        case 32: *dest++ = (BYTE) *Buffer;
                 *dest++ = (BYTE) *Buffer;
                 *dest++ = (BYTE) *Buffer;
                 *dest++ = 0;
                 break;
        }
        Buffer += 2;
      }
    }
  }
  else{
    dest = &ximage->data[ximage->xoffset];
    for (y=0; y<SizeY; y++){
      YUVtoRGB(Buffer, dest, SizeX, BPP);
      Buffer += SizeX*2;
      dest += ximage->bytes_per_line;
    }
  }
  XPutImage(dpy, win, gc, ximage, 0,0,0,0, SizeX, SizeY);
}

/**************************************************************************/

void record(char *Buffer, int SizeX, int SizeY, int MaxCount, int *PicNumber){
  int Nr, x, y;
  BYTE *FBuffer[MaxCount];
  char WinName[30];
  int ch = 0;
  char FileName[12];
  BYTE *dest, *src, uv, y2;
  XEvent event;
  
  for (Nr=0; Nr<MaxCount; Nr++){
    sprintf(WinName, "recording frame %i", Nr+1);
    XChangeProperty(dpy, win, XA_WM_NAME, XA_STRING, 8,
                    PropModeReplace, WinName, strlen(WinName));
    XSync(dpy, 0);
    V_WaitGrabbed();
    V_FreezeVideo();
    dest = (BYTE *)Buffer;
    for (y=0; y<SizeY; y++){
      memmove(dest, VidMem+2048L*y, SizeX*2);
      dest+= SizeX*2;
    }
    V_UnfreezeVideo();
    FBuffer[Nr] = (BYTE *)malloc(SizeX*SizeY+(SizeX*(SizeY/2)));
    src = (BYTE *)Buffer;
    dest = FBuffer[Nr];
    for (y=0; y<SizeY; y++){
      for (x=0; x<SizeX; x+=2){
        *dest++ = *src++;
        uv = *src++ & 0xf0;
        *dest++ = *src++;
        uv |= *src++ >> 4;
        *dest++ = uv;
      }
    }
  }
  Nr = 0;
  do{
    if (ch != 'w'){
      sprintf(WinName, "display frame %i", Nr+1);
      XChangeProperty(dpy, win, XA_WM_NAME, XA_STRING, 8,
                      PropModeReplace, WinName, strlen(WinName));
    }
    src = FBuffer[Nr];
    dest = (BYTE*) Buffer;
    for (y=0; y<SizeY; y++){
      for (x=0; x<SizeX; x+= 2){
        *dest++ = *src++;
        y2 = *src++;
        uv = *src++;
        *dest++ = uv & 0xf0;
        *dest++ = y2;
        *dest++ = (uv & 0x0f) << 4;
      }
    }
    Display_video(Buffer, SizeX, SizeY, 1);
    while (XCheckWindowEvent(dpy, win, (long)-1, &event) == FALSE);
    if (event.type == KeyPress){
      ch = XKeycodeToKeysym(dpy, event.xkey.keycode,0);
      switch(ch){
        case XK_Right: if (Nr < (MaxCount-1)) Nr++;
                  break;
        case XK_Left: if (Nr > 0) Nr--;
                  break;
        case 'w': sprintf(FileName, "pic%03i.tga", *PicNumber);
                  *PicNumber += 1;
                  WriteFileTGA(Buffer, FileName, SizeX, SizeY);
                  sprintf(WinName, "%s saved", FileName);
                  XChangeProperty(dpy, win, XA_WM_NAME, XA_STRING, 8,
                                  PropModeReplace, WinName, strlen(WinName));
                  break;
      }
    }
  } while (ch != XK_Escape);
  for (Nr=0; Nr<MaxCount; Nr++){
    free(FBuffer[Nr]);
  }
  sprintf(WinName, "XVideo %ix%i", SizeX, SizeY);
  XChangeProperty(dpy, win, XA_WM_NAME, XA_STRING, 8,
                  PropModeReplace, WinName, strlen(WinName));
}

/**************************************************************************/

void Help(){
  printf("XVideo V0.21  (C)1997 by B.Schwall\n");
  printf("video_x [optinos]\n");
  printf(" -s SizeX SizeY\n");
  printf("        SizeX  horizontal video resolution\n");
  printf("        SizeY  vertical video resolution\n");
  printf(" -t FPS\n");
  printf("        FPS    maximum number of frames displayed per second\n");
  printf(" -n Num\n");
  printf("        Num    number of frames recorded with 'r'\n");
  printf("\n");
  printf(" used keys:\n");
  printf(" l,+  increase volume\n");
  printf(" q,-  decrease volume\n");
  printf(" m    mute\n");
  printf(" b    switch between BW and color display\n");
  printf(" s    freeze and unfreeze the video\n");
  printf(" #    change freezed frames\n");
  printf(" w    save frame\n");
  printf(" r    record 15 frames for saving\n");
  printf("      cursor left  display next frame\n");
  printf("      cursor right display previous frame\n");
  printf("      w            save frame\n");
}

/**************************************************************************/

int main(int argc, char *argv[])
{
  char ch = 0;
  int Mute = 0;
  int Volume;
  int Freeze = 0;
  int Quality = 1;
  int XSize, YSize, NumFrames;
  char *Buffer[2];
  int BufferNr = 0;
  int PicNr = 0;
  char FileName[12];
  unsigned long time1, time2, time3, timeout;
  long Count=0;
  XEvent event;

  XSize = 240;
  YSize = 180;
  timeout = 12;
  NumFrames = 20;
  while (argc > 1){
    switch (argv[1][1]){
    case 'h': Help();
              exit(0);
    case 's': if (argc < 4) {
                printf("missing argument for -s\n");
                exit(0);
              }
              sscanf(argv[2], "%i", &XSize);
              sscanf(argv[3], "%i", &YSize);
              argv += 2;
              argc -= 2;
              break;
    case 't': { int i;
                if (argc < 3) {
                  printf("missing argument for -t\n");
                  exit(0);
                }
                sscanf(argv[2], "%i", &i);
                timeout = i;
                argv++;
                argc--;
              }
              break;
    case 'n': if (argc < 3) {
                pritnf("missing argument for -v\n");
                exit(0);
              }
              sscanf(argv[2], "%i", &NumFrames);
              argv++;
              argc--;
              break;
    }
    argc--;
    argv++;
  }
  timeout = (unsigned long)CLOCKS_PER_SEC / timeout;
  
  init(XSize, YSize);

  Volume = V_GetVolume();
  V_CreateWindow(0,0,XSize,YSize); 

  V_DisableVideo();
  Buffer[0]=(char *)malloc(XSize*YSize*BPPpot);
  Buffer[1]=(char *)malloc(XSize*YSize*BPPpot);
  InitializeTimer();

  do{
    time1=ReadTimer();
    while (XCheckWindowEvent(dpy, win, (long)-1, &event) == FALSE){
      if (Freeze == 0){
        time3 = ReadTimer();
        Grab_video(Buffer[BufferNr], XSize, YSize);
        Display_video(Buffer[BufferNr], XSize, YSize, Quality);
        BufferNr ^= 1;
        Count++;
        while ((ReadTimer()-time3) < timeout);
      } /* Freeze == 0 */
    } /* while */
    time2=ReadTimer()-time1;
    if (event.type == KeyPress){
      ch = XKeycodeToKeysym(dpy, event.xkey.keycode,0);
      switch(ch){
        case 's': Freeze ^= 1;
                  if (Freeze == 1){
                    Display_video(Buffer[BufferNr], XSize, YSize, Quality);
                  }
                  break;
        case 'l': Volume += 8;
                  if (Volume > 255) Volume = 255;
                  V_SetVolume(Volume);
                  break;
        case 'q': Volume -= 8;
                  if (Volume < 0) Volume = 0;
                  V_SetVolume(Volume);
                  break;
        case '+': Volume += 2;
                  if (Volume > 255) Volume = 255;
                  V_SetVolume(Volume);
                  break;
        case '-': Volume -= 2;
                  if (Volume < 0) Volume = 0;
                  V_SetVolume(Volume);
                  break;
        case 'm': if (Mute == 0){
                    Mute = Volume;
                    V_SetVolume(0);
                  }
                  else {
                    V_SetVolume(Mute);
                    Mute = 0;
                  }
                  break;
        case 'b': Quality ^= 1;
                  Count = 0;
                  if (Freeze == 1){
                    Display_video(Buffer[BufferNr], XSize, YSize, Quality);
                  }
                  break;
        case 'w': sprintf(FileName, "pic%03i.tga", PicNr++);
                  WriteFileTGA(Buffer[BufferNr], FileName, XSize, YSize);
                  Freeze = 0;
                  break;
        case '#': if (Freeze == 1) {
                    BufferNr ^= 1;
                    Display_video(Buffer[BufferNr], XSize, YSize, Quality);
                  }
                  break;                    
        case 'r': record(Buffer[BufferNr], XSize, YSize, NumFrames, &PicNr);
                  break;
      } /* switch(ch) */
    } /* key pressed */
    else{
      ch = 0;
    }
  } while (ch != 27);
  done();
  free(Buffer[0]);
  free(Buffer[1]);
  printf("%iX%i: %i pictures, nearly %f frames per second\n",
          XSize, YSize, (int)Count, 
          (float)Count/((float)time2/CLOCKS_PER_SEC));
  RestoreTimer();
  return 0;
}
