/*
sub2ppm  copyright 2005 Jan Panteltje.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "sub2ppml.h"


char *strsave(char *s) /* save string s somewhere */
{
char *p;

p = malloc(strlen(s) + 1);
if(!p) return 0;
strcpy(p, s);
return p;
} /* end function strsave */



int print_usage()
{
fprintf(stdout, "\
Usage:               sub2ppml [-d] [-f fps] [-h] -i file.sub -o file.ppml [-p] [-s slice_level]\n\
-d                   debug (prints functions and arguments).\n\
-f fps               frames per second (float) (default 25.0).\n\
-h                   help (this help).\n\
-i filename.sub      file generated by vob2sub (submux-dvd package).\n\
-o outputfile.ppml   ppml file for control of subtitler-yuv or transcode subtitler, (default out.ppml).\n\
-p                   enables preview in transcode subtitler or subtitler_yuv (default off).\n\
-s slicelevel        slice level for the .ppm files, this sets when outline becomes visible (default 18).\n\
");

return 1;
} /* end function print_usage */



int main(int argc, char **argv)
{
int a;
FILE *sub_fptr;
FILE *ppml_fptr;
FILE *pptr;
int start_hour, start_minute, start_second, start_millisecond;
int stop_hour, stop_minute, stop_second, stop_millisecond;
double dstart, dstop;
int xpos, ypos;
char *bitmap_filename;
int line;
char temp[80];
int start_frame, stop_frame, old_start_frame, old_stop_frame;
double fps;
int slice_level;
int preview_flag;

/* defaults */
output_filename = strsave("out.ppml");
if(! output_filename)
	{
	fprintf(stderr, "sub2ppml: main(): could not allocate space for output_filename, aborting.\n");

	exit(1);
	}


fprintf(stderr, "\nsub2ppml-%s Copyright Jan Panteltje 2005 and later\n", VERSION);

/* defaults */
debug_flag = 0;

line = 0;
start_frame = 0;
stop_frame = 0;
old_start_frame = -1;
old_stop_frame = -1;
fps = 25.0;
slice_level = 18;
input_filename = NULL;
preview_flag = 0;
/* end defaults */

while(1)
	{
	a = getopt(argc, argv, "df:hi:o:ps:");
	if(a == -1) break;

	switch(a)
		{
		case 'd':
			debug_flag = 1;
			break;
		case 'f':
			fps = atof(optarg);
			if(fps <= 0.0)
				{
				fprintf(stderr, "sub2ppml: main(): fps should be a positive value, aborting.\n");

				exit(1);				
				}
			break;
		case 'h':
			print_usage();
			exit(1);
			break;
		case 'i':
			input_filename = strsave(optarg);
			if(! input_filename)
				{
				fprintf(stdout, "sub2ppml: main(): could not allocate space for input_filename, aborting.\n");

				exit(1);
				}

			break;
		case 'o':
			output_filename = strsave(optarg);
			if(! output_filename)
				{
				fprintf(stdout, "sub2ppml: main(): could not allocate space for output_filename, aborting.\n");

				exit(1);
				}

			break;
		case 'p':
			preview_flag = 1;
			break;
		case 's':
			slice_level = atoi(optarg);
			break;
        case -1:
        	break;
		case '?':
			if (isprint(optopt) )
 				{
 				fprintf(stdout, "sub2ppml: main(): unknown option `-%c'.\n", optopt);
 				}
			else
				{
				fprintf(stdout, "sub2ppml: main(): unknown option character `\\x%x'.\n",\
				optopt);
				}
			print_usage();

			exit(1);
			break;			
		default:
			print_usage();

			exit(1);
			break;
		}/* end switch a */
	}/* end while getopt() */

if(debug_flag)
	{
	fprintf(stderr,\
	"debug_flag=%d input_filename=%s output_filename=%s preview_flag=%d fps=%.2f slice_level=%d\n",
	debug_flag, input_filename, output_filename, preview_flag, fps, slice_level);
	}

sub_fptr = fopen(input_filename, "r");
if(! sub_fptr)
	{
	fprintf(stderr, "sub2ppml: main(): could not open file %s for read, aborting.\n", input_filename);
	print_usage();

	exit(1);		
	}

ppml_fptr = fopen(output_filename, "w");
if(! ppml_fptr)
	{
	fprintf(stderr, "sub2ppml: main(): could not open file %s for write, aborting.\n", output_filename);

	exit(1);		
	}

/* create ppml header */
fprintf(ppml_fptr, "*main main_movie\n");
fprintf(ppml_fptr, "1 *main show_output=%d\n\n", preview_flag);

while(1)
	{
	// sub0000.bmp 00:00:38,89 00:00:43,26 624 68  53  500

	a = fscanf(sub_fptr, "%as %d:%d:%d,%d %d:%d:%d,%d %d %d %d %d",\
	&bitmap_filename,\
	&start_hour, &start_minute, &start_second, &start_millisecond,\
	&stop_hour, &stop_minute, &stop_second, &stop_millisecond,\
	&a, &a,\
	&xpos, &ypos);

	line++;
	if(a == EOF) break;

	if(a != 13)
		{
		fprintf(stderr, "sub2ppml: main(): syntax error in file %s line %d, aborting.\n",\
		input_filename, line);

		exit(1);
		}

	if(debug_flag)
		{
		fprintf(stderr,\
		"main(): bitmap_filename=%s start_hour=%d start_minute=%d start_second=%d start_millisecond=%d\n\
stop_hour=%d stop_minute=%d stop_second=%d stop_millisecond=%d\n\
xpos=%d ypos=%d\n",\
		bitmap_filename,\
		start_hour, start_minute, start_second, start_millisecond,\
		stop_hour, stop_minute, stop_second, stop_millisecond,\
		xpos, ypos);
		}


	/* calculate the start_frame and stop_frame */

	dstart =\
	(3600000.0 * start_hour) + (60000.0 * start_minute) + (start_second * 1000.0) + start_millisecond;
	start_frame = (int)( (dstart / 1000.0) * fps ); 

	dstop =\
	(3600000.0 * stop_hour) + (60000.0 * stop_minute) + (stop_second * 1000.0) + stop_millisecond;
	stop_frame = (int)( (dstop / 1000.0) * fps ); 

	if(debug_flag)
		{
		fprintf(stderr,\
		"sub2ppml: main(): line=%d start_frame=%d stop_frame=%d\n",\
		line, start_frame, stop_frame);
		}

	/* convert .bmp pictures to .ppm */
	sprintf(temp, "bmptoppm %s > %s.ppm 2>/dev/zero", bitmap_filename, bitmap_filename); 
	pptr = popen(temp, "w");		
	pclose(pptr);

	/* write to ppml file */

	/* prevent problems */
	if(start_frame <= old_stop_frame) old_stop_frame--;

	// remove any old subtitle
	// 20 *sun0 kill
	if(old_stop_frame > 0) fprintf(ppml_fptr, "%d *sub%d kill\n", old_stop_frame, old_start_frame); 

	// define a new subtitle object
	// *sun0 picture /video/test2/sub0000.ppm
	fprintf(ppml_fptr, "*sub%d picture %s.ppm\n", start_frame, bitmap_filename);

	// set the subtitle properties, make it appear
	// 1 *sun0 slice=18 xpos=20 ypos=200
	fprintf(ppml_fptr, "%d *sub%d slice=%d xpos=%d ypos=%d\n\n",\
	start_frame, start_frame, slice_level, xpos, ypos);

	/* free memory */	
	free(bitmap_filename);

	old_start_frame = start_frame;
	old_stop_frame = stop_frame;

	} /* end for all lines in .sub file */


fclose(ppml_fptr);
fclose(sub_fptr);

fprintf(stderr, "sub2ppml: ready\n");

exit(0);
} /* end function main */

