/*
 * This file is part of the fbdvd program
 * Copyright (C) 2001 Mark Sanderson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <sys/time.h>
#include "ac3/ac3.h"
#include "input/dvd.h"
#include "input/nav_read.h"
#include "output/sound.h"
#include "output/video.h"
#include "output/fbtools.h"
#include "play.h"
#include "main.h"

int video_width;
int video_height;
int video_aspect;
int video_framerate = 0;
int video_bitrate;
int video_bufsize;
int video_constrained;

double frame_time, frame_length;
int frame_count=0, samples_count=0;
double audio_time;
double now;
double samples_delay;
double last_frame;

int ac3_started, audio_sync;

mpeg2dec_t mpeg;
unsigned char *mpeg_start, *mpeg_ptr, *mpeg_end;
dsi_t dsi_pack;
int vobu_next, vobu_end;

#define STCODE(p,a,b,c,d) ((p)[0] == a && (p)[1] == b && (p)[2] == c && (p)[3] == d)

#define FILL_MIN 0
#define FILL_INCR 1
#define FILL_MAX 2

typedef struct queue_s {
   int in, out, used, n, min, max;
   struct block {
      unsigned char *data;
      int len, block;
   } *entry;
} queue_t;

typedef struct block_t {
   int used;
   off_t pos;
   double scr, pts;
   int sequence_start;
} block_info;

dvd_block *buffers;
queue_t video_queue, audio_queue;
int buffer_full;
int buffer_in;
int buffer_n;
int buffer_max;
block_info *buffer_info;

ac3_frame_t *ac3_frame;
int video_ready;

double mpeg2_framelength[] = {
   0, 1.001/24, 1.0/24, 1.0/25, 1.001/30, 1.0/30, 1.0/50, 1.001/60, 1.0/60,
   0, 0, 0, 0, 0, 0, 0
};

double dvd_framelength[] = {
   1.0/30, 1.0/25, 1.001/30, 1.001/24
};

char *mpeg2_aspect[] = {
      "Forbidden",
      "1:1",
      "4:3",
      "16:9",
      "2.21:1",
      "Invalid Aspect Ratio",
      "Invalid Aspect Ratio",
      "Invalid Aspect Ratio"
};


void queue_init(queue_t *queue, int max);
void queue_close(queue_t *queue);
int queue_add(queue_t *queue, unsigned char *data, int len);
unsigned char *queue_get(queue_t *queue, int *len);
void queue_flush(queue_t *queue, int n);


void play_init() {

   /* Based on a 1 video/1 audio stream sample:
    * Ratio of video:audio data 28.3784451241:1
    * average sequence is 1 nav packet, 3 audio packets, 84 video packets
    *                  of 88 packets = 250ms (6 frames, 12000 samples)
    * average audio packet = (ac3=2 samples/byte, 1536 samples/768 byte frame, mpeg=2015 bytes/packet)
    *                  4030 samples, 2.62 ac3 frames, 0.0839579413392 seconds
    * average video frame = 14.06 packets
    *
    * each buffer is 250ms
    */
   
   queue_init(&video_queue, 84);
   queue_init(&audio_queue, 3);

   buffer_full = 0;
   buffer_in = 0;
   buffer_n = 0;
   buffer_max = (88 * input_buffers) + 1;
   buffers = (dvd_block *)calloc(buffer_max, sizeof(dvd_block));
   buffer_info = (block_info *)calloc(buffer_max, sizeof(block_info));
   if (!buffers || !buffer_info) {
      fprintf(stderr, "Out of memory\n");
      exit(1);
   }
   
   ac3_frame = 0;
   ac3_started = 0;
   audio_sync = 0;
   frame_time = 0;
   /* This sets 23.976 fps which is always used during film_mode */
   frame_length = 1.001 / 24;
   video_ready = 0;
   last_frame = 0;
   
   mpeg2_init(&mpeg, vo_mm_accel, &vo.vo);
   mpeg2_drop(&mpeg, drop_frames);
   mpeg_start = 0;
}

void play_close() {
   free(buffers);
   free(buffer_info);
}

void queue_init(queue_t *queue, int buffer_size) {
   bzero(queue, sizeof(queue_t));
   queue->min = buffer_size;
   queue->max = buffer_size * input_buffers;
   queue->entry = (struct block *)calloc(queue->max, 3 * sizeof(struct block));
   if (!queue->entry) {
      fprintf(stderr, "Out of memory\n");
      exit(1);
   }
}

void queue_close(queue_t *queue) {
   free(queue->entry);
}

/* Link a block from buffers to a queue */
int queue_add(queue_t *queue, unsigned char *data, int len) {
   if (queue->n == (queue->max - 1))
      return 1;

   buffer_info[buffer_in].used = 1;
   buffer_n++;
   
   queue->entry[queue->in].block = buffer_in;
   queue->entry[queue->in].data = data;
   queue->entry[queue->in].len = len;
   queue->n++;
   queue->in++;
   if (queue->in == queue->max)
      queue->in = 0;
   
   return 0;
}

/* Retrieve a block from a queue, but don't free it from buffers */
unsigned char *queue_get(queue_t *queue, int *len) {
   unsigned char *data;
   
   if (queue->out == queue->in)
      return 0;
   
   data = queue->entry[queue->out].data;
   *len = queue->entry[queue->out].len;

   queue->out++;
   if (queue->out == queue->max)
      queue->out = 0;
   
   return data;
}

/* Unlink n retrieved blocks from buffers */
void queue_flush(queue_t *queue, int n) {

   while (n-- && queue->used != queue->out) {

      buffer_info[queue->entry[queue->used].block].used = 0;
      buffer_n--;

      queue->n--;
      queue->used++;
      if (queue->used == queue->max)
         queue->used = 0;
   }
}

void parse_mpeg(unsigned char *data) {
   char txt[256];
   
   video_width       = (data[0]<<4) | (data[1]>>4);
   video_height      = ((data[1]&15)<<8) | data[2];
   video_aspect      = (data[3]>>4)&7;
   /* Set from the navigation packet instead
    * Why does the rate say 29.97 when only 23.976 frames are delivered?
    * video_framerate   = data[3]&15;
    * frame_length = mpeg2_framelength[video_framerate];
    */
   video_bitrate     = (data[4]<<10) | (data[5]<<2) | (data[6]>>6);
   video_bufsize     = ((data[6]&63)<<4) | (data[7]>>4);
   video_constrained = (data[7]>>4) & 1;

#ifdef SHOW_INFO
   sprintf(txt, "MPEG2: %dx%d %s %.3ff/s %.2fMb/s %dkb buffer %s",
	   video_width, video_height,
	   mpeg2_aspect[video_aspect],
	   1.0 / frame_length,
	   (double)video_bitrate * 0.0004, video_bufsize * 2,
	   video_constrained ? "Constrained parameters" : "Not constrained");
   fb_puts(0,1,txt);
#endif
}

double get_scr(int block) {
   unsigned char *data = (unsigned char *)&buffers[block];
   unsigned long SCR_ms, SCR_ls;
   
   SCR_ms = ((data[4]>>3)&0x7)<<30;
   SCR_ms |= (((((data[4]&0x3)<<13)|(data[5]<<5)|(data[6]>>3))&0x7fff) << 15);
   SCR_ms |= (((data[6]&0x3)<<13)|(data[7]<<5)|(data[8]>>3))&0x7fff;
   SCR_ls = ((data[8]&0x3)<<7)|(data[9]>>1);
   scr = ((SCR_ls / 300.0) + SCR_ms) / 90000.0;
   
   return scr;
}

int parse_block() {
   unsigned char *data, *pes;
   int len, peslen, hdrlen;
   
   data = (unsigned char *)&buffers[buffer_in];
   buffer_info[buffer_in].pos = dvd_pos - 1;
   buffer_info[buffer_in].scr = get_scr(buffer_in);
   buffer_info[buffer_in].pts = 0;
   buffer_info[buffer_in].sequence_start = 0;
   
   if (!STCODE (data,0x00,0x00,0x01,0xba)) {   
      // fprintf(stderr, "Not Pack start code\n");
      return -1;
   }
   
   pes = data + 14 + (data[13] & 0x07);
   
   /* System Header Pack Layer */
   if (STCODE (pes,0x00,0x00,0x01,0xbb)) {
      peslen = (pes[0x04] << 8) + pes[0x05] + 6;
      pes += peslen;
   }
   if (pes[0x00] || pes[0x01] || pes[0x02] != 0x01 || pes[0x03] < 0xb3)
      return 0;
   
   peslen = (pes[0x04] << 8) + pes[0x05];
   hdrlen = pes[0x08] + 3;
   data = pes + hdrlen + 6;
   len = peslen - hdrlen;
   
   switch (pes[0x03]) {
      
    case 0xe0: /* Primary video */
      if (STCODE(data, 0x00,0x00,0x01,0xb3)) {
	 parse_mpeg(data + 4);
	 buffer_info[buffer_in].sequence_start = 1;
      }
      
      if (pes[0x7] & 0x80) {
	 unsigned long PTS;
	 PTS = (pes[0x9] >> 1) & 0x03ULL;
	 PTS = (PTS << 8) | (pes[0xa] & 0xFFULL);
	 PTS = (PTS << 7) | ((pes[0xb] >> 1) & 0x7FULL);
	 PTS = (PTS << 8) | (pes[0xc] & 0xFFULL);
	 PTS = (PTS << 7) | ((pes[0xd] >> 1) & 0x7FULL);
	 buffer_info[buffer_in].pts = PTS / 90000.0;
      }
      return queue_add(&video_queue, data, len);
      
    case 0xbd: /* AC3 audio */
      if (pes[0x7] & 0x80) {
	 unsigned long PTS;
	 PTS = (pes[0x9] >> 1) & 0x03ULL;
	 PTS = (PTS << 8) | (pes[0xa] & 0xFFULL);
	 PTS = (PTS << 7) | ((pes[0xb] >> 1) & 0x7FULL);
	 PTS = (PTS << 8) | (pes[0xc] & 0xFFULL);
	 PTS = (PTS << 7) | ((pes[0xd] >> 1) & 0x7FULL);
	 buffer_info[buffer_in].pts = PTS / 90000.0;
      }
      
      if (*data == (0x80 | audio_track)) {
	 int ac3_start;
	 
	 if (ac3_started) {
	    ac3_start = 4;
	 } else {	    
	    ac3_start = (data[2]<<8) + data[3] + 3;
	    ac3_started = 1;
	 }
         return queue_add(&audio_queue, data + ac3_start, len - ac3_start);
      }
      return 0;
      
    case 0xb9: /* Program finish */
      dvd_eof = 1;
      return 0;
      
    case 0xbf: /* pci/navigation packet */
      navRead_DSI( &dsi_pack, &((unsigned char *)&buffers[buffer_in])[1031], sizeof(dsi_t) );
      if (dvd_angle > 1) {
	 int offset = dsi_pack.sml_agli.data[ dvd_angle - 1 ].address;
	 if (offset != 0);
            dvd_vobseek(dvd_pos + (off_t)offset);
      }
      vobu_end = (int)dvd_pos + dsi_pack.dsi_gi.vobu_ea;
      if (dsi_pack.vobu_sri.next_vobu != SRI_END_OF_CELL ) {
	 vobu_next = (int)dvd_pos + ( dsi_pack.vobu_sri.next_vobu & 0x7fffffff );
      } else {
	 dvd_next_cell();
	 vobu_next = dvd_start_block;
      }
      /* Use this instead of the mpeg specified rate, see decode_frame() */
      if (!film_mode) {
         video_framerate = (dsi_pack.dsi_gi.c_eltm.frame_u & 0xc0) >> 6;
         frame_length = dvd_framelength[video_framerate];
      }
      return 0;
      
    default: /* MPEG audio */
      if (pes[0x03] == (0xc0|audio_track)) {
         return 0;//queue_add(&mpeg_audio_queue, data, len);
      }
      
      fprintf(stderr,"Unparsed mpeg pes packet = %#x\n", pes[3]);
   }
   return 0;
}

int ac3_fill_buffer(uint_8 **bstart, uint_8 **bend) {
   unsigned char *data;
   int len;
   
   queue_flush(&audio_queue, 1);
   data = queue_get(&audio_queue, &len);
   if (data) {
      *bstart = data;
      *bend = data + len;
      return 0;      
   }
   return 1;
}

int decode_frame() {
   int n, pts_block, frames, len;

   while (mpeg_start || video_queue.out != video_queue.in) {

      pts_block = video_queue.entry[video_queue.out].block;
      if (buffer_info[pts_block].pts > 0) {
	 /* Assumption: The PTS represents the start of the buffer containing the decoded sequence
	  * of pictures (access unit), which for DVDs should always be 250ms long */
	 double pts = buffer_info[pts_block].pts - 0.250;
	 
	 /* If pts goes backwards, assume it has been reset. eg when entering a new cell */
	 if (pts < (frame_time - (1.0/90))) {
	    audio_time -= (frame_time - pts);
	    audio_sync = 2;
	 }
	 frame_time = pts;
      }
      
      if (!mpeg_start) {
         mpeg_start = queue_get(&video_queue, &len);
         mpeg_ptr = mpeg_start;
	 mpeg_end = mpeg_start + len;
      }
      if (!mpeg_start)
         return 0;
      
      frames = frames_in;
      n = mpeg2_decode_frame(&mpeg, &mpeg_ptr, mpeg_end);
      if (n) {
         frame_count += n;
	 frame_time += n * frame_length;
	 return (frames_in > frames);
      }
      if (mpeg_ptr >= mpeg_end) {
         queue_flush(&video_queue, 1);
	 mpeg_start = 0;
      }

   }
   return 0;
}

void decode_audio() {
   if (audio_queue.in != audio_queue.out) {
      int pts_block = audio_queue.entry[audio_queue.out].block;
      
      ac3_frame = ac3_decode_frame();
      if (ac3_frame) {
         samples_count += AC3_BLOCK_SAMPLES;
         audio_time += ((double)AC3_BLOCK_SAMPLES / audio_rate);
	 
         if (audio_sync != 1 && buffer_info[pts_block].pts > 0) {
      	    audio_sync = 1;
	    audio_time = buffer_info[pts_block].pts;
         }
      }
   } else {
      audio_sync = 0;
      if (dvd_eof)
         queue_flush(&audio_queue, audio_queue.max);
   }
}

void fill(int amount) {
   int err, limit = buffer_max;

   while (!dvd_eof) {

      if (amount == FILL_MIN) {
	 if ((video_queue.n >= video_queue.min) &&
	     (audio_queue.n >= audio_queue.min))
	 return;
      }
   
      if (!buffer_full) {
	 /* This can't fill up more than all the queue's combined (so it allows that amount + 1 blocks) */
         while (buffer_info[buffer_in].used) {
  	    buffer_in++;
            if (buffer_in >= buffer_max)
	       buffer_in = 0;
         }
	 
         if (vobu_end && (int)dvd_pos == vobu_end) {
	    if (vobu_next != (vobu_end + 1)) {
	       dvd_vobseek((off_t)vobu_next);
	    }
	 }
         err = dvd_readblock(&buffers[buffer_in]);
         if (err)
  	    return;
      }

      switch (parse_block()) {
       case -1: /* Not an mpeg block */
         if (!limit--) {
	    fprintf(stderr, "Vob file contains no more useful data\n");
	    dvd_eof = 1;
	    return;
         }
       case 0: /* Normal block, succesfully added to a queue */
         buffer_full = 0;
	 break;
       default:
	 /* Block couldn't be added to an output queue, so it try again later
	  * and don't read any more blocks in until this one is done */
         buffer_full = 1;
         return;
      }
      
      if (amount == FILL_INCR)
	 return;
   }
}

void flush_buffers() {
   video_queue.in = video_queue.out = video_queue.used = video_queue.n = 0;
   audio_queue.in = audio_queue.out = audio_queue.used = audio_queue.n = 0;

   buffer_full = buffer_in = buffer_n = 0;
   bzero(buffer_info, buffer_max * sizeof(block_info));
}

void play_seek(off_t where) {
   int len;

   vobu_end = vobu_next = 0;
   dvd_eof = 0;
   dvd_vobseek(where);
   dvd_set_cell(where);
   
   /* Skip to a sequence start packet
    * which seems to always start with a nav packet */
   while (!dvd_eof) {
      flush_buffers();
      fill(FILL_INCR);
      if (video_queue.n && buffer_info[0].sequence_start)
	 break;
   }

   ac3_reset();
   ac3_started = 0;
   ac3_frame = 0;
   audio_sync = 0;
   last_frame = 0;
   
   if (film_mode)
      frame_length = 1.001 / 24;
   
   video_ready = 0;
   mpeg_start = 0;

   fill(FILL_MAX);
}

/* sync:  1 video is too fast (wait some time)
 *       -1 video is too slow (drop a frame)
 *        0 video is in sync now (play a frame) */
int sync() {

   if (!audio_sync) {
      struct timeval tv;
      
      /* No audio currently so just wait until the last frame's time is up
       * Very inaccurate but if there is no sound you probably won't care */
      gettimeofday(&tv, 0);
      now = (double)tv.tv_sec + (double)tv.tv_usec / 1000000.0;
      return (now < (last_frame + frame_length));
   }
   
   /* Synchronize video to audio */
   /* audio_time = time stamp at sample 0 from ac3_frame the last decoded frame */
   /* samples_delay = time from current sample till sound is emptied (ie up to audio_time) */
   /* frame_time = time stamp to show next frame */
   samples_delay = (double)sound_delay() / audio_rate;
   now = audio_time - samples_delay;
   if (!ac3_frame) {
      /* If the last frame is in the sound buffer then samples_delay is measured to the end of that frame
       * so add the frame length to audio_time */
      now += (AC3_BLOCK_SAMPLES / audio_rate);
   }
   /* Allow for time to render the frame */
   now += frame_length;

   if (now < frame_time)
      return 1;
   if (now > (frame_time + frame_length))
      return -1;
   return 0;
}

int play() {
   int ret, n;
   char txt[256];

   fill(FILL_MIN);

#ifdef SHOW_INFO
   sprintf(txt, "Playing: scr=%f pos=%d:%d f=%d/%d s=%d v=%d a=%d %d      ",
	   scr, (int)dvd_pos, dvd_eof, frames_out, frame_count, samples_count,
	   video_queue.n, audio_queue.n, buffer_n);
   fb_puts(0,0,txt);
#endif
   
   /* Decode 250ms of sound if possible */
   for(n=0; n<3; n++) {
      if (!ac3_frame)
         decode_audio();

      /* Play the frame straight away, if possible */
      if (ac3_frame) {
         /* If the sound is stopped then wait until (video_time>=sound_scr)
          * otherwise calculate the sound delay offset for video_time */
         ret = sound_write((unsigned char *)ac3_frame->audio_data);
         if (ret)
            ac3_frame = 0;
      }
   }
   
   /* Decode a video frame */
   if (!video_ready) {
      video_ready = decode_frame();
      if (sync() == -1) {
         mpeg2_drop(&mpeg, 1);
      } else if (!drop_frames) {
	 mpeg2_drop(&mpeg, 0);
      }
   }

   /* Wait until time is due */

   if (sync() == 1) {
      fill(FILL_INCR);
      while (sync() == 1 && !buffer_full)
	 fill(FILL_INCR);
      if (sync() == 1) {
	 sleep(0);
         return (dvd_eof && !buffer_n);
      }
   }

   /* Draw a frame if one is available (this should flip a double buffer at vsync) */
   if (video_ready) {
      video_render();
      video_ready = 0;
      if (!audio_sync) {
         struct timeval tv;
      
         gettimeofday(&tv, 0);
         last_frame = (double)tv.tv_sec + (double)tv.tv_usec / 1000000.0;
      }
   }
   
   /* Return 1 if finished */
   return (dvd_eof && !buffer_n);
}

void play_one() {
   int i, n;
   
   i = frames_out;
   for(n = 0; i == frames_out && n < 16; n++) {
      mpeg2_drop(&mpeg, 0);
      play();
   }
   mpeg2_drop(&mpeg, drop_frames);
}

int play_loaddvd() {

   if (dvd_open(device))
      return 1;

   if (dvd_read())
      return 1;

   play_seek((off_t)dvd_start_block);
   
   return 0;
}

