/*
 * This file is part of the fbdvd program
 * Copyright (C) 2001 Mark Sanderson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include "video.h"
#include "fbtools.h"
#include "mmx.h"
#include "mm_accel.h"
#include "../play.h"
#include "../main.h"

//#define OUTPUT_YUV y

void (* video_render)(void);
video_instance_t vo;
video_frame_t *video_buffer;
int frames_in, frames_out;
int split_yuv = 1;

int aspect, swidth, pan;
static mmx_t scale_a[64];
static mmx_t scale_b[64];
int scale_src[64];

void pan_scan(int direction) {
   int max;

   max = vo.width - (fb_var.xres * vo.width / swidth);
   if (direction == 0) /* Center */
      pan = max / 2;
   else
      pan += direction;
   if (pan < 0) pan = 0;
   if (pan > max) pan = max;
}

/* Horizontal scaling to aspect 16:9 */
void video_render_aspect() {
   int a, b, x, xs, y, fl, ml, left, top, width, height;
   unsigned char *mbuf, *fbuf, *src, *dest;

   /* 24/32bit only */
   fl = fb_fix.line_length;
   b = fb_var.bits_per_pixel / 8;
   ml = vo.width * b;
   mbuf = video_buffer->rgbdata;
   width = swidth;
   if (width > fb_var.xres) {
      width = fb_var.xres;
      mbuf += pan * b;
      left = 0;
   } else {
      left = (int)((fb_var.xres - width) / 2) * b;
   }
   height = vo.height;
   if (height > fb_var.yres) {
      height = fb_var.yres;
      mbuf +=  (int)((vo.height - fb_var.yres) / 2) * ml;
      top = 0;
   } else {
      top = (int)((fb_var.yres - vo.height) / 2) * fl;
   }
   fbuf = fb_mem + fb_mem_offset + left + top;
   for(y = 0; y < height; y++) {
      src = mbuf;
      dest = fbuf;
      xs = 0;
      for (x = 0; x < width; x++) {
	 movq_m2r (*src, mm0);      // mm0 = 00 BB GG RR 00 bb gg rr
	 movq_r2r (mm0, mm1);       // mm1 = 00 BB GG RR 00 bb gg rr
	 pxor_r2r (mm2, mm2);       // mm2 =0
	 punpcklbw_r2r (mm2, mm0);  // mm0 = 00 00 00 bb 00 gg 00 rr
	 pmullw_m2r (scale_a[xs], mm0);
	 punpckhbw_r2r (mm2, mm1);  // mm1 = 00 00 00 BB 00 GG 00 RR
	 pmullw_m2r (scale_b[xs], mm1);
	 paddw_r2r (mm1, mm0);
	 psrlw_i2r (7, mm0);        // mm0 = (rr*scale_a + RR*scale_b) / 256
	 packuswb_r2r (mm2, mm0);
	 movq_r2m (mm0, *dest);
	 src += scale_src[xs];
	 dest += b;
         xs++;
	 if (xs == aspect)
	    xs = 0;
      }
      mbuf += ml;
      fbuf += fl;
   }
   emms();
   frames_out++;
}

/* Integer scaled in both dimensions to width 1024 at aspect 16:9 */
void video_render_scaled() {
   int x, xs, y, ys, w, b, l, left, top, scale;
   unsigned char *fbuf;
   
   b = fb_var.bits_per_pixel / 8;
   w = 1024 * b;
   l = fb_fix.line_length;
   left = (fb_var.xres - 1024) * b / 2;
   top = (int)((fb_var.yres - (1.2 * vo.height))) * l / 2;
   fbuf = fb_mem + fb_mem_offset + left + top;
   ys = 2;
   if (vo.width == 720)
      scale = 214;
   else
      scale = 220;
   if (b == 4) {
      unsigned long *mbuf, line[1024];
      
      mbuf = (unsigned long *)video_buffer->rgbdata;
      for(y = 0; y < vo.height; y++) {
         for(xs = 304,x = 0; x < 1024; x++) {
	    line[x] = *mbuf;
	    if ((xs -= scale) < 0)
	       xs += vo.width;
	    else
	       mbuf++;
         }
         memcpy(fbuf, line, w);
         if (!ys--) {
	    ys = 4;
            fbuf += l;
	    memcpy(fbuf, line, w);
         }
         fbuf += l;
      }
   } else if (b == 3) {
      unsigned char *mbuf, line[3072];
      
      mbuf = video_buffer->rgbdata;
      for(y = 0; y < vo.height; y++) {
         for(xs = 304,x = 0; x < 3072;) {
	    line[x++] = mbuf[0];
	    line[x++] = mbuf[1];
	    line[x++] = mbuf[2];
	    if ((xs -= scale) < 0)
	       xs += vo.width;
	    else
	       mbuf += 3;
         }
         memcpy(fbuf, line, w);
         if (!ys--) {
	    ys = 4;
            fbuf += l;
	    memcpy(fbuf, line, w);
         }
         fbuf += l;
      }
   } else if (b == 2) {
      unsigned short *mbuf, line[1024];
      
      mbuf = (unsigned short *)video_buffer->rgbdata;
      for(y = 0; y < vo.height; y++) {
         for(xs = 304,x = 0; x < 1024; x++) {
	    line[x] = *mbuf;
	    if ((xs -= scale) < 0)
	       xs += vo.width;
	    else
	       mbuf++;
         }
         memcpy(fbuf, line, w);
         if (!ys--) {
	    ys = 4;
            fbuf += l;
	    memcpy(fbuf, line, w);
         }
         fbuf += l;
      }
   }
   frames_out++;
}

/* Unscaled, uncorrected for aspect */
void video_render_direct() {
   int x, y, b, fl, ml, left, top, width, height;
   unsigned char *mbuf, *fbuf;

   b = fb_var.bits_per_pixel / 8;
   fl = fb_fix.line_length;
   ml = vo.width * b;
   mbuf = video_buffer->rgbdata;
   if (vo.width > fb_var.xres) {
      mbuf += pan * b;
      left = 0;
      width = fl;
   } else {
      left = (int)((fb_var.xres - vo.width) / 2) * b;
      width = vo.width * b;
   }
   height = vo.height;
   if (height > fb_var.yres) {
      height = fb_var.yres;
      mbuf +=  (int)((vo.height - fb_var.yres) / 2) * ml;
      top = 0;
   } else {
      top = (int)((fb_var.yres - vo.height) / 2) * fl;
   }
   fbuf = fb_mem + fb_mem_offset + left + top;
   for(y = 0; y < height; y++) {
      memcpy(fbuf, mbuf, width);
      mbuf += ml;
      fbuf += fl;
   }
   frames_out++;
}

int alloc_frames (vo_instance_t * vo_instance,
   int w, int h, int frame_size,
   void (* copy) (vo_frame_t *, uint8_t **),
   void (* field) (vo_frame_t *, int),
   void (* draw) (vo_frame_t *)) {
   
   video_instance_t *instance = (video_instance_t *) vo_instance;
   uint8_t *alloc;
   int i, size;

   size = w * h / 4;
   alloc = (uint8_t *)memalign (16, 18 * size);
   if (alloc == NULL)
      return 1;
   
   for (i = 0; i < 3; i++) {
      instance->frame_ptr[i] = (vo_frame_t *)&instance->frame[i];
      instance->frame_ptr[i]->base[0] = alloc;
      instance->frame_ptr[i]->base[1] = alloc + 4 * size;
      instance->frame_ptr[i]->base[2] = alloc + 5 * size;
      if (split_yuv) {
         instance->frame_ptr[i]->copy = copy;
         instance->frame_ptr[i]->field = field;
      } else {
         instance->frame_ptr[i]->copy = NULL;
         instance->frame_ptr[i]->field = NULL;
      }
      instance->frame_ptr[i]->draw = draw;
      instance->frame[i].rgbdata = instance->rgbdata + i * instance->bytes;
      instance->frame_ptr[i]->instance = (vo_instance_t *) instance;
      alloc += 6 * size;
   }
   return 0;
}

void video_out_close(vo_instance_t * vo_instance) {
   video_instance_t *instance = (video_instance_t *) vo_instance;

   free (instance->rgbdata);
   free (instance->frame_ptr[0]->base[0]);
}

void video_copy(vo_frame_t *vo_frame, uint8_t **src) {
   video_frame_t *frame = (video_frame_t *) vo_frame;
   video_instance_t *instance = (video_instance_t *) frame->vo.instance;

   yuv2rgb (frame->rgb_ptr, src[0], src[1], src[2], vo.width, 16,
      frame->rgb_stride, frame->yuv_stride, frame->yuv_stride >> 1);
   frame->rgb_ptr += frame->rgb_stride << 4;
}

void video_field(vo_frame_t *vo_frame, int flags) {
   video_frame_t *frame = (video_frame_t *) vo_frame;
   video_instance_t *instance = (video_instance_t *) frame->vo.instance;

   frame->rgb_ptr = frame->rgbdata;
   if ((flags & VO_TOP_FIELD) == 0)
      frame->rgb_ptr += instance->rgbstride;
}

void video_draw(vo_frame_t *vo_frame) {
   video_frame_t *frame = (video_frame_t *) vo_frame;

   video_buffer = frame;
#ifndef OUTPUT_YUV
   if (!split_yuv)
      yuv2rgb (frame->rgb_ptr, frame->vo.base[0], frame->vo.base[1], frame->vo.base[2], vo.width, vo.height,
         frame->rgb_stride, frame->yuv_stride, frame->yuv_stride >> 1);
#endif
   frames_in++;
}

int video_setup(vo_instance_t *vo_instance, int w, int h) {
   video_instance_t *instance = (video_instance_t *) vo_instance;
   double c, s;
   int a, b, x, o;
   unsigned long long ma, mb;
   
   instance->width = w;
   instance->height = h;
   instance->prediction_index = 1;
   instance->rgbstride = w * fb_var.bits_per_pixel / 8;
   instance->bytes = instance->rgbstride * h;
   instance->rgbdata = (unsigned char *)malloc(instance->bytes * 3);
   if (!instance->rgbdata)
      return 1;

   if (scale_mode == 0) {              /* Only 24/32 bit fb mode */
      if (fb_var.bits_per_pixel >= 24) 
         video_render = video_render_aspect;
   } else if (scale_mode == 1) {       /* This mode should be the most portable */
      video_render = video_render_direct;
   } else if (scale_mode == 2) {       /* Only 16:9 aspect */
      if (video_aspect == 3 && fb_var.xres >= 1024 && fb_var.yres >= (1.2 * h))
         video_render = video_render_scaled;
   }
   
   /* Calculate the fraction of each pixel (n or n+1) to horizontally scale
    * the given picture dimensions to aspect size */
   if (video_aspect == 2)
      swidth = h * 4 / 3;
   else
      swidth = h * 16 / 9;
   
   if (video_render == video_render_direct)
      swidth = w;
   
   if (swidth == 640) {
      if (w == 720) { //720=>640
         aspect = 8;
         s = 9.0 / 8;
      } else {        //704=>640
         aspect = 10;
         s = 11.0 / 10;
      }
   } else if (swidth == 768) {
      if (w == 720) { //720=>768
         aspect = 16;
         s = 15.0 / 16;
      } else {        //704=>768
         aspect = 12;
         s = 11.0 / 12;
      }
   } else if (swidth == 1024) {
      aspect = 64;
      if (w == 720)   //720=>1024
         s = 45.0 / 64;
      else            //704=>1024
         s = 44.0 / 64;
   } else {
      if (w == 720) { //720=>853
         aspect = 32;
         s = 27.0 / 32;
      } else {        //704=>853
         aspect = 40;
         s = 33.0 / 40;
      }
   }
   o = 1;
   for(x = 0; x < aspect; x++) {
      c = x * s;
      b = (int)(c * 128) & 127;
      a = 128 - b;
      scale_src[x] = 0;
      while (((x + 1) * s) >= o) {
	 o++;
         scale_src[x] += fb_var.bits_per_pixel / 8;
      }
      ma = a; mb = b;
      scale_a[x] = (mmx_t)(ma | (ma<<16) | (ma<<32));
      scale_b[x] = (mmx_t)(mb | (mb<<16) | (mb<<32));
   }
   pan = (w - (fb_var.xres * w / swidth)) / 2;

   return alloc_frames ((vo_instance_t *) instance,
      w, h, sizeof (video_frame_t),
      video_copy, video_field, video_draw);
}

vo_frame_t *video_getframe(vo_instance_t *vo_instance, int flags) {
   video_instance_t *instance = (video_instance_t *) vo_instance;
   video_frame_t *frame;

   if (flags & VO_PREDICTION_FLAG) {
       instance->prediction_index ^= 1;
       frame = (video_frame_t *)instance->frame_ptr[instance->prediction_index];
   } else {
       frame = (video_frame_t *)instance->frame_ptr[2];
   }

   frame->rgb_ptr = frame->rgbdata;
   frame->rgb_stride = instance->rgbstride;
   frame->yuv_stride = video_width;
   if ((flags & VO_TOP_FIELD) == 0)
      frame->rgb_ptr += frame->rgb_stride;
   if ((flags & VO_BOTH_FIELDS) != VO_BOTH_FIELDS) {
      frame->rgb_stride <<= 1;
      frame->yuv_stride <<= 1;
   }

   return (vo_frame_t *)frame;
}

void video_init() {
   
   bzero(&vo, sizeof(vo));
   vo.vo.setup = video_setup;
   vo.vo.close = video_out_close;
   vo.vo.get_frame = video_getframe;
   
   vo_mm_accel = mm_accel() | MM_ACCEL_MLIB;
   yuv2rgb_init(fb_var.bits_per_pixel, MODE_RGB);
   
   frames_in = frames_out = 0;
   
   video_render = video_render_direct;
}


