/*
 * This file is part of the fbdvd program
 * Copyright (C) 2001 Mark Sanderson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <stdio.h>
#include <sys/asoundlib.h>
#include "sound.h"   
#include "fbtools.h"

//#define VISUAL_SOUND y

// Sound card:
// Channel / name / plug
// 0 / Front left / Red left
// 1 / Front right / Red right
// 2 / Rear left / White left
// 3 / Rear right / White right
// 4 / Center / Black left
// 5 / Subwoofer / Black right

int audio_rate;
int sound_block_size, samples_out;
int samples_in, samples_base, sample_size;
int snd_open;
snd_pcm_t *snd;
struct snd_pcm_channel_info snd_info;

int sound_delay() {
   snd_pcm_channel_status_t snd_status;
   
   bzero(&snd_status, sizeof(snd_status));
   snd_status.channel = SND_PCM_CHANNEL_PLAYBACK;
   snd_status.mode = SND_PCM_MODE_BLOCK;
   snd_pcm_channel_status(snd, &snd_status);
   
   samples_out = (snd_status.scount / sample_size) + samples_base;

   return samples_in - samples_out;
}

int sound_flush() {
   snd_pcm_channel_status_t snd_status;
   int flushed;
   
   bzero(&snd_status, sizeof(snd_status));
   snd_status.channel = SND_PCM_CHANNEL_PLAYBACK;
   snd_status.mode = SND_PCM_MODE_BLOCK;
   snd_pcm_channel_status(snd, &snd_status);
   
   samples_out = (snd_status.scount / sample_size) + samples_base;
   flushed = samples_in - samples_out;
   samples_out = samples_base = samples_in;
   
   snd_pcm_channel_flush(snd, SND_PCM_CHANNEL_PLAYBACK);
   snd_pcm_channel_prepare(snd, SND_PCM_CHANNEL_PLAYBACK);
      
   return flushed;
}

void sound_reset() {
   if (!snd_open)
      return;

   sound_flush();
   samples_in = samples_out = samples_base = 0;
}

#ifdef VISUAL_SOUND
void visual_sound(unsigned char *data) {
   int x, y, top, l, b;
   long *pos;
   short *sd = (short *)data;
   
   if(0){FILE *f=fopen("snd.dat","ab");fwrite(data,4,1536,f);fclose(f);}
   
   top = 704;
   l = fb_fix.line_length;
   b = fb_var.bits_per_pixel / 8;
   bzero(fb_mem + top * l, 64 * l);
   for(x = 128; x < 896; x++) {
      y = (*sd>>10) + 32;
      pos = (long *)(fb_mem + x * b + (y + top) * l);
      *pos = 0xffffff;
      sd += 4;
   }
}
#endif

int sound_write(unsigned char *data) {
   snd_pcm_channel_status_t snd_status;
   int n;
   
   if (!snd_open)
      return 1;

   // Alsa recommend using snd_pcm_plugin_* calls so that 8bit sounds work
   // Only snd_pcm_channel_* calls report free buffer space
   bzero(&snd_status, sizeof(snd_status));
   snd_status.channel = SND_PCM_CHANNEL_PLAYBACK;
   snd_status.mode = SND_PCM_MODE_BLOCK;
   snd_pcm_channel_status(snd, &snd_status);
   
   samples_out = (snd_status.scount / sample_size) + samples_base;
   
   n = snd_status.free / sound_block_size;
   if ((n > 0 && snd_status.status == SND_PCM_STATUS_RUNNING) ||
      snd_status.status == SND_PCM_STATUS_PREPARED) {
     
#ifdef VISUAL_SOUND
      visual_sound(data);
#endif
      samples_in += (sound_block_size / sample_size);
      snd_pcm_write(snd, data, sound_block_size);
      return 1;
   
   } else if (snd_status.status != SND_PCM_STATUS_RUNNING) {
      sound_flush();
   }

   return 0;
}

void sound_init(int card, int device, int *channels, int samples_per_block, int rate) {
   struct snd_pcm_channel_params snd_params;
   int err;
   
   snd_open = 0;
   samples_in = samples_out = samples_base = 0;
   audio_rate = rate; // 48khz or 96khz for dvd
   
   if ((err = snd_pcm_open(&snd, card, device, SND_PCM_OPEN_PLAYBACK)) < 0) {
      fprintf(stderr, "Playback open error: %s\n", snd_strerror(err));
      return;
   }
   
   bzero(&snd_info, sizeof(snd_info));
   snd_info.channel = SND_PCM_CHANNEL_PLAYBACK;
   if ((err = snd_pcm_channel_info(snd, &snd_info)) < 0) {
      fprintf(stderr, "Playback info error: %s\n", snd_strerror(err));
      return;
   }
   
   if (*channels > snd_info.max_voices) {
      *channels = snd_info.max_voices;
      fprintf(stderr, "Using %d audio channels\n", *channels);
   }
   sample_size = *channels * 2;
   sound_block_size = samples_per_block * sample_size;
   
   bzero(&snd_params, sizeof(snd_params));
   snd_params.channel = SND_PCM_CHANNEL_PLAYBACK;
   snd_params.mode = SND_PCM_MODE_BLOCK;
   snd_params.format.interleave = 1;
   snd_params.format.format = SND_PCM_SFMT_S16_LE;
   snd_params.format.rate = audio_rate;
   snd_params.format.voices = *channels;
   snd_params.start_mode = SND_PCM_START_DATA;
   snd_params.stop_mode = SND_PCM_STOP_STOP;
   snd_params.buf.block.frag_size = sound_block_size;
   snd_params.buf.block.frags_min = -1;
   snd_params.buf.block.frags_max = -1;
   if ((err = snd_pcm_channel_params(snd, &snd_params)) < 0) {
      fprintf(stderr, "Playback params error: %s\n", snd_strerror(err));
      return;
   }
   snd_pcm_channel_prepare(snd, SND_PCM_CHANNEL_PLAYBACK);
   
   snd_open = 1;
}

void sound_close() {
   if (!snd_open)
      return;
   
   snd_pcm_playback_drain(snd);
   snd_pcm_close(snd);
   snd_open = 0;
}
