/*
 * This file is part of the fbdvd program
 * Copyright (C) 2001 Mark Sanderson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <stdio.h>
#include <termios.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <getopt.h>
#include "play.h"
#include "input/dvd.h"
#include "output/fbtools.h"
#include "output/sound.h"
#include "output/video.h"

char *device = "/dev/dvd";
int scale_mode = 0;
#if 0
int dvd_title = 2;
int dvd_chapter = 1;
int dvd_angle = 1;
int start_block = 0;
int looping = 0;
int audio_track = 1;
int sound_card = 0;
int sound_device = 0;
int sound_channels = 2;
#elif 1
int dvd_title = 1;
int dvd_chapter = 1;
int dvd_angle = 1;
int start_block = 0;
int looping = 0;
int audio_track = 0;
int sound_card = 0;
int sound_device = 0;
int sound_channels = 2;
#else
int dvd_title = 1;
int dvd_chapter = 1;
int dvd_angle = 1;
int start_block = 240000;
int looping = 0;
int audio_track = 0;
int sound_card = 1;
int sound_device = 0;
int sound_channels = 6;
#endif

/*
 * Each buffer is 250ms
 * mpeg2 stream buffer of 1136kb = 1.7seconds (from my test sample)
 */
int input_buffers = 7;
extern int buffer_n;

static int vob_state;
int film_mode;
int drop_frames = 0;
double scr = 0.0;

void console(int acquire) {
   if (acquire) {
      video_render();
      vob_state = 5;
   } else {
      vob_state = 5;
   }
}

void display_init() {
   char *fbdev = 0, *fontfile = 0, *mode = 0;
   int vt = 0;
   
   fb_init(fbdev, fontfile, mode, vt);
   fb_cleanup_fork();
   fb_switch_init(console);
   
   video_init();
}

void display_close() {
   fb_cleanup();
}

void audio_init() {
   ac3_config_t ac3_config;
   int sc = sound_channels;
   
   sound_init(sound_card, sound_device, &sound_channels, AC3_BLOCK_SAMPLES, 48000);

   ac3_config.fill_buffer_callback = ac3_fill_buffer;
   ac3_config.num_output_ch = sound_channels;
   ac3_config.flags = AC3_MMX_ENABLE;
   ac3_init(&ac3_config);
   if (sc != 2)
      fprintf(stderr,"Audio output on %d channels\n", sound_channels);
}

void audio_close() {
   sound_close();
}

void restart(off_t where) {
   play_seek(where);
   vob_state = 2;
}

void user_input() {
   int key, key2, n;
   
   key = fb_getkey();
   if (key == 27) {
      key2 = fb_getkey();
      if (key2 == 91) {
	 key = 256;
         for(n=0; n<3; n++) {
	    key2 = fb_getkey();
	    if (key2 < 32 || key2 > 126) break;
	    key = key * 124 + key2;
	 }
      }
   }
   
#ifdef SHOW_INFO
   if (key>=0) {
      char txt[128];
      
      sprintf(txt, "Key=%d     ", key);
      fb_puts(0,3,txt);
   }
#endif
   
   switch (key) {
    case 'Q':
    case 'q':
    case 27:
      vob_state = -1;
      break;
    case '0':
      audio_track = -1; restart(dvd_pos); break;
    case '1':
      audio_track = 0; restart(dvd_pos); break;
    case '2':
      audio_track = 1; restart(dvd_pos); break;
    case '3':
      audio_track = 2; restart(dvd_pos); break;
    case '4':
      audio_track = 3; restart(dvd_pos); break;
    case '5':
      audio_track = 4; restart(dvd_pos); break;
    case '6':
      audio_track = 5; restart(dvd_pos); break;
    case '7':
      audio_track = 6; restart(dvd_pos); break;
    case '8':
      audio_track = 7; restart(dvd_pos); break;
    case 10:
    case 31809:
    case 31810:
    case 31811:
    case 31812:
    case 'M':
    case 'm':
    case 3947605:
    case 'T':
    case 't':
    case 3947606:
      break;
    case ' ':
    case 'P':
    case 'p':
//      n = dvd_pos - input_buffers * 88;
      n = dvd_pos - buffer_n;
      if (n < 0) n = 0;
      restart(n);
      break;
    case 'S':
    case 's':
      restart(0);
      vob_state = 4;
      break;
    case 'F':
    case 'f':
      vob_state = 4;
      break;
    case '=':
    case '+':
      restart(dvd_pos + 1024);
      break;
    case '-':
    case '_':
      n = dvd_pos - 1024;
      if (n < 0) n = 0;
      restart(n);
      break;
    case '<':
    case ',':
      pan_scan(-1);
      break;
    case '>':
    case '.':
      pan_scan(1);
      break;
    case '?':
    case '/':
      pan_scan(0);
      break;
   }
}

void parse_args(int argc, char *argv[]) {
   int c;
   static struct option long_opts[] = {
       {"dvd",         1, 0, 'v'},
       {"file",        1, 0, 'f'},
       {"title",       1, 0, 't'},
       {"chapter",     1, 0, 'p'},
       {"angle",       1, 0, 'g'},
       {"audio",       1, 0, 'a'},
       {"card",        1, 0, 'c'},
       {"device",      1, 0, 'd'},
       {"channels",    1, 0, 'n'},
       {"seek",        1, 0, 's'},
       {"scale",       1, 0, 'm'},
       {"drop",        0, 0, 'r'},
       {"help",        0, 0, 'h'},
       {0,             0, 0, 0}
   };
  
#ifdef SHOW_INFO
   fprintf(stderr, "DVD Player "VERSION" Copyright (C) 2001 Mark Sanderson\n");
#endif 
   for (;;) {
      if ((c = getopt_long(argc, argv, "v:f:t:p:g:a:c:d:n:s:m:rh", long_opts, NULL)) == -1)
	 break;
      switch (c) {
       case 0:
	 /* long option */
	 break;
       case 'v':
       case 'f':
	 device = optarg;
         break;
       case 't':
	 dvd_title = atoi(optarg);
         break;
       case 'p':
	 dvd_chapter = atoi(optarg);
         break;
       case 'g':
	 dvd_angle = atoi(optarg);
         break;
       case 'a':
	 audio_track = atoi(optarg) - 1;
         break;
       case 'c':
	 sound_card = atoi(optarg);
         break;
       case 'd':
	 sound_device = atoi(optarg);
         break;
       case 'n':
	 sound_channels = atoi(optarg);
         break;
       case 's':
	 start_block = atoi(optarg);
         break;
       case 'm':
	 scale_mode = atoi(optarg);
         break;
       case 'r':
	 drop_frames = 1;
         break;
       case 'h':
       default:
         fprintf(stderr,
            "Options:\n"
	    "  -h | --help           this message\n"
	    "  -d | --dvd <device>   specify dvd drive, or vob file (/dev/dvd)\n"
	    "  -f | --file <file>    specify dvd drive, or vob file (/dev/dvd)\n"
	    "  -t | --title <n>      dvd program title to play (1)\n"
	    "  -p | --chapter <n>    chapter from dvd title (1)\n"
	    "  -g | --angle <n>      view angle for dvd title (1)\n"
	    "  -a | --audio <n>      audio track number (1)\n"
	    "  -c | --card <n>       alsa sound card (0)\n"
	    "  -d | --device <n>     alsa sound device (0)\n"
	    "  -n | --channels <n>   sound output channels, use 6 for AC3 5.1 (2)\n"
	    "  -s | --seek <n>       start block within title (0)\n"
	    "  -m | --scale <0|1|2>  0 = normal aspect scaling\n"
	    "                        1 = no scaling\n"
	    "                        2 = 1024 pixel integer scaling (0)\n"
	    "  -r | --drop           drop lots of frames\n");
	 exit(1);
      }
   }
}

int main(int argc, char *argv[]) {
   int err;

   parse_args(argc, argv);

   display_init();
   audio_init();
   play_init();
   
   // Event loop
   vob_state = 1;
   
   while (vob_state >= 0) {
      
      user_input();
      
      switch (vob_state) {
	 
       case 0: /* idle / paused */
	 sleep(0);
	 break;
	 
       case 1: /* Opening */
	 err = play_loaddvd();
         if (err) {
	    vob_state = -1;
	 } else {
	    vob_state = 2;
            if (start_block)
	       play_seek(start_block);
	 }
	 break;
	 
       case 2: /* Playing */
         err = play();
	 if (err) 
	    vob_state = 3;
	 break;
	 
       case 3: /* Finished */
	 dvd_close();
	 if (looping)
	    vob_state = 1;
	 else
	    vob_state = -1;
	 break;

       case 4: /* Freeze frame */
	 play_one();
	 vob_state = 0;
	 break;
	 
       case 5: /* Suspended by console */
	 fb_switch_ack();
	 vob_state = 0;
	 break;
      }
   }
   
   play_close();
   dvd_close();
   display_close();
   audio_close();
}


