/*
 * This file is part of the fbdvd program
 * Copyright (C) 2001 Mark Sanderson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>		// exit ()
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <dlfcn.h>
#include <inttypes.h>
#include <sys/errno.h>
#include <linux/cdrom.h>

#include "dvd_udf.h"
#include "dvd.h"
#include "../main.h"

#ifdef LIB_CSS
extern int CSSisEncrypted (int fd);
extern int CSSAuthDisc (int fd, char *key_disc);
extern int CSSAuthTitle (int fd, char *key_title, int lba);
extern int CSSGetASF (int fd);
extern int CSSDecryptTitleKey (char *key_title, char *key_disc);
extern void CSSDescramble (u_char *sec, u_char *key);
#endif

// Public
int dvd_eof;
off_t dvd_pos;

// Private
static struct {
   int fd, file;
   uint8_t key_disc[DVD_VIDEO_LB_LEN];
   uint8_t key_title[5];
   off_t vob_start;      // start of title (file)
   off_t ifo_start;
} dvd;

int dvd_open (char *device) {

   bzero(&dvd, sizeof(dvd));
   dvd.fd = -1;
   dvd.file = 1;
   dvd_eof = 0;
   dvd_pos = 0;
   
//   if ((dvd.fd = open(device, O_RDONLY | O_NONBLOCK)) < 0) {
   if ((dvd.fd = open(device, O_RDONLY)) < 0) {
      fprintf(stderr, "Error opening %s (%s)\n", device, strerror (errno));
      return 1;
   }

   return 0;
}

int dvd_open_ifo(int title) {
   char filename[32], ifoname[32];

   if (title == 0) {
      sprintf(ifoname, "video_ts.ifo");
   } else {
      sprintf(ifoname, "vts_%02d_0.ifo", title);
   }
   
   sprintf(filename, "/video_ts/%s", ifoname);
   dvd.ifo_start = UDFFindFile(dvd.fd, filename);
   if (dvd.ifo_start == 0) {
      dvd.ifo_start = UDFFindFile(dvd.fd, ifoname);
   }
   if (dvd.ifo_start == 0)
      return 1;
   
   return 0;
}

int dvd_open_vob (int title, int chapter) {
   char vobname[32];
   
   /* Open the vob file */
   sprintf(vobname, "/video_ts/vts_%02d_%d.vob", title, chapter);
   dvd.vob_start = UDFFindFile(dvd.fd, vobname);
   if (dvd.vob_start == 0) {
      sprintf(vobname, "vts_%02d_%d.vob", title, chapter);
      dvd.vob_start = UDFFindFile(dvd.fd, vobname);
   }
   if (dvd.vob_start == 0) {
      fprintf(stderr, "Couldn't open vob file %s\n", vobname);
      return 1;
   }
   
   dvd.file = 0;
   
   // authorization for the right area ...
   dvd_vobseek(dvd.vob_start);

#ifdef LIB_CSS
   if (CSSisEncrypted(dvd.fd)) {
      
      if (CSSAuthDisc (dvd.fd, dvd.key_disc) < 0) {
	 fprintf(stderr, "CSSAuthDisc (%s)\n", strerror (errno));
	 return 1;
      }
      
      if (CSSAuthTitle (dvd.fd, dvd.key_title, (int)dvd.vob_start) < 0)
         fprintf(stderr, "Authenticate title (%s)\n", strerror (errno));

      if (CSSDecryptTitleKey (dvd.key_title, dvd.key_disc) < 0)
         fprintf(stderr, "Decrypting title (%s)\n", strerror (errno));
   
      fprintf(stderr, "Reading encrypted dvd file %s\n", vobname);
      return 0;
   }
#endif
   
   fprintf(stderr, "Reading unencrypted dvd file %s\n", vobname);
   return 0;
}

int dvd_close () {
   dvd_authinfo ai;
   int i;

   dvd_eof = 1;
   if (dvd.fd < 0) return;

#ifdef LIB_CSS
   if (!dvd.file) {
      // reset all AGIDs on the DVD drive
      for (i=0; i<4; i++) {
         memset (&ai,  0, sizeof (dvd_authinfo));
         ai.type = DVD_INVALIDATE_AGID;
         ai.lsa.agid = i;
         ioctl (dvd.fd, DVD_AUTH, &ai);
      }
   }
#endif
   
   close (dvd.fd);
   dvd.fd = -1;

   return 0;
}

/* Offset in sectors from the beginning of the dvd */
int dvd_vobseek (off_t pos) {
   off_t s;
   
   if (dvd.file)
      s  = pos * (off_t)DVD_VIDEO_LB_LEN;
   else
      s  = (dvd.vob_start + pos) * (off_t)DVD_VIDEO_LB_LEN;
   if (lseek (dvd.fd, s, SEEK_SET) == -1) {
      fprintf(stderr, "Error in lseek at pos 0x%Lx\n", s);
      return 1;
   }
   
   dvd_pos = pos;
   return 0;
}

/* Offset in bytes from the beginning of the ifo file */
off_t dvd_ifoseek (off_t pos) {
   off_t s;
   
   s = (dvd.ifo_start * (off_t)DVD_VIDEO_LB_LEN) + pos;
   if (lseek (dvd.fd, s, SEEK_SET) == -1) {
      fprintf(stderr, "Error in lseek at pos 0x%Lx\n", s);
      return -1;
   }
   
   return pos;
}

int dvd_readblock(dvd_block *data) {
   int n;
   
   dvd_pos++;
   n = read(dvd.fd, data, DVD_VIDEO_LB_LEN);
   if (n == -1) {
      if (errno == EAGAIN)
	 return 2;
   }
   if (n < DVD_VIDEO_LB_LEN) {
      bzero(data, DVD_VIDEO_LB_LEN);
      dvd_eof = 1;
      return 1;
   }
   
#ifdef LIB_CSS
   if (!dvd.file)
      CSSDescramble((unsigned char *)data, dvd.key_title);
#endif

   return 0;
}

int dvd_readbytes(void *data, int len) {
   return read(dvd.fd, data, len);

}

