/* SIMPLE - Simple Is a Macro Processing Language Element */
/* Copyright (c) 1998 David A. Madore (david.madore@ens.fr) */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* This file contains what is in effect the lexer: it takes care of
 * reading the input files and producing a token list.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "simple.h"
#include "token.h"

typedef struct file_struct {
  char *name; /* Allocated by pushinputfile() */
  FILE *f;
} file_struct;

static file_struct *file_stack = NULL;
static int file_stack_depth = 0;
static int file_stack_allsize = 0;

void
push_input_file(char *name)
     /* Add a new file to the stack of input files. */
{
  char *nname; /* Just a comodity. */
  if (file_stack_depth==file_stack_allsize) {
    if ((file_stack=REALLOC(file_stack,
			    (++file_stack_allsize)*sizeof(file_struct)))
	==NULL)
      fatal(0,"Out of memory!");
  }
  if (strcmp(name,"-")==0) name="stdin";
  if ((nname=malloc(strlen(name)+1))==NULL)
    fatal(0,"Out of memory!");
  strcpy(nname,name);
  file_stack[file_stack_depth].name=nname;
  if (strcmp(nname,"stdin")==0) file_stack[file_stack_depth].f=stdin;
  else {
    if ((file_stack[file_stack_depth].f=fopen(nname,"r"))==NULL)
      fatal(0,"Can't open %s",name);
  }
  file_stack_depth++;
}

int
get_next_char(void)
     /* Return the next input character. */
{
  int c;
  while (file_stack_depth>0) {
    if ((c=getc(file_stack[file_stack_depth-1].f))!=EOF)
      return c;
    else {
      file_stack_depth--;
      fclose(file_stack[file_stack_depth].f);
      FREE(file_stack[file_stack_depth].name);
      /* We don't bother realloc()ating the filestack to a shorter
       * size. */
    }
  }
  return EOF;
}

static struct {
  int type;
  token_list equiv;
} chartab[256];

static int lexer_state = 0;

static int active_char;
static int active_pos;

char
tokenize(int c,token *tk)
     /* Return 1 if a token has been discovered and store it in tk */
{
  switch (lexer_state) {
  case 0: /* Normal mode */
    if (c==EOF) { *tk=EOF; return 1; }
    switch (chartab[c].type) {
    case 0: *tk=c; return 1;
    case 1: lexer_state=1; return 0;
    case 2: lexer_state=2; active_char=c; active_pos=0; return 0;
    case 3: lexer_state=3; return 0;
    case 100: lexer_state=100; return 0;
    }
    break;
  case 1: /* Quoting one character */
  case 101: /* Quoting one character in a string */
    lexer_state--;
    *tk=c; return 1; /* Quoting EOF is not an error. */
    break;
  case 100: /* Quoting a string */
    switch (chartab[c].type) {
    case 0:
    case 2:
    case 3: *tk=c; return 1;
    case 1: lexer_state=101; return 0;
    case 100: lexer_state=0; return 0;
    }
    break;
  case 3: /* Gobbling a comment */
    if (c==EOF)
      fatal(0,"Comment interrupted by end of file.");
    /* Stupid error isn't it? */
    if (c=='\n') {
      lexer_state=0;
      return 0;
    }
    return 0;
    break;
  case 2: /* Inside an active character */
    if (active_pos>=chartab[active_char].equiv.size) {
      lexer_state=0;
      return 0;
    }
    *tk = chartab[active_char].equiv.l[active_pos++];
    return 1;
    break;
  default:
    fatal(1,"Impossible lexer state reached!");
  }
  return 0; /* To keep gcc happy */
}

token
input_token(void)
     /* Return the next input token */
{
  token tk;
  while (!tokenize((lexer_state!=2?get_next_char():0),&tk));
  return tk;
}

void
syntax_cartype_ordinary(unsigned char c)
{
  free_list(&chartab[(int)c].equiv);
  chartab[(int)c].type=0;
  make_empty_list(&chartab[(int)c].equiv);
}

void
syntax_cartype_escape_next(unsigned char c)
{
  free_list(&chartab[(int)c].equiv);
  chartab[(int)c].type=1;
  make_empty_list(&chartab[(int)c].equiv);
}

void
syntax_cartype_escape_string(unsigned char c)
{
  free_list(&chartab[(int)c].equiv);
  chartab[(int)c].type=100;
  make_empty_list(&chartab[(int)c].equiv);
}

void
syntax_cartype_comment(unsigned char c)
{
  free_list(&chartab[(int)c].equiv);
  chartab[(int)c].type=3;
  make_empty_list(&chartab[(int)c].equiv);
}

void
syntax_cartype_active(unsigned char c,token_list *tl)
{
  free_list(&chartab[(int)c].equiv);
  chartab[(int)c].type=2;
  make_empty_list(&chartab[(int)c].equiv);
  copy_list(&chartab[(int)c].equiv,tl);
}

void
init_syntax(void)
     /* Initialize the syntactic analyser */
{
  int c;
  for (c=0;c<256;c++)
    make_empty_list(&chartab[c].equiv);
  chartab['`'].type=1;
  chartab['"'].type=100;
  chartab['%'].type=3;
  chartab['<'].type=2;
  append_token(&chartab['<'].equiv,BEGIN_COMMAND);
  chartab['>'].type=2;
  append_token(&chartab['>'].equiv,END_COMMAND);
  chartab['|'].type=2;
  append_token(&chartab['|'].equiv,NEXT_PARAM);
  chartab['['].type=2;
  append_token(&chartab['['].equiv,OPEN_QUOTE);
  chartab[']'].type=2;
  append_token(&chartab[']'].equiv,CLOSE_QUOTE);
  chartab['#'].type=2;
  append_token(&chartab['#'].equiv,QUOTE_NEXT);
  chartab['@'].type=2;
  append_token(&chartab['@'].equiv,AT_SIGN);
}
