/*  console_beep V0.2  by Josef Pavlik <jet@pavlik.it>
 *
 * history: 
 *   V0.2 sep 03 - destination file may be only stdout or /dev/console
 *   V0.1 oct 97 - initial version
 *
 *  Uses linux console commands instead of direct access to i/o ports
 *  May be more secure.
 *  If you want beep on remote machine (via telnet), you must have write 
 *  permition to remote /dev/console or beep must be setuid root
 *  if you run beep on remote machine and destination is stdout (default)
 *  you hear beep on local machine. 
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#define SQRT12_2  (1.05946)
#define SQRT6_2   (SQRT12_2*SQRT12_2)

#define NOTE_A (440)
#define NOTE_B (NOTE_A*SQRT6_2)
#define NOTE_C (NOTE_B*SQRT12_2)
#define NOTE_D (NOTE_C*SQRT6_2)
#define NOTE_E (NOTE_D*SQRT6_2)
#define NOTE_F (NOTE_E*SQRT12_2)
#define NOTE_G (NOTE_F*SQRT6_2)



const double note[7]=
{
  NOTE_A,
  NOTE_B,
  NOTE_C,
  NOTE_D,
  NOTE_E,
  NOTE_F,
  NOTE_G
};
 

FILE *dest;

void beep(int freq, int length)
{
  fprintf(dest,"\33[10;%d]\33[11;%d]\a\33[10]\33[11]", freq, length);
  fflush(dest);
  if (length) usleep(length*1000L);
}

enum{ST_NORM, ST_NUMBER, ST_COMMENT, ST_NOTE};

int main(int argc, char *argv[])
{
  int freq, length;
  int command=0;
  FILE *in=NULL;
  int ch;
  int parms,state;
  int parm[2];
  char *destfilename;
  double ffreq=0;

//  printf("argc=%d, argv[0]=%s, argv[1]=%s, argv[2]=%s, argv[3]=%s", argc, argv[0], argv[1], argv[2], argv[3]);
  if (argc<3 && !(argc>1 && *argv[1]=='@'))
  {
    fprintf(stderr, "Usage: beep { freq | 0 } { length | 0 } [/dev/console]\n"
                    "    or beep @ [file] [/dev/console]\n");
    exit(2);
  }

  if (*argv[1]=='@')
  {
    if (argc==2 || (argc>2 && *argv[2]=='-' && argv[2][1]=='\0')) 
    {
      in=stdin;
    }
    else
    {
      in=fopen(argv[2],"r");
      if (!in)
      {
        perror("ring: could not open input file");
        exit(3);
      }
    }
  }
  else
  {
    command=1;
  }
  dest=stdout;
  destfilename=NULL;
  if (argc>3)
  {
    destfilename=argv[3];
  }
  else if (argc==3 && argv[1][0]=='@' && argv[1][1]==' ') // batch file
  {
    destfilename=&argv[1][2];
  }
  if (destfilename && !strcmp("/dev/console",destfilename)) // we can open only /dev/console as output device
  { 
    dest=fopen(destfilename, "w");
    if (!dest) 
    {
      perror("Can't open destination file");
      exit(1);
    }
  }

  if (command)
  {
    freq=atoi(argv[1]);
    length=atoi(argv[2]);
    beep(freq, length);
  }
  else
  {
//    while (fscanf(in,"%d %d\n", &freq, &length)==2)
//    {
//      beep(freq, length);
//    }
    state=ST_NORM;
    parms=0;
    ch=getc(in);
    while(ch>=0)
    {
       switch(state)
       {
         case ST_NORM: 
           if (ch=='#') 
           {
              state=ST_COMMENT;
              ch=getc(in);
           }
           else if (isdigit(ch))
           {
              state=ST_NUMBER;
              parm[parms]=0;
           }
           else if (toupper(ch)>='A' && toupper(ch)<='G')
           {
              state=ST_NOTE;
              ffreq=note[toupper(ch)-'A'];
              ch=getc(in);
           }
           else
           {
              ch=getc(in);
           }
           break;

         case ST_NUMBER:
           if (isdigit(ch))
           {
             parm[parms]*=10;
             parm[parms]+=ch-'0';
             ch=getc(in);
           }
           else
           {
             state=ST_NORM;
             parms++;
             if (parms==2)
             {
               beep(parm[0], parm[1]);
               parms=0;
             }
           }
           break;

         case ST_COMMENT:
           if (ch=='\n')
           {
             state=ST_NORM;
           }
           ch=getc(in);
           break;     
     
         case ST_NOTE:
           if (ch=='#')
           {
             ffreq*=SQRT12_2;
           }
           else if (ch=='b')
           {
             ffreq/=SQRT12_2;
           }
           else if (isdigit(ch))
           {
             ffreq*=pow(2, ch-'4');
           }
           else
           {
             parm[parms++]=(int)ffreq;
             state=ST_NORM;
             break;
           }
           ch=getc(in);
           break;
      }     
    }     
  }
  fclose(dest);
  exit(0);
}
