/*
  cddarec.c  --  record raw big-endian-format sound files
  Copyright (C) 1998, 1999 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define	DEBUG
#undef DEBUG

#include <stdio.h>		/* fprintf() */
#include <errno.h>		/* error numbers */
#include <unistd.h>		/* getopt(), read(), write(), close(), STDIN_FILENO */
#include <sys/types.h>		/* open() */
#include <sys/stat.h>		/* open() */
#include <fcntl.h>		/* open(), O_RDONLY */
#include <stdlib.h>		/* malloc() */
#include <sys/ioctl.h>		/* ioctl() */
#include <ctype.h>		/* tolower() */
#include "ss_defaults.h"	/* email address */
#include <linux/soundcard.h>	/* SNDCTL_DSP_* */
#include <linux/cdrom.h>	/* CD_FRAMESIZE_RAW, ... */
#include <string.h>		/* strcmp() */
#include <pthread.h>		/* pthread_*() */

#define VERSION_NUMBER 			"0.20"
#define DATE_OF_LAST_MODIFICATION	"1999/02/12"
#define	name_of_this_program		cddarec
#define	Name_of_this_program		Cddarec
#define	NAME_OF_THIS_PROGRAM		CDDAREC

#define INFO_CHANNEL	stderr
#define HELP_CHANNEL	stdout
#define ERROR_CHANNEL	stderr
#define VERSION_CHANNEL	stdout
#define DEBUG_CHANNEL	stderr

#define DEFAULT_DSP_SPEED		( CD_FRAMESIZE_RAW/((DEFAULT_DSP_STEREO+1)*DEFAULT_DSP_SAMPLESIZE/8)*CD_FRAMES )
#define MIN_DSP_SPEED			100
#define MAX_DSP_SPEED			88200
#define DEFAULT_DSP_SAMPLESIZE		16
#define DEFAULT_DSP_DEVICE_NAME		"/dev/dsp"
#define DEFAULT_DSP_STEREO		1
#define	LITTLE_ENDIAN_SOUNDCARD
#define ACCESS_MODE		(S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH)
#define	DEFAULT_NUMBER_OF_BUFFERS	2
#define	NUMBER_OF_BUFFERS_MIN		1
#define	NUMBER_OF_BUFFERS_MAX		1024
/* here: buffer sizes in frames == CD_FRAMESIZE_RAW bytes */
/* converted to bytes after option processing */
#define	DEFAULT_BUFFER_SIZE		75
#define	BUFFER_SIZE_MIN			1
#define	BUFFER_SIZE_MAX			( 60*CD_SECS*CD_FRAMES )
#define KEY_BUFFER_SIZE			32

/* number of bytes -> number of frames */
#define NB2NF(NB)	( (NB) / CD_FRAMESIZE_RAW )
/* number of frames -> number of full minutes */
#define	NF2NM(NF)	( (NF) / CD_FRAMES / CD_SECS )
/* number of frames -> number of remaining seconds */
#define	NF2NS(NF)	( (NF) / CD_FRAMES - NF2NM(NF)*CD_SECS )
/* number of frames -> number of remaining frames */
#define	NF2NRF(NF)	( (NF) - (NF2NM(NF)*CD_SECS+NF2NS(NF))*CD_FRAMES )


typedef unsigned char	UCHAR;


struct thread_info {
  UCHAR*	addr;		/* buffer address */
  int		size;		/* valid size */
};


int	shared_err= 0;			/* shared master/thread infos */
int	shared_fd= STDOUT_FILENO;	/* out file descriptor */
char*	shared_fn= NULL;		/* out file name */
char*	shared_pn= NULL;		/* main program name */



int
dsphlp( char* fn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s)\n", fn, VERSION_NUMBER, DATE_OF_LAST_MODIFICATION );
  fprintf( HELP_CHANNEL, "Record raw big-endian sound files.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [file]\n", fn );
  fprintf( HELP_CHANNEL, "switches:\n");
  fprintf( HELP_CHANNEL, "  -h\t help, write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level at %s\n", INFO_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation info to %s and exit sucessfully\n", VERSION_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n");
  fprintf( HELP_CHANNEL, "  -d DEVICE\t input device (default is %s)\n", DEFAULT_DSP_DEVICE_NAME );
  fprintf( HELP_CHANNEL, "  -n BUFFERS\t number of buffers (default is %d)\n", DEFAULT_NUMBER_OF_BUFFERS );
  fprintf( HELP_CHANNEL, "  -s SIZE\t buffer size in frames or msf (default is %d)\n", DEFAULT_BUFFER_SIZE );
  fprintf( HELP_CHANNEL, "Start/stop recording by hitting s/q (+ENTER).\n");
  return( 0 );
}


int
dspver( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MODIFICATION );
  fprintf( VERSION_CHANNEL, "Compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  DEFAULT_BUFFER_SIZE       : %d\n",
           DEFAULT_BUFFER_SIZE );
  fprintf( VERSION_CHANNEL, "  DEFAULT_NUMBER_OF_BUFFERS : %d\n", 
           DEFAULT_NUMBER_OF_BUFFERS );
  fprintf( VERSION_CHANNEL, "  LITTLE_ENDIAN_SOUNDCARD   : " );
#ifdef	LITTLE_ENDIAN_SOUNDCARD
  fprintf( VERSION_CHANNEL, "defined\n" );
#else
  fprintf( VERSION_CHANNEL, "undefined\n" );
#endif
  return( 0 );
}


int
getintoptarg( char opt, char* arg, int vmn, int vmx, char* pn ) {
  char *p;
  int val= strtol( arg, &p, 0 );
  if( p==arg || *p!='\0' )
  { /* nothing read */
    fprintf( ERROR_CHANNEL, "%s: Argument `%s' of option -%c not a number.\n",
             pn, arg, opt );
    exit( 1 );
  }
  if( val<vmn || val>vmx )
  { /* out of interval */
    fprintf( ERROR_CHANNEL,
             "%s: Argument value of option -%c (%d) not in [%d,%d].\n",
            pn, opt, val, vmn, vmx);
    exit( 1 );
  }
  return( val );
}


int
getframeoptarg( char opt, char* arg, int vmn, int vmx, char* pn ) {
/* frames or msf (mm:ss:ff, ss maybe > 59, ff maybe > 74) */
/* 1000		= 1000 frames */
/* 12:34	= :12:34	= 00:12:34 */
  int i= 0;
  int colons= 0;
  char str[128]= "";
  char* p= arg;
  char* pold;
  int val[3]= { 0, 0, 0 };
  /* copy argument string while dropping white spaces and counting colons */
  while( i<=strlen(arg) && i<128 ) {
    if( *p == ':' ) colons++;
    if( *p != ' ' && *p != '\t' ) str[i++]= *p++;
    else p++;
  }
  if( colons > 2 ) goto GETFRAMEOPTARG_ERROR;
  p= str;
  for( i=colons; i>=0; i-- ) {
    pold= p;
    switch( *p ) {
      case ':'	: p++;
      case '\0'	: continue;
      default	:
        val[i]= strtol( pold, &p, 0 );
        if( p == pold ) goto GETFRAMEOPTARG_ERROR;
        if( *p == ':' ) p++;
    }
  }
  return( (val[2]*CD_SECS+val[1])*CD_FRAMES + val[0] );
GETFRAMEOPTARG_ERROR:
  fprintf( ERROR_CHANNEL,
           "%s: Argument `%s' of option -%c neither in frames nor msf.\n",
           pn, str, opt );
  exit( 1 );
}


struct thread_info*
create_thread_info( UCHAR* addr, int size ) {
  struct thread_info* tip;
  if( (tip=malloc(sizeof(struct thread_info))) == NULL ) return( NULL );
  tip->addr= addr;
  tip->size= size;
  return( tip );
}


void*
client( void* p ) {
  struct thread_info* tip= p;		/* thread info pointer */
  int nbw;
#ifdef	DEBUG
  char str[128];
  sprintf( str, "%s: DEBUG: Slave: start addr= %08x\n",
           shared_pn, (unsigned int)tip->addr );
  write( STDERR_FILENO, str, strlen(str) );
#endif
#ifdef	LITTLE_ENDIAN_SOUNDCARD
{
int i;
UCHAR byte;
for(i=0; i<tip->size; i+=2) {
  byte			= tip->addr[i];
  tip->addr[i]		= tip->addr[i+1];
  tip->addr[i+1]	= byte;
}
}
#endif	/* LITTLE_ENDIAN_SOUNDCARD */
  if( (nbw=write(shared_fd,tip->addr,tip->size)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot write to ", shared_pn );
    if( shared_fd == STDOUT_FILENO )
      fprintf( ERROR_CHANNEL, "stdout. " );
    else
      fprintf( ERROR_CHANNEL, "file `%s'. ", shared_fn );
    fprintf( ERROR_CHANNEL, "%s.\n", sys_errlist[errno] );
    shared_err= -1;
  }
  sync();
#ifdef	DEBUG
  sprintf( str, "%s: DEBUG: Slave: end   addr= %08x\n",
           shared_pn, (unsigned int)tip->addr );
  write( STDERR_FILENO, str, strlen(str) );
#endif
  free( tip );
  return( NULL );
}


int
main( int argc, char** argv)
/*
SOLYGA --------------------
SOLYGA main(argc, argv) cddarec
SOLYGA record big-endian sound file

SOLYGA Steffen Solyga <solyga@absinth.net>
SOLYGA started      : Sat Jan 24 22:51:42 MET 1998 @beast

big-endian is		msbl lsbl msbr lsbr ...
little-endian is	lsbl msbl lsbr msbr ... (80x86 machines)
*/
{
  int dsp_speed = DEFAULT_DSP_SPEED;
  int dsp_samplesize = DEFAULT_DSP_SAMPLESIZE;
  int dsp_stereo = DEFAULT_DSP_STEREO;
  int dsp_buffer_size;
  int number_of_buffers= DEFAULT_NUMBER_OF_BUFFERS;
  int buffer_size= DEFAULT_BUFFER_SIZE;
  UCHAR* buffer= NULL;
  UCHAR* bp= NULL;
  int buffer_index;
  pthread_t thread= 0;
  struct thread_info* tip;
  char default_dsp_fn[32]= DEFAULT_DSP_DEVICE_NAME;
  char* dsp_fn= default_dsp_fn;
  int dsp_fd;
  int nbr;			/* number of bytes read */
  int tnbr= 0;			/* total number of bytes read */
  int tnbw= 0;			/* total number of bytes written */
  int verbose= 0;
  int c;			/* option char */
  int tmp;			/* ioctl(SNDCTL_DSP_*) */
  char key_buf[KEY_BUFFER_SIZE];
  int key;
  int retval= 0;
  /***** process options *****/
  while( (c=getopt(argc,argv,"d:hn:s:vV")) != EOF )
    switch( c ) {
      case 'd': /* set dsp device file */
        dsp_fn= optarg;
        break;
      case 'h': /* display help and exit sucessfully */
        dsphlp(*argv);
        retval= 0; goto DIE_NOW;
      case 'n': /* set number of buffers */
        number_of_buffers= getintoptarg( c, optarg, NUMBER_OF_BUFFERS_MIN,
                                          NUMBER_OF_BUFFERS_MAX, *argv );
        break;
      case 's': /* set buffer size */
        buffer_size= getframeoptarg( c, optarg, BUFFER_SIZE_MIN, 
                                       BUFFER_SIZE_MAX, *argv );
        break;
      case 'v': /* set verbose mode */
        verbose++;
        break;
      case 'V': /* display version to VERSION_CHANNEL and exit sucessfully */
        dspver( *argv );
        retval= 0; goto DIE_NOW;
      case '?': /* refer to -h and exit unsucessfully */
        fprintf( ERROR_CHANNEL, "Try `%s -h' for more information.\n", *argv );
        retval= 1; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Options bug! E-mail me at %s.\n",
                 *argv, MY_EMAIL_ADDRESS );
        retval= 1; goto DIE_NOW;
    }

/* convert buffer size from frames to bytes */
  buffer_size*= CD_FRAMESIZE_RAW;

/* init of shared vaiables / open output file */
  shared_pn= *argv;
  if( optind<argc && strcmp(argv[optind],"-") ) {
    shared_fn= argv[optind];
    if((shared_fd=open(shared_fn,O_WRONLY|O_CREAT|O_TRUNC,ACCESS_MODE))==-1) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open file `%s' for writing. %s.\n",
              shared_pn, shared_fn, sys_errlist[errno] );
      retval= 1; goto DIE_NOW;
    }
  }
  
/* set stdin non-blocking */
  if( fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Can't set stdin non-blocking.\n", shared_pn );
    retval= 1; goto DIE_NOW;
  }

/* open audio device */
  if( (dsp_fd=open(dsp_fn, O_RDONLY, 0)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot open device `%s' for reading. %s.\n",
            shared_pn, dsp_fn, sys_errlist[errno] );
    retval= 1; goto DIE_NOW;
  }

/* set sample size (resolution) */
  tmp= dsp_samplesize;
  ioctl( dsp_fd, SNDCTL_DSP_SAMPLESIZE, &dsp_samplesize );
  if( tmp != dsp_samplesize ) {
    fprintf( ERROR_CHANNEL, "%s: Can't set sample size to %d bit. ",
             shared_pn, tmp );
    if( dsp_samplesize == -1 ) {
      fprintf( ERROR_CHANNEL, "%s.\n", sys_errlist[errno] );
      retval= 1; goto DIE_NOW;
    }
    else fprintf( ERROR_CHANNEL, "Set to %d bit.\n", dsp_samplesize );
  }

/* set mono/stereo mode */
  if( ioctl(dsp_fd,SNDCTL_DSP_STEREO,&dsp_stereo) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Can't set mono/stereo mode. %s.\n",
            shared_pn, sys_errlist[errno] );
    retval= 1; goto DIE_NOW;
  }

/* set sample speed (rate) */
  tmp= dsp_speed;
  ioctl( dsp_fd, SNDCTL_DSP_SPEED, &dsp_speed );
  if( dsp_speed != tmp ) {
    fprintf( ERROR_CHANNEL, "%s: Can't set sample speed to %d Hz. ",
             shared_pn, tmp );
    if( dsp_speed == -1 ) {
      fprintf( ERROR_CHANNEL, " %s.\n", sys_errlist[errno] );
      retval= 1; goto DIE_NOW;
    }
    else fprintf( ERROR_CHANNEL, "Set to %d Hz.\n", dsp_speed );
  }

/* get kernel's dsp buffer size */
  ioctl( dsp_fd, SNDCTL_DSP_GETBLKSIZE, &dsp_buffer_size );

/* allocate for buffer */
  if( (buffer=malloc(number_of_buffers*buffer_size)) == NULL ) {
    fprintf( ERROR_CHANNEL,
             "%s: Can't allocate %1.1f Mbytes (total) for audio buffer(s).\n",
             shared_pn, number_of_buffers*buffer_size/1024.0/1024.0 );
    retval= 1; goto DIE_NOW;
  }

/* give some info */
  if(verbose)
  { /* print some info */
    int nf= NB2NF( buffer_size );
    fprintf( INFO_CHANNEL, "%s: Sampling frequency %d Hz, ",
             shared_pn, dsp_speed );
    fprintf( INFO_CHANNEL, "%s, ", dsp_stereo?"stereo":"mono" );
    fprintf( INFO_CHANNEL, "sample size %d bit.\n", dsp_samplesize );
    if( verbose - 1 )
      fprintf( INFO_CHANNEL,
               "%s: %d buffer%s a %d kB, %d frames, %02d:%02d:%02d msf\n",
               shared_pn, number_of_buffers, number_of_buffers>1?"s":"",
               buffer_size/1024, nf, NF2NM(nf), NF2NS(nf), NF2NRF(nf) );
    if( number_of_buffers == 1 )
      fprintf( INFO_CHANNEL,
               "%s: Quitting automatically (only) after %02d:%02d (mm:ss).\n",
               shared_pn, NF2NM(nf), NF2NS(nf) );
    fprintf( INFO_CHANNEL, "%s: Writing to ", shared_pn );
    if( shared_fd==STDOUT_FILENO ) fprintf( INFO_CHANNEL, "stdout" );
    else fprintf( INFO_CHANNEL, "file `%s'", shared_fn );
    fprintf( INFO_CHANNEL, ".\n" );
  }

/* record file */
  tnbr=0;
  buffer_index= number_of_buffers - 1;
  if( verbose )
    fprintf( INFO_CHANNEL, "%s: Keys: s - start, q - quit\n", shared_pn );
  for( *key_buf='\0'; ; ) {
    read( STDIN_FILENO, key_buf, KEY_BUFFER_SIZE );
    key= tolower( *key_buf );
    if( key == 's' ) { *key_buf= '\0'; break; }
    if( key == 'q' ) { retval= 0; goto DIE_NOW; }
  }
  if( verbose ) {
    fprintf( INFO_CHANNEL, "%s: Starting recording.", shared_pn );
    if( number_of_buffers != 1 )
      fprintf( INFO_CHANNEL, " Hit q (+ENTER) to quit.\n" );
    else
      fprintf( INFO_CHANNEL, " Hit CTRL-C to abort or wait.\n" );
  }
/* ---------- recording loop start ---------- */
  do {
    buffer_index= ( buffer_index + 1 ) % number_of_buffers;
    bp= buffer + buffer_index * buffer_size;
#ifdef	DEBUG
{
  char str[128];
  sprintf( str, "%s: DEBUG: Master: Executing read()\n", shared_pn );
  write( STDERR_FILENO, str, strlen(str) );
}
#endif
    if( (nbr=read(dsp_fd,bp,buffer_size)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot read from %s. %s.\n",
               shared_pn, dsp_fn, sys_errlist[errno] );
      retval= 1; goto DIE_NOW;
    }
    if( nbr < buffer_size ) {
      fprintf( ERROR_CHANNEL, "%s: Less bytes read than requested.\n",
               shared_pn );
      fprintf( ERROR_CHANNEL, "%s: This is a bug! E-mail me at %s.\n",
               shared_pn, MY_EMAIL_ADDRESS );
      retval= 1; goto DIE_NOW;
    }
    if( (tip=create_thread_info(bp,nbr)) == NULL ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot allocate for thread info.\n",
               shared_pn );
      retval= 1; goto DIE_NOW;
    }
    if( number_of_buffers == 1 )
      fprintf( INFO_CHANNEL, "%s: Quitting recording, flushing buffer.\n",
               shared_pn );
#ifdef	DEBUG
{
  char str[128];
  sprintf( str, "%s: DEBUG: Master: calling slave\n", shared_pn );
  write( STDERR_FILENO, str, strlen(str) );
  sprintf( str, "%s: DEBUG: Master:      addr= %08x (frame %d)\n",
           shared_pn, (unsigned int)bp, tnbr/CD_FRAMESIZE_RAW );
  write( STDERR_FILENO, str, strlen(str) );
}
#endif
    pthread_detach( thread );
    pthread_create( &thread, NULL, client, tip );
    tnbr+= nbr;
    read( STDIN_FILENO, key_buf, KEY_BUFFER_SIZE );
    key= tolower( *key_buf );
    if( number_of_buffers == 1 ) key= 'q';
  } while( shared_err != -1 && key != 'q' && key != 'x' );
/* ---------- recording loop end ---------- */
  if( shared_err == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot write to ", shared_pn );
    if( shared_fd == STDOUT_FILENO )
      fprintf( ERROR_CHANNEL, "stdout. " );
    else
      fprintf( ERROR_CHANNEL, "file `%s'. ", shared_fn );
    fprintf( ERROR_CHANNEL, "%s.\n", sys_errlist[errno] );
    retval= 1; goto DIE_NOW;
  }
  pthread_join( thread, NULL );
  tnbw= tnbr;
  if( verbose && number_of_buffers != 1 )
    fprintf( INFO_CHANNEL, "%s: Quitting recording.\n", shared_pn );


/* prepare to exit */
  if( verbose ) {
    int nf= (int)( NB2NF(tnbw)*((1.0*DEFAULT_DSP_SPEED)/dsp_speed));
    fprintf( INFO_CHANNEL, "%s: %d bytes, %d frames, %02d:%02d:%02d msf\n",
             shared_pn, tnbw, NB2NF(tnbw), NF2NM(nf), NF2NS(nf), NF2NRF(nf) );
  }
  if( tnbr%CD_FRAMESIZE_RAW ) {
    fprintf( ERROR_CHANNEL,
             "%s: Some error occured. Last frame is incomplete.\n",
             shared_pn );
    retval= 1; goto DIE_NOW;
  }
  if( shared_fd != STDOUT_FILENO && close(shared_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't close `%s' file successfully.\n",
             shared_pn, shared_fn );
  }
  if( close(dsp_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't close `%s' successfully.\n",
             shared_pn, dsp_fn );
  }

DIE_NOW:
  if( buffer != NULL ) free( buffer );
  close( dsp_fd );
  close( shared_fd );
  exit( retval );
}
