/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void AFsetNH (const char String[])
  struct AF_NHpar *AFgetNH (void)

Purpose:
  Set defaults for headerless audio files
  Get defaults for headerless audio files

Description:
  This routine sets or gets default audio file data parameters.  These
  parameters are used for files with unrecognized (non-standard) headers or
  files with no headers (raw audio files).  The AFsetNH routine must be called
  before opening the file with AFopenRead.  The parameters for AFsetNH are
  determined from an input string which consists of a list of parameters
  separated by commas.  The form of the list is
    "Format, Start, Sfreq, Swapb, Nchan, ScaleF"
  Format: File data format
      The lowercase versions of these format specifiers cause a headerless
      file to be accepted only after checking for standard file headers; the
      uppercase versions cause a headerless file to be accepted without
      checking the file header.
       "undefined"                - Headerless files will be rejected
       "mu-law8" or "MU-LAW8"     - 8-bit mu-law data
       "A-law8" or "A-LAW8"       - 8-bit A-law data
       "unsigned8" or "UNSIGNED8" - offset-binary 8-bit integer data
       "integer8" or "INTEGER8"   - two's-complement 8-bit integer data
       "integer16" or "INTEGER16" - two's-complement 16-bit integer data
       "float32" or "FLOAT32"     - 32-bit floating-point data
       "text" or "TEXT"           - text data
  Start: byte offset to the start of data (integer value)
  Sfreq: sampling frequency in Hz (floating-point number)
  Swapb: Data byte swap parameter
       "native" - no byte swapping
       "little-endian" - file data is in little-endian byte order and will
          be swapped if the current host uses big-endian byte order
       "big-endian" - data is in big-endian byte order and will be
          swapped swapped if the current host uses little-endian byte order
       "swap" - swap the data bytes
  Nchan: number of channels
      The data consists of interleaved samples from Nchan channels
  ScaleF: Scale factor
      Scale factor applied to the data from the file
  The default values correspond to the following string
    "undefined, 0, 8000., native, 1, 1.0"

  Leading and trailing white-space (as defined by isspace) is removed from
  each item.  In the input string, any of the parameters may be omitted, in
  which case whatever value has been previously set remains in effect for
  that parameter.  The string ",512, 10000." would set the Start and Sfreq
  parameters and leave the other parameters undisturbed.

  If the input string contains has a leading '$', the string is assumed to
  specify the name of an environment variable after the '$'.  This routine uses
  the value of this environment variable to determine the parameters.  If this
  routine is called as AFsetNH("$RAWAUDIOFILE"), this routine would look for
  the parameter string in environment variable RAWAUDIOFILE.

  The procedure AFgetNH is used internally by the audio file routines.  It
  returns a pointer to a structure of type AF_NHpar.

Parameters:
  AFsetNH:
   -> const char String[]
      String containing the list of parameters for headerless files or the name
      of an environment variable (with a leading $)
  AFgetNH:
  <-  struct AF_NHpar *AFgetNH
      Structure with file data parameters
        int      Accept - Headerless flag
        int      Format - Data format
        long int Start  - Offset in bytes to the start of data
        float    Sfreq  - Sampling frequency
        int      Swapb  - Byte swap flag
        long int Nchan  - Number of channels
        float    ScaleF - Scale factor

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.31 $  $Date: 1996/08/14 17:52:59 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFsetNH.c 1.31 1996/08/14 AFsp-V2R1 $";

#include <stdlib.h>	/* getenv prototype */
#include <string.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>

/* Keyword tables */
static const char *Format_keys[] = {
  "u*ndefined",
  "m*u-law8",
  "A-l*aw8",
  "u*nsigned8",
  "integer8",
  "i*nteger16",
  "f*loat32",
  "t*ext",
  "M*U-LAW8",
  "A-L*AW8",
  "U*NSIGNED8",
  "INTEGER8",
  "I*NTEGER16",
  "F*LOAT32",
  "T*EXT",
  NULL
};
static const char *Swapb_keys[] = {
  "n*ative", "l*ittle-endian", "b*ig-endian", "s*wap", NULL};

/* Values corresponding to keywords */
static const int Format_values[2*NFD-1] = {
  FD_UNDEF,
  FD_MULAW8, FD_ALAW8, FD_UINT8, FD_INT8, FD_INT16, FD_FLOAT32, FD_TEXT,
  FD_MULAW8, FD_ALAW8, FD_UINT8, FD_INT8, FD_INT16, FD_FLOAT32, FD_TEXT
};
static const int Accept_values[2*NFD-1] = {
  NH_REJECT,
  NH_ACCEPT, NH_ACCEPT, NH_ACCEPT, NH_ACCEPT, NH_ACCEPT, NH_ACCEPT, NH_ACCEPT,
  NH_FORCE,  NH_FORCE,  NH_FORCE,  NH_FORCE,  NH_FORCE,  NH_FORCE,  NH_FORCE
};
static const int Swapb_values[] = {
  DS_NATIVE, DS_EL, DS_EB, DS_SWAP };

/* Initialize to the default values */
static struct AF_NHpar NHpar = {
  NH_REJECT, FD_INT16, 0L, 8000., DS_NATIVE, 1L, 1. };

void
AFsetNH (String)

     const char String[];

{
  int n, k, nc;
  int Start, Nchan;
  const char *p;
  char *token;

/* Check for an environment variable */
  if (String[0] == '$') {
    p = getenv (&String[1]);
    if (p == NULL)
      p = "";
  }
  else
    p = String;

/* Allocate temporary storage */
  nc = strlen (p);
  token = (char *) UTmalloc (nc + 1);

/* Separate the parameters */
  k = 0;
  while (p != NULL) {
    p = STfindToken (p, ",", "", token, 1, nc);
    if (token[0] != '\0') {

      /* Decode the parameter values */
      switch (k) {
      case 0:
	n = STkeyMatch (token, Format_keys);
	if (n < 0)
	  UTwarn ("AFsetNH - Invalid format keyword, \"%.10s\"", token);
	else {
	  NHpar.Accept = Accept_values[n];
	  NHpar.Format = Format_values[n];
	}
	break;
      case 1:
	if (STdec1int (token, &Start))
	  UTwarn ("AFsetNH - Invalid start value, \"%.10s\"", token);
	else
	  NHpar.Start = Start;
	break;
      case 2:
	if (STdec1float (token, &NHpar.Sfreq))
	  UTwarn ("AFsetNH - Invalid sampling frequency, \"%.10s\"", token);
	break;
      case 3:
	n = STkeyMatch (token, Swapb_keys);
	if (n < 0)
	  UTwarn ("AFsetNH - Invalid swap keyword, \"%.10s\"", token);
	else
	  NHpar.Swapb = Swapb_values[n];
	break;
      case 4:
	if (STdec1int (token, &Nchan))
	  UTwarn ("AFsetNH - Invalid number of channels, \"%.10s\"", token);
	else
	  NHpar.Nchan = Nchan;
	break;
      case 5:
	if (STdec1float (token, &NHpar.ScaleF))
	  UTwarn ("AFsetNH - Invalid scale factor, \"%.10s\"", token);
	break;

      }
    }
    ++k;
  }
  UTfree ((void *) token);

  return;
}

struct AF_NHpar *
AFgetNH ()

{
  return &NHpar;
}
